##*************************************************************************##
##  CUBE        http://www.scalasca.org/                                   ##
##*************************************************************************##
##  Copyright (c) 2024                                                     ##
##  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          ##
##                                                                         ##
##  This software may be modified and distributed under the terms of       ##
##  a BSD-style license.  See the COPYING file in the package base         ##
##  directory for details.                                                 ##
##*************************************************************************##


# This cmake script will define the following variables:
#
# CUBEW_REVISION
#   Latest commit sha.
# CUBEW_FULL_NAME
#   e.g. "CUBEW-4.8.2-rc1" if repo is tagged or "CUBEW-branch-issueXX" if repo is a branch
# CUBEW_VERSION
#   Full git tag e.g. "4.8.2-rc1" if tagged, otherwise 0.0.0
# CUBEW_VERSION_MAJOR
#   Major version from CUBEW_VERSION.
# CUBEW_VERSION_MINOR
#   Minor version from CUBEW_VERSION.
# CUBEW_VERSION_PATCH
#   Patch version from CUBEW_VERSION.


# todo: CUBEW_VERSION_SUFFIX seems not to be used anywhere

# --- default values
set(PACKAGE_NAME CubeW)
set(CUBEW_VERSION_MAJOR 4)
set(CUBEW_VERSION_MINOR 9)
set(CUBEW_VERSION_PATCH 0)
set(CUBEW_VERSION 4.9.0)
set(CUBEW_LIBRARY_CURRENT 0)
set(CUBEW_LIBRARY_REVISION 0)
set(CUBEW_LIBRARY_AGE 0)


# --- get version info from build-config ---

if ( EXISTS "${CMAKE_SOURCE_DIR}/build-config/VERSION" )
    file(READ "${CMAKE_SOURCE_DIR}/build-config/VERSION" CONTENT)

    # get CUBEW version info from build-config
    string(REGEX MATCH "package.major=([0-9]*)" TMP "${CONTENT}")
    set( CUBEW_VERSION_MAJOR "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "package.minor=([0-9]*)" TMP "${CONTENT}")
    set( CUBEW_VERSION_MINOR "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "package.bugfix=([0-9]*)" TMP "${CONTENT}")
    set( CUBEW_VERSION_PATCH "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "package.suffix=([^\n]*)" TMP "${CONTENT}")
    if( NOT ${CMAKE_MATCH_1} STREQUAL "" )
      set( CUBEW_VERSION_SUFFIX "-${CMAKE_MATCH_1}" )
    endif()
    set( CUBEW_VERSION "${CUBEW_VERSION_MAJOR}.${CUBEW_VERSION_MINOR}.${CUBEW_VERSION_PATCH}" )
    set( CUBEW_FULL_NAME ${PACKAGE_NAME}-${CUBEW_VERSION}${CUBEW_VERSION_SUFFIX} )

    # get library version info from build-config
    string(REGEX MATCH "library.current=([0-9]*)" TMP "${CONTENT}")
    set( CUBEW_LIBRARY_CURRENT "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "library.revision=([0-9]*)" TMP "${CONTENT}")
    set( CUBEW_LIBRARY_REVISION "${CMAKE_MATCH_1}" )
    string(REGEX MATCH "library.age=([0-9]*)" TMP "${CONTENT}")
    set( CUBEW_LIBRARY_AGE "${CMAKE_MATCH_1}" )
endif()

# --- helper function to execute git commands ---
# executes a git command and returns the RESULT_VARIABLE
# ARGUMENTS: the arguments of the git executable
# OUTPUT: the result is written to this variable name
function(git_execute ARGUMENTS OUTPUT )
    separate_arguments(ARGUMENTS)
    execute_process(
        COMMAND ${GIT_EXECUTABLE} ${ARGUMENTS}
        WORKING_DIRECTORY ${CMAKE_SOURCE_DIR}
        OUTPUT_VARIABLE OUTPUT_STR
        RESULT_VARIABLE ERROR_CODE
        ERROR_VARIABLE ERR
        OUTPUT_STRIP_TRAILING_WHITESPACE
    )
    set(${OUTPUT} "${OUTPUT_STR}" PARENT_SCOPE)
endfunction()


# --- check if source is git-controlled ---
set( GIT_CONTROLLED false )
if ( EXISTS "${CMAKE_SOURCE_DIR}/.git" )
    find_package(Git)
    if ( ${GIT_FOUND} )
        set( GIT_CONTROLLED true )
    endif()
else()
    message(STATUS "Not a git-controlled repository.")
endif()

# --- get version info from git ---
if ( ${GIT_CONTROLLED} )
  git_execute("rev-parse --short HEAD" CUBEW_REVISION )
  if (NOT "${CUBEW_REVISION}" STREQUAL "")
        git_execute("git tag --contains HEAD" TAGS )
        if ( NOT "${TAGS}" STREQUAL "") # git tagged release
              string(REGEX REPLACE "^v" "" CUBEW_VERSION "${TAG}")
              string(REGEX MATCH "^([0-9]+)\\.([0-9]+)\\.([0-9]+)(-([a-z0-9]+))?" CUBEW_VERSION ${CUBEW_VERSION})
              set(CUBEW_VERSION_MAJOR ${CMAKE_MATCH_1})
              set(CUBEW_VERSION_MINOR ${CMAKE_MATCH_2})
              set(CUBEW_VERSION_PATCH ${CMAKE_MATCH_3})
              set(CUBEW_VERSION_SUFFIX ${CMAKE_MATCH_4})
              set(CUBEW_FULL_NAME ${PACKAGE_NAME}-${CUBEW_VERSION})
        else () # git branch -> CUBEW_FULL_PACKAGE_NAME shows branch name
              git_execute("describe --all" BRANCH )
              string(REGEX MATCH "^.+/([a-z0-9]+)" MATCH_PACKAGE_NAME ${BRANCH})
              set(CUBEW_FULL_NAME ${PACKAGE_NAME}-branch-${CMAKE_MATCH_1})
        endif ()
  endif ()
endif()

if(NOT DEFINED CUBEW_FULL_NAME) # git not available or repo not git-controlled
    message(STATUS "Failed to determine CUBEW_VERSION. Using default version \"${CUBEW_VERSION}\".")
endif()
