/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/**
 * \file cubew_tar_writing.c
 * \brief Provides a file name transformation scheme: name -> tar file + shift. Produces tared cube report.
 *
 *
 *
 */

#define _FILE_OFFSET_BITS 64
#define _LARGEFILE_SOURCE

#include "config.h"
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <inttypes.h>

#include "cubew_memory.h"
#include "cubew_report_layouts.h"
#include "cubew_tar_writing.h"
#include "cubew_meta_data_writer.h"
#include "cubew_compat_platform.h"

#include "UTILS_Error.h"

#define MEMORY_TRACING_PREFIX "[TAR WRITER]"



static
char*
cube_get_tared_cube_name__( char* cubename )
{
    char* buff = ( char* )CUBEW_CALLOC( 1, strlen( cubename ) + 7, MEMORY_TRACING_PREFIX "Allocate cubename" );
    strcat( buff, cubename );
    strcat( buff, ".cubex" );
    return buff;
}



static
void
cube_calculate_checksum__( tar_gnu_header* header )
{
    // Use snprintf to ensure null termination and prevent buffer overflow
    snprintf( header->checksum, sizeof( header->checksum ), "%s", "        " ); // 8 spaces

    uint8_t* pos      = ( uint8_t* )header;
    uint64_t checksum = 0;      // Use uint64_t to prevent overflow for checksum calculation

    for ( unsigned i = 0; i < ( sizeof( tar_gnu_header ) ); i++ )
    {
        checksum += ( uint64_t )( *pos ); // Cast to uint64_t for addition
        pos++;
    }

    snprintf( header->checksum, sizeof( header->checksum ), "%6.6lo", ( unsigned long )checksum );
    // Note: The %6.6lo format specifier might truncate a very large 64-bit checksum.
    // If checksum exceeds 2^48 - 1 (since 6 octal digits give 18 bits, and 6 * 8 = 48 bits for 6 bytes in checksum field)
    // this could be an issue, but is standard for tar.
}


static
void
cube_set_size_and_calculate_checksum__( tar_gnu_header* header, uint64_t size )
{
    // Ensure null termination and correct formatting for size field
    snprintf( header->size, sizeof( header->size ), "%11.11lo", ( unsigned long )size );
    cube_calculate_checksum__( header );
}


static
tar_gnu_header*
cube_create_tar_header__( report_layout_writer* tar_writer, char* filename, size_t size )
{
    tar_gnu_header* tar_header = ( tar_gnu_header* )CUBEW_CALLOC( 1, sizeof( tar_gnu_header ), MEMORY_TRACING_PREFIX "Allocate tar header" );
    if ( tar_header == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT, "CUBEW_CALLOC failed for tar_header in cube_create_tar_header__." );
        return NULL; // Return NULL on allocation failure
    }

    // Use snprintf consistently for safety and null-termination, respecting field sizes
    snprintf( tar_header->name, sizeof( tar_header->name ), "%s", filename );
    snprintf( tar_header->mode, sizeof( tar_header->mode ), "%s", tar_writer->mode );
    // Note: UID/GID fields are 8 bytes, but format specifier is %7.7lo, leaving space for null terminator.
    snprintf( tar_header->uid, sizeof( tar_header->uid ),  "%7.7lo",  ( unsigned long )( tar_writer->uid )  & 0x1FFFFF );
    snprintf( tar_header->gid, sizeof( tar_header->gid ), "%7.7lo", ( unsigned long )( tar_writer->gid )  & 0x1FFFFF  );
    unsigned int mtime = time( NULL );
    snprintf( tar_header->mtime, sizeof( tar_header->mtime ), "%11.11lo", ( unsigned long )mtime );
    snprintf( tar_header->typeflag, sizeof( tar_header->typeflag ), "%s", "0" ); // typeflag is 1 byte, so 1 for size
    snprintf( tar_header->uname, sizeof( tar_header->uname ), "%s", tar_writer->username );
    snprintf( tar_header->gname, sizeof( tar_header->gname ), "%s", tar_writer->group );

    const char* _magic = "ustar"; // Use const char* for string literals
    snprintf( tar_header->magic, sizeof( tar_header->magic ), "%s", _magic );
    const char* _version = "00";  // Use const char* for string literals
    snprintf( tar_header->version, sizeof( tar_header->version ), "%s", _version );

    cube_set_size_and_calculate_checksum__( tar_header, size );

    return tar_header;
}


static
void
cube_tar_file_finish__( report_layout_writer* tar_writer, uint64_t size )
{
    uint64_t actual_pos = ftell( tar_writer->tar );

    if ( cubew_fseeko( tar_writer->tar, tar_writer->header_position, SEEK_SET ) != 0 )
    {
        UTILS_WARNING( "Cannot seek to the header of the current file to write the checksum%s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
    }
    cube_set_size_and_calculate_checksum__( tar_writer->actual_tar_header, size );
    fwrite( tar_writer->actual_tar_header, 1, sizeof( tar_gnu_header ), tar_writer->tar );
    if ( cubew_fseeko( tar_writer->tar, actual_pos, SEEK_SET )  != 0 )
    {
        UTILS_WARNING( "Cannot seek to the end of the current file to proceed with the writing of the tared cube file %s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
    }
    uint64_t difference = ( ( size / sizeof( tar_empty_block ) + 1 ) * sizeof( tar_empty_block )  - size );
    char*    _tmp       = ( char* )CUBEW_CALLOC( difference, sizeof( char ), MEMORY_TRACING_PREFIX "Allocate tail of a tar block" );
    fwrite( _tmp, 1, difference, tar_writer->tar );
    CUBEW_FREE( _tmp, MEMORY_TRACING_PREFIX "Release tail in a tar block" );
    CUBEW_FREE( tar_writer->actual_tar_header, MEMORY_TRACING_PREFIX "Release tar header" );
    tar_writer->actual_tar_header = NULL;
}

static
void
cube_tar_data_file_finish__( report_layout_writer* tar_writer, uint64_t size )
{
    uint64_t difference = ( ( size / sizeof( tar_empty_block ) + 1 ) * sizeof( tar_empty_block )  - size );
    char*    _tmp       = ( char* )CUBEW_CALLOC( difference, sizeof( char ), MEMORY_TRACING_PREFIX "Allocate tail of a tar block" );
    fwrite( _tmp, 1, difference, tar_writer->tar );
    CUBEW_FREE( _tmp, MEMORY_TRACING_PREFIX "Release tail in a tar block" );
    CUBEW_FREE( tar_writer->actual_tar_header, MEMORY_TRACING_PREFIX "Release tar header" );
    tar_writer->actual_tar_header = NULL;
}

static
void
cube_tar_finish__( report_layout_writer* tar_writer )
{
    tar_empty_block* block = ( tar_empty_block* )CUBEW_CALLOC( 1, sizeof( tar_empty_block ), MEMORY_TRACING_PREFIX "Allocate tarblock " );
    fwrite( ( char* )block, 1, sizeof( tar_empty_block ), tar_writer->tar );
    fwrite( ( char* )block, 1, sizeof( tar_empty_block ), tar_writer->tar );
    CUBEW_FREE( block, MEMORY_TRACING_PREFIX "Release block " );
}

static
void
fill_pax_header__( tar_gnu_header* tar4pax, tar_gnu_header* pax_block, uint64_t size )
{
    memset( pax_block, 0, TAR_BLOCKSIZE );
    snprintf( ( char* )pax_block, sizeof( tar_empty_block ), "xx size=%"  PRIu64 "\n", size );
    uint8_t len = strlen( ( char* )pax_block );
    snprintf( ( char* )pax_block, sizeof( tar_empty_block ), "%2.2d size=%"  PRIu64 "\n", len, size );
    strncpy( tar4pax->typeflag, "x", 1 );
    cube_set_size_and_calculate_checksum__( tar4pax, len );
}

FILE*
cube_report_anchor_start( report_layout_writer* tar_writer )
{
    /* if metric is not finished yet, one has to */

    if ( tar_writer == NULL )
    {
        UTILS_WARNING( "Non stanard run. Create faked tar_writer with temp name of cube \"__NOFILE__\". \n" );
        tar_writer = cube_writing_start( "__NOFILE__", CUBE_MASTER );
    }
    char* anchorname = cube_get_path_to_anchor( tar_writer->cubename );
    tar_writer->actual_tar_header = cube_create_tar_header__( tar_writer, anchorname, 0 );
    CUBEW_FREE( anchorname, MEMORY_TRACING_PREFIX "Release anchor name" );
    tar_writer->header_position = ftell( tar_writer->tar );
    if ( tar_writer->header_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed for tar_writer->header_position for the position befor the writing the header in cube_report_anchor_start." );
        goto cleanup_error;
    }
    fwrite( tar_writer->actual_tar_header, 1, sizeof( tar_gnu_header ), tar_writer->tar );
    tar_writer->file_start_position = ftell( tar_writer->tar );
    if ( tar_writer->header_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed for tar_writer->header_position for the position after the writing the header in cube_report_anchor_start." );
        goto cleanup_error;
    }
    tar_writer->anchor_writing = CUBE_TRUE;
    return tar_writer->tar;
cleanup_error:
    CUBEW_FREE( tar_writer->actual_tar_header, MEMORY_TRACING_PREFIX "Release actual_tar_header on error in cube_report_anchor_start" );
    tar_writer->actual_tar_header = NULL; // Prevent double free
    return NULL;
}




FILE*
cube_report_metric_data_start( report_layout_writer* tar_writer, cube_metric* met )
{
    // Handle case where tar_writer might be NULL (e.g., from a faked call)
    if ( tar_writer == NULL )
    {
        UTILS_WARNING( "Non standard run. Creating faked tar_writer with temp name of cube \"__NOFILE__\". \n" );
        tar_writer = cube_writing_start( "__NOFILE__", CUBE_MASTER );
        if ( tar_writer == NULL ) // Check if faked writer creation also failed
        {
            UTILS_ERROR( CUBEW_ABORT, "Failed to create faked tar_writer. Aborting." );
            return NULL;
        }
    }

    char* dataname = cube_get_path_to_metric_data( tar_writer->cubename, met );
    if ( dataname == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "Allocation failed for dataname in cube_report_metric_data_start." );
        return NULL; // Critical allocation failure, return NULL
    }

    uint64_t size = cube_metric_size_of_data_file( met );

    // Create the main header for the metric data file
    tar_writer->actual_tar_header = cube_create_tar_header__( tar_writer, dataname, size );
    if ( tar_writer->actual_tar_header == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "Allocation failed for actual_tar_header in cube_report_metric_data_start." );
        CUBEW_FREE( dataname, MEMORY_TRACING_PREFIX "Release dataname on error" );
        return NULL; // Critical allocation failure
    }

    // Variables for PAX header, initialized to NULL for safe cleanup
    char*           paxname            = NULL;
    tar_gnu_header* tar4pax            = NULL;
    tar_gnu_header* pax_header_content = NULL; // Renamed for clarity: this holds the content block

    // If file size exceeds standard TAR limits, create a PAX extended header
    if ( size >= 0x1FFFFFFFFULL ) /* Using ULL for unsigned long long literal */
    {
        // Allocate paxname using CUBEW_CALLOC for consistency with custom memory management
        paxname = ( char* )CUBEW_CALLOC( 1, 11 + strlen( dataname ) + 1, MEMORY_TRACING_PREFIX "Allocate metric pax name" ); // +1 for null terminator
        if ( paxname == NULL )
        {
            UTILS_ERROR( CUBEW_ABORT,  "CUBEW_CALLOC failed for paxname." );
            goto cleanup_error;                                                     // Jump to cleanup on error
        }
        snprintf( paxname, 11 + strlen( dataname ) + 1, "PaxHeader/%s", dataname ); // Use snprintf for safety

        // Create the header for the PAX block itself
        tar4pax = cube_create_tar_header__( tar_writer, paxname, TAR_BLOCKSIZE ); // TAR_BLOCKSIZE (512) is the size of the PAX content block
        if ( tar4pax == NULL )
        {
            UTILS_ERROR( CUBEW_ABORT,  "cube_create_tar_header__ failed for tar4pax." );
            goto cleanup_error;
        }

        // Create the block that will contain the actual PAX header content
        // This block needs to be TAR_BLOCKSIZE (512 bytes)
        pax_header_content = ( tar_gnu_header* )CUBEW_CALLOC( 1, TAR_BLOCKSIZE, MEMORY_TRACING_PREFIX "Allocate PAX block content" );
        if ( pax_header_content == NULL )
        {
            UTILS_ERROR( CUBEW_ABORT,  "CUBEW_CALLOC failed for pax_header_content." );
            goto cleanup_error;
        }

        // Fill the content block with the extended size information
        // fill_pax_header__ uses `snprintf` which is good.
        fill_pax_header__( tar4pax, pax_header_content, size );

        // Write the PAX header and its content block to the tar file
        fwrite( tar4pax, 1, sizeof( tar_gnu_header ), tar_writer->tar );
        fwrite( pax_header_content, 1, TAR_BLOCKSIZE, tar_writer->tar ); // Write the content of the PAX header block

        // Free the locally allocated PAX related memory
        CUBEW_FREE( paxname, MEMORY_TRACING_PREFIX "Release report_layout_writer metric pax name" );
        CUBEW_FREE( tar4pax, MEMORY_TRACING_PREFIX "Release tar header for PAX block" );
        CUBEW_FREE( pax_header_content, MEMORY_TRACING_PREFIX "Release PAX block content" );
        // Set pointers to NULL after freeing to prevent use-after-free or double-free if error path is re-entered
        paxname            = NULL;
        tar4pax            = NULL;
        pax_header_content = NULL;
    }

    CUBEW_FREE( dataname, MEMORY_TRACING_PREFIX "Release metric data name" );

    // Store the header position and file start position
    tar_writer->header_position = ftell( tar_writer->tar );
    if ( tar_writer->header_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed for tar_writer->tar in cube_report_metric_data_start." );
        goto cleanup_error;
    }

    // Write the actual_tar_header (which might be updated later)
    fwrite( tar_writer->actual_tar_header, 1, sizeof( tar_gnu_header ), tar_writer->tar );

    tar_writer->file_start_position = ftell( tar_writer->tar );
    if ( tar_writer->file_start_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed for tar_writer->tar (post-header) in cube_report_metric_data_start." );
        goto cleanup_error;
    }

    met->start_pos_of_datafile = tar_writer->file_start_position;
    return tar_writer->tar;

cleanup_error:
    // This block ensures all allocated memory is freed on error path
    CUBEW_FREE( paxname, MEMORY_TRACING_PREFIX "Release paxname on error in cube_report_metric_data_start" );
    CUBEW_FREE( tar4pax, MEMORY_TRACING_PREFIX "Release tar4pax on error in cube_report_metric_data_start" );
    CUBEW_FREE( pax_header_content, MEMORY_TRACING_PREFIX "Release pax_header_content on erro in cube_report_metric_data_startr" );
    CUBEW_FREE( tar_writer->actual_tar_header, MEMORY_TRACING_PREFIX "Release actual_tar_header on error in cube_report_metric_data_start" );
    tar_writer->actual_tar_header = NULL;     // Prevent double free
    CUBEW_FREE( dataname, MEMORY_TRACING_PREFIX "Release dataname on error in cube_report_metric_data_start" );
    // Consider cleaning up tar_writer itself if the error is severe enough to invalidate it
    // For now, assuming tar_writer is valid unless created by the faked path
    return NULL;
}


FILE*
cube_report_metric_index_start( report_layout_writer* tar_writer, cube_metric* met )
{
    if ( tar_writer == NULL )
    {
        UTILS_WARNING( "Non stanard run. Create faked tar_writer with temp name of cube \"__NOFILE__\". \n" );
        tar_writer = cube_writing_start( "__NOFILE__", CUBE_MASTER );
    }

    char* indexname = cube_get_path_to_metric_index( tar_writer->cubename, met );
    tar_writer->actual_tar_header = cube_create_tar_header__( tar_writer, indexname, 0 );
    CUBEW_FREE( indexname, MEMORY_TRACING_PREFIX "Release metric index name"  );
    tar_writer->header_position = ftell( tar_writer->tar );
    if ( tar_writer->header_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed for save the position in tar_writer->header_position in cube_report_metric_index_start." );
        goto cleanup_error;
    }
    fwrite( tar_writer->actual_tar_header, 1, sizeof( tar_gnu_header ), tar_writer->tar );
    tar_writer->file_start_position = ftell( tar_writer->tar );
    if ( tar_writer->header_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed for save the new position in tar_writer->header_position in cube_report_metric_index_start." );
        goto cleanup_error;
    }
    return tar_writer->tar;
cleanup_error:
    CUBEW_FREE( tar_writer->actual_tar_header, MEMORY_TRACING_PREFIX "Release actual_tar_header on error in cube_report_metric_index_start" );
    tar_writer->actual_tar_header = NULL; // Prevent double free
    return NULL;
}


FILE*
cube_report_misc_data_start( report_layout_writer* tar_writer, const char* dataname )
{
    if ( tar_writer == NULL )
    {
        UTILS_WARNING( "Non stanard run. Create faked tar_writer with temp name of cube \"__NOFILE__\". \n" );
        tar_writer = cube_writing_start( "__NOFILE__", CUBE_MASTER );
    }
    else
    {
        if ( tar_writer->actual_metric != NULL )
        {
            cube_metric_finish( tar_writer->actual_metric, 1 );  /* 1 - signalizes, that writing will be  done. This call can be performed only by CUBE_MASTER */
        }
        tar_writer->actual_metric = NULL;

        // from here we finished writing metric.

        char* _dataname = cube_get_path_to_misc_data( tar_writer->cubename, dataname );
        tar_writer->actual_tar_header = cube_create_tar_header__( tar_writer, _dataname, 0 );
        CUBEW_FREE( _dataname, MEMORY_TRACING_PREFIX "Release _dataname " );
        tar_writer->header_position = ftell( tar_writer->tar );
        if ( tar_writer->header_position == -1L ) // Check ftell error
        {
            UTILS_ERROR( CUBEW_ABORT,  "ftell failed for save the position in tar_writer->header_position in cube_report_misc_data_start." );
            goto cleanup_error;
        }
        fwrite( tar_writer->actual_tar_header, 1, sizeof( tar_gnu_header ), tar_writer->tar );
        tar_writer->file_start_position = ftell( tar_writer->tar );
        if ( tar_writer->header_position == -1L ) // Check ftell error
        {
            UTILS_ERROR( CUBEW_ABORT,  "ftell failed for save the new position in tar_writer->header_position in cube_report_misc_data_start." );
            goto cleanup_error;
        }
    }
    return tar_writer->tar;
cleanup_error:
    CUBEW_FREE( tar_writer->actual_tar_header, MEMORY_TRACING_PREFIX "Release actual_tar_header on error in cube_report_metric_index_start" );
    tar_writer->actual_tar_header = NULL; // Prevent double free
    return NULL;
}

void
cube_report_metric_data_finish( report_layout_writer* tar_writer, cube_metric* met, FILE* file )
{
    if ( cubew_fseeko( tar_writer->tar, 0, SEEK_END ) != 0 )
    {
        UTILS_WARNING( "Cannot seek to the end of the data  file to finish its writing in the tared cube file %s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
    }
    /* check for finish*/
    uint64_t current_position = ftell( tar_writer->tar );
    if ( current_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed to report the current position in cube_report_metric_data_finish." );
        return;
    }
    uint64_t size = current_position - ( tar_writer->file_start_position );
    cube_tar_data_file_finish__( tar_writer, size );
}


void
cube_report_misc_data_finish( report_layout_writer* tar_writer, FILE* file )
{
    if ( cubew_fseeko( tar_writer->tar, 0, SEEK_END ) != 0 )
    {
        UTILS_WARNING( "Cannot seek to the end of the data  file to finish its writing in the tared cube file %s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
    }
    /* check for finish*/
    uint64_t current_position = ftell( tar_writer->tar );
    if ( current_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed to report the current position in cube_report_metric_data_finish." );
        return;
    }
    uint64_t size = current_position - ( tar_writer->file_start_position );
    cube_tar_file_finish__( tar_writer, size );
}

void
cube_report_metric_data_weak_finish( report_layout_writer* tar_writer, cube_metric* met, FILE* file )
{
    /*  in the case, .cubex container is being written in parallel pattern, then non-master should not write anything
       in tar case this does nothing*/
}


void
cube_report_metric_index_finish( report_layout_writer* tar_writer, cube_metric* met, FILE* file )
{
    if ( cubew_fseeko( tar_writer->tar, 0, SEEK_END ) != 0 )
    {
        UTILS_WARNING( "Cannot seek to the end of index file to finish its writing in the tared cube file %s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
    }
    uint64_t size = ftell( tar_writer->tar ) - ( tar_writer->file_start_position );
    cube_tar_file_finish__( tar_writer, size );
}


void
cube_report_anchor_finish( report_layout_writer* tar_writer, FILE* file )
{
    if ( cubew_fseeko( tar_writer->tar, 0, SEEK_END ) != 0 )
    {
        UTILS_WARNING( "Cannot seek to the end of anchor file to finish its writing in the tared cube file %s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
    }
    uint64_t current_position = ftell( tar_writer->tar );
    if ( current_position == -1L ) // Check ftell error
    {
        UTILS_ERROR( CUBEW_ABORT,  "ftell failed to report the current position in cube_report_anchor_finish." );
        return;
    }
    uint64_t size = current_position - ( tar_writer->file_start_position );
    cube_tar_file_finish__( tar_writer, size );
}






report_layout_writer*
cube_writing_start( char* cubename, enum CubeFlavours_t cf )
{
    report_layout_writer* tar_writer = ( tar_writer_t* )CUBEW_CALLOC( 1, sizeof( tar_writer_t ), MEMORY_TRACING_PREFIX "Allocate tar writer" );
    if ( tar_writer == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "CUBEW_CALLOC failed for tar_writer in cube_writing_start." );
        return NULL; // Cannot even allocate the main struct
    }

    // Initialize all pointers to NULL for safe cleanup in case of partial allocation failure
    tar_writer->cubename          = NULL;
    tar_writer->mode              = NULL;
    tar_writer->username          = NULL;
    tar_writer->group             = NULL;
    tar_writer->actual_tar_file   = NULL;
    tar_writer->tar               = NULL;
    tar_writer->actual_metric     = NULL;
    tar_writer->actual_tar_header = NULL; // Ensure all fields are initialized to NULL

    // Allocate members of the struct, checking for NULL after each allocation
    tar_writer->cubename = cubew_strdup( cubename );
    if ( tar_writer->cubename == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "cubew_strdup failed for cubename." );
        goto cleanup_error;
    }

    tar_writer->mode = cubew_strdup( "0000600" );
    if ( tar_writer->mode == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "cubew_strdup failed for mode." );
        goto cleanup_error;
    }

    tar_writer->username = cubew_strdup( "nouser" );
    if ( tar_writer->username == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "cubew_strdup failed for username." );
        goto cleanup_error;
    }

    tar_writer->group = ( char* )CUBEW_CALLOC( 32, sizeof( char ), MEMORY_TRACING_PREFIX "Allocate group name" );
    if ( tar_writer->group == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "CUBEW_CALLOC failed for group." );
        goto cleanup_error;
    }
    strcpy( tar_writer->group, "users" ); // strcpy is okay here as destination (32 bytes) is larger than "users" (5 bytes + null)

    tar_writer->uid = cube_getuid();
    tar_writer->gid = cube_getgid();

    tar_writer->actual_tar_file = cube_get_tared_cube_name__( cubename );
    if ( tar_writer->actual_tar_file == NULL )
    {
        UTILS_ERROR( CUBEW_ABORT,  "cube_get_tared_cube_name__ failed." );
        goto cleanup_error;
    }

    tar_writer->tar = fopen( tar_writer->actual_tar_file, "wb" );
    if ( tar_writer->tar == NULL )
    {
        UTILS_WARNING( "Cannot open tared cube file %s. \n", tar_writer->actual_tar_file );
        perror( "The following error occurred" );
        UTILS_WARNING( "Returning NULL after cleanup.\n" );
        goto cleanup_error; // Jump to cleanup on fopen failure
    }

    tar_writer->header_position     = 0;
    tar_writer->file_start_position = 0;
    tar_writer->anchor_writing      = CUBE_FALSE;
    tar_writer->cube_flavour        = cf;
    return tar_writer;

cleanup_error:
    // This block ensures all previously allocated memory is freed on error
    if ( tar_writer->actual_tar_file )
    {
        CUBEW_FREE( tar_writer->actual_tar_file, MEMORY_TRACING_PREFIX "Release tar file name (error)" );
    }
    if ( tar_writer->cubename )
    {
        CUBEW_FREE( tar_writer->cubename, MEMORY_TRACING_PREFIX "Release cube filename (error)" );
    }
    if ( tar_writer->username )
    {
        CUBEW_FREE( tar_writer->username, MEMORY_TRACING_PREFIX "Release username (error)" );
    }
    if ( tar_writer->mode )
    {
        CUBEW_FREE( tar_writer->mode, MEMORY_TRACING_PREFIX "Release permission mode (error)" );
    }
    if ( tar_writer->group )
    {
        CUBEW_FREE( tar_writer->group, MEMORY_TRACING_PREFIX "Release group name (error)" );
    }
    // Note: tar_writer->tar only needs fclose if it was successfully opened.
    // The tar_writer struct itself is the last thing to free
    if ( tar_writer )
    {
        CUBEW_FREE( tar_writer, MEMORY_TRACING_PREFIX "Release tar writer (error)" );
    }
    return NULL;
}




void
cube_report_write_metric_row( report_layout_writer* tar_writer, cube_metric* met )
{
    if ( tar_writer->cube_flavour == CUBE_SLAVE )
    {
        return;
    }
    if ( tar_writer->actual_metric == met )
    {
        return;
    }
    if ( tar_writer->actual_metric != NULL )
    {
        cube_metric_finish( tar_writer->actual_metric, 1 );  /* 1 - signalizes, that writing will be  done. This call can be performed only by CUBE_MASTER */
    }
    met->data_file            = cube_report_metric_data_start( tar_writer, met );
    tar_writer->actual_metric = met;
}






report_layout_writer*
cube_writing_end( report_layout_writer* tar_writer )
{
    if ( tar_writer->cube_flavour != CUBE_MASTER )
    {
        return NULL;
    }
    cube_tar_finish__( tar_writer );
    CUBEW_FREE( tar_writer->actual_tar_file, MEMORY_TRACING_PREFIX "Release tar file name " );
    CUBEW_FREE( tar_writer->cubename, MEMORY_TRACING_PREFIX "Release cube filename " );
    CUBEW_FREE( tar_writer->username, MEMORY_TRACING_PREFIX "Release username " );
    CUBEW_FREE( tar_writer->mode, MEMORY_TRACING_PREFIX "Release permission mode" );
    CUBEW_FREE( tar_writer->group, MEMORY_TRACING_PREFIX "Release group name" );
    fclose( tar_writer->tar );
    CUBEW_FREE( tar_writer, MEMORY_TRACING_PREFIX "Release tar writer" );
    tar_writer = NULL;
    return tar_writer; // return NULL to sugnalize the freeing of the object
}
