/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/**
 * \file cube4_pop_metrics.cpp
 * \brief Calculates different POPx metrics for selected cnode for every profile
 *
 */

#include "config.h"
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <iostream>
#include <limits>
#include <regex>
#include <sstream>
#include <string>
#include <unistd.h>
#include <vector>

#include "cube4_pop_metrics_help.h"
#include "CubeIoProxy.h"
#include "CubeZfstream.h"
#include "CubeServices.h"
#include "CubeError.h"

#include "PerformanceAnalysis.h"
#include "PerformanceTest.h"
#include "POPCalculation.h"
#include "POPAuditAnalysis.h"
#include "POPHybridAuditAnalysisAdd.h"
#include "POPHybridAuditAnalysis.h"
#include "BSPOPHybridAuditAnalysis.h"


using namespace std;
using namespace cube;
using namespace services;
/**
 * Main program.
 * -
 * -
 */
void
popcalculation::printHelp( popcalculation::POP_ANALYSIS pop_analysis )
{
    cube::CubeProxy*                     cube                 = new CubeIoProxy();
    popcalculation::PerformanceAnalysis* performance_analysis = nullptr;
    switch ( pop_analysis )
    {
        case popcalculation::HYBRID_ADD:
            performance_analysis = new hybaddanalysis::POPHybridAuditPerformanceAnalysisAdd( cube );
            break;
        case popcalculation::HYBRID_MULT:
            performance_analysis = new hybanalysis::POPHybridAuditPerformanceAnalysis( cube );
            break;
        case popcalculation::HYBRID_BSC:
            performance_analysis = new bscanalysis::BSPOPHybridAuditPerformanceAnalysis( cube );
            break;
        case popcalculation::PURE_MPI:
        default:
            performance_analysis = new mpianalysis::POPAuditPerformanceAnalysis( cube );
            break;
    }
    std::cout << " -- Analysis -- " << std::endl;
    std::cout << performance_analysis->name() << std::endl << std::endl;
    std::cout << performance_analysis->getAnchorHowToMeasure() << std::endl << std::endl;

    std::list<popcalculation::PerformanceTest*> pop_tests     = performance_analysis->getPOPTests();
    std::list<popcalculation::PerformanceTest*> io_tests      =  performance_analysis->getIOTests();
    std::list<popcalculation::PerformanceTest*> add_tests     =     performance_analysis->getAdditionalTests();
    std::list<popcalculation::PerformanceTest*> control_tests = performance_analysis->getControlTests();

    std::cout << " -- POP Ffficiencies -- " << std::endl;

    std::for_each( pop_tests.cbegin(), pop_tests.cend(), [ ]( const popcalculation::PerformanceTest* pt ){
        std::cout << pt->getHelpUrl() << std::endl << std::endl;
    } );
    std::cout << " -- IO Ffficiencies -- " << std::endl;

    std::for_each( io_tests.cbegin(), io_tests.cend(), [ ]( const popcalculation::PerformanceTest* pt ){
        std::cout << pt->getHelpUrl() << std::endl << std::endl;
    } );
    std::cout << " -- Additional Ffficiencies -- " << std::endl;
    std::for_each( add_tests.cbegin(), add_tests.cend(), [ ]( const popcalculation::PerformanceTest* pt ){
        std::cout << pt->getHelpUrl() << std::endl << std::endl;
    } );
    std::cout << " -- Control -- " << std::endl;
    std::for_each( control_tests.cbegin(), control_tests.cend(), [ ]( const popcalculation::PerformanceTest* pt ){
        std::cout << pt->getHelpUrl() << std::endl << std::endl;
    } );

    delete performance_analysis;
    delete cube;
}
