/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/**
 * \file cube4_pop_metrics.cpp
 * \brief Calculates different POPx metrics for selected cnode for every profile
 *
 */

#include "config.h"
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <iostream>
#include <limits>
#include <regex>
#include <sstream>
#include <string>
#include <unistd.h>
#include <vector>
#include <future>

#include "cube4_pop_metrics_helper.h"
#include "cube4_pop_metrics_help.h"
#include "Cube.h"
#include "CubeZfstream.h"
#include "CubeServices.h"
#include "CubeError.h"
#include "POPCalculation.h"
#include "PerformanceAnalysis.h"
#include "PerformanceTest.h"
#include "POPAuditAnalysis.h"
#include "POPHybridAuditAnalysisAdd.h"
#include "POPHybridAuditAnalysis.h"
#include "BSPOPHybridAuditAnalysis.h"


using namespace std;
using namespace cube;
using namespace services;
/**
 * Main program.
 * -
 * -
 */


int
main( int argc, char* argv[] )
{
    int                                                ch;
    std::string                                        analysis_name;
    std::vector <std::pair<std::string, std::string> > inputs;

    std::vector<std::pair<std::string, std::vector< std::pair<cube::list_of_cnodes, std::vector<double> > > > >                               pop_values;
    std::vector<std::string >                                                                                                                 pop_metric_names;
    std::vector<std::pair<std::string, std::vector< std::pair<cube::list_of_cnodes, std::vector<double> > > > >                               gpu_values;
    std::vector<std::string >                                                                                                                 gpu_metric_names;
    std::vector<std::pair<std::string, std::vector< std::pair<cube::list_of_cnodes, std::vector<double> > > > >                               io_values;
    std::vector<std::string >                                                                                                                 io_metric_names;
    std::vector<std::pair<std::string, std::vector< std::pair<cube::list_of_cnodes, std::vector<double> > > > >                               add_values;
    std::vector<std::string >                                                                                                                 add_metric_names;
    std::vector<std::pair<std::string, std::vector< std::pair<cube::list_of_cnodes, std::vector< std::tuple<double, double, double> > > > > > control_values;
    std::vector<std::string >                                                                                                                 control_metric_names;

    string                                 analysis             = "mpi";
    popcalculation::POP_ANALYSIS           pop_analysis         = popcalculation::PURE_MPI;
    popcalculation::PerformanceAnalysis*   performance_analysis = nullptr;
    bool                                   accumulated          = true;
    size_t                                 n_fois               = 0;
    std::vector<std::string>               cnode_names;
    std::vector<std::vector<std::string> > fois_names;
    std::string                            cnode_name = "__NO_NAME__";
    const std::string                      USAGE      = "Usage: " + std::string( argv[ 0 ] ) + " [-i] [-a <pop analysis>] [-c <FOI(s)>] [-h] [-? <pop analysis>] <cube1> <cube2> ..\n"

                                                        "  -a     Name of the POP analysis. Options: mpi,hybrid-add, hybrid-mult, bsc. Default: mpi\n"
                                                        "  -c     Name(s) or ID(s) of FOI(s). Default: calltree root(s). Comma separated\n"
                                                        "  -i     Performs the calculation individually for every listed FOI.\n"
                                                        "         Default: if omitted, the calculation is performed in an accumulated manner\n"
                                                        "  -h     Help; Output a brief help message.\n\n"
                                                        "  -?     Help; Output a long detailed help message.\n\n"
                                                        "Report bugs to <" PACKAGE_BUGREPORT ">\n";

    while ( ( ch = getopt( argc, argv, "ia:c:h?:" ) ) != -1 )
    {
        switch ( ch )
        {
            case 'i':
                accumulated = false;
                break;
            case 'a':
                analysis     = optarg;
                pop_analysis = ( analysis == "mpi" ) ?
                               popcalculation::PURE_MPI :
                               ( ( analysis == "hybrid-add" ) ?
                                 popcalculation::HYBRID_ADD :
                                 ( ( analysis == "hybrid-mult" )  ?
                                   popcalculation::HYBRID_MULT :
                                   ( ( analysis == "bsc" ) ?
                                     popcalculation::HYBRID_BSC :
                                     popcalculation::PURE_MPI
                                   )
                                 )
                               );
                break;
            case 'c':
                cnode_name = optarg;
                break;
            case 'h':
                cerr << USAGE << endl;
                exit( EXIT_SUCCESS );
                break;
            case '?':
                if ( !optarg )
                {
                    cerr << USAGE << endl;
                    exit( EXIT_SUCCESS );
                }
                analysis     = optarg;
                pop_analysis = ( analysis == "mpi" ) ?
                               popcalculation::PURE_MPI :
                               ( ( analysis == "hybrid-add" ) ?
                                 popcalculation::HYBRID_ADD :
                                 ( ( analysis == "hybrid-mult" )  ?
                                   popcalculation::HYBRID_MULT :
                                   ( ( analysis == "bsc" ) ?
                                     popcalculation::HYBRID_BSC :
                                     popcalculation::PURE_MPI
                                   )
                                 )
                               );
                popcalculation::printHelp( pop_analysis );
                exit( EXIT_SUCCESS );
                break;
            default:
                cerr << USAGE << "\nError: Wrong arguments.\n";
                exit( EXIT_FAILURE );
        }
    }

    popcalculation::parse_csint( cnode_name, cnode_names );
    if ( argc - optind > 0 )
    {
        string cur;
        int    ncube = 0;
        for ( int i = optind; i < argc; i++ )
        {
            cur = argv[ i ];
            std::string text = "Profile ";
            text += std::to_string( ncube++ );
            inputs.push_back( std::make_pair( text, std::string( cur ) ) );
        }
    }
    else
    {
        cerr << USAGE << "\nError: At least one file is required.\n\n";
        exit( EXIT_FAILURE );
    }

    cube::CubeProxy* cube = nullptr;


    for ( unsigned i = 0; i < inputs.size(); i++ )
    {
        try
        {
            std::cout << "Reading " << inputs[ i ].second << " ... " << flush;
            cube = CubeProxy::create( inputs[ i ].second );
            cube->openReport();
            std::cout << " done." << endl;


            switch ( pop_analysis )
            {
                case popcalculation::HYBRID_ADD:
                    performance_analysis = new hybaddanalysis::POPHybridAuditPerformanceAnalysisAdd( cube );
                    break;
                case popcalculation::HYBRID_MULT:
                    performance_analysis = new hybanalysis::POPHybridAuditPerformanceAnalysis( cube );
                    break;
                case popcalculation::HYBRID_BSC:
                    performance_analysis = new bscanalysis::BSPOPHybridAuditPerformanceAnalysis( cube );
                    break;
                case popcalculation::PURE_MPI:
                default:
                    performance_analysis = new mpianalysis::POPAuditPerformanceAnalysis( cube );
                    break;
            }




            analysis_name = performance_analysis->name();

            if ( !performance_analysis->isActive() )
            {
                std::cout << "[WARNING] It seems that the POP Analysis \"" << analysis_name << "\" cannot be used with this profile. Interpret the result with care." << std::endl;
            }

            std::cout << "Calculating";


            std::list<popcalculation::PerformanceTest*> pop_metrics = performance_analysis->getPOPTests();
            pop_metric_names.clear();
            std::for_each( pop_metrics.crbegin(), pop_metrics.crend(), [ &pop_metric_names ]( const popcalculation::PerformanceTest* pt ){
                pop_metric_names.push_back( pt->name() );
            } );
            std::list<popcalculation::PerformanceTest*> gpu_metrics = performance_analysis->getGPUTests();
            gpu_metric_names.clear();
            std::for_each( gpu_metrics.crbegin(), gpu_metrics.crend(), [ &gpu_metric_names ]( const popcalculation::PerformanceTest* pt ){
                gpu_metric_names.push_back( pt->name() );
            } );

            std::list<popcalculation::PerformanceTest*> io_metrics = performance_analysis->getIOTests();
            io_metric_names.clear();
            std::for_each( io_metrics.crbegin(), io_metrics.crend(), [ &io_metric_names ]( const popcalculation::PerformanceTest* pt ){
                io_metric_names.push_back( pt->name() );
            } );

            std::list<popcalculation::PerformanceTest*> add_metrics = performance_analysis->getAdditionalTests();
            add_metric_names.clear();
            std::for_each( add_metrics.crbegin(), add_metrics.crend(), [ &add_metric_names ]( const popcalculation::PerformanceTest* pt ){
                add_metric_names.push_back( pt->name() );
            } );
            std::list<popcalculation::PerformanceTest*> control_metrics = performance_analysis->getControlTests();
            control_metric_names.clear();
            std::for_each( control_metrics.crbegin(), control_metrics.crend(), [ &control_metric_names ]( const popcalculation::PerformanceTest* pt ){
                control_metric_names.push_back( pt->name() );
            } );

            std::vector<cube::list_of_cnodes> list_of_fois;
            std::map<uint64_t, cube::Cnode*>  map_cnodes;

            auto create_cnodes = [ &map_cnodes, &cube ]( std::string& cnode_name_ )
                                 {
                                     if ( cnode_name_ == "__NO_NAME__" )
                                     {
                                         const std::vector<cube::Cnode*>& croots = cube->getRootCnodes();
                                         std::for_each( croots.begin(), croots.end(), [ &map_cnodes ]( cube::Cnode* c ){
                        map_cnodes[ c->get_id() ] = c;
                    } );
                                         std::cout << std::endl;
                                     }
                                     else
                                     {
                                         const std::vector<cube::Cnode*>& allcnodes = cube->getCnodes();
                                         bool                             foundname = false;
                                         std::for_each( allcnodes.begin(), allcnodes.end(), [ &map_cnodes, &cnode_name_, &foundname ]( cube::Cnode* c ){
                        if ( cnode_name_ ==  ( c->get_callee()->get_name() ) )
                        {
                            map_cnodes[ c->get_id() ] = c;
                            foundname = true;
                        }
                    } );

                                         if ( !foundname ) // if no name has been found -> try as ID... worse case -> ID=0
                                         {
                                             // here we are coz name didnt match... try to interpret it as an ID
                                             uint32_t cnode_id = std::atoi( cnode_name_.c_str() );
                                             std::for_each( allcnodes.begin(), allcnodes.end(), [ &map_cnodes, &cnode_id ]( cube::Cnode* c ){
                            if ( c->get_id() == cnode_id )
                            {
                                map_cnodes[ c->get_id() ] = c;
                            }
                        } );
                                         }
                                     }
                                 };


            std::for_each( cnode_names.begin(), cnode_names.end(), create_cnodes );


            std::vector<std::string> _foi_names;
            if ( accumulated )
            {
                std::string          foi_name;
                cube::list_of_cnodes cnodes;
                std::for_each( map_cnodes.cbegin(), map_cnodes.cend(), [ &cnodes ]( const std::pair<uint64_t, cube::Cnode*>& c )
                {
                    cube::cnode_pair cpair;
                    cpair.first  = c.second;
                    cpair.second = cube::CUBE_CALCULATE_INCLUSIVE;
                    cnodes.push_back( cpair );
                } );
                popcalculation::POPCalculation::correctCnodes( cube, cnodes );
                std::for_each( cnodes.cbegin(), cnodes.cend(), [ &foi_name ]( const cube::cnode_pair& c )
                {
                    foi_name += ( ( !foi_name.empty() ) ? std::string( "," ) : std::string( "" ) ) +  c.first->get_callee()->get_name() + "[id=" + std::to_string( c.first->get_id() ) + "]";
                } );
                list_of_fois.push_back( cnodes );
                _foi_names.push_back( foi_name );
            }
            else
            {
                std::for_each( map_cnodes.cbegin(), map_cnodes.cend(), [ &_foi_names, &list_of_fois, &cube ]( const std::pair<uint64_t, cube::Cnode*>& c )
                {
                    std::string foi_name;
                    cube::list_of_cnodes cnodes;
                    cube::cnode_pair cpair;
                    cpair.first  = c.second;
                    cpair.second = cube::CUBE_CALCULATE_INCLUSIVE;
                    cnodes.push_back( cpair );
                    popcalculation::POPCalculation::correctCnodes( const_cast<cube::CubeProxy*>( cube ), cnodes );
                    std::for_each( cnodes.cbegin(), cnodes.cend(), [ &foi_name ]( const cube::cnode_pair& c )
                    {
                        foi_name += ( ( !foi_name.empty() ) ? std::string( "," ) : std::string( "" ) ) +  c.first->get_callee()->get_name() + "[id=" + std::to_string( c.first->get_id() ) + "]";
                    } );
                    list_of_fois.push_back( cnodes );
                    _foi_names.push_back( foi_name );
                } );
            }
            fois_names.push_back( _foi_names );


// ----   ASYNC CALCULATION
            std::list<popcalculation::PerformanceTest*> all_tests = performance_analysis->getAllTestsForCalculation();
// ---- DONE, NOW COLLECT RESULTS




            std::vector< std::pair<cube::list_of_cnodes, std::vector<double> > > lcvalues;

            std::for_each( list_of_fois.cbegin(), list_of_fois.cend(), [ &pop_metrics, &lcvalues, &all_tests, &cube ]( const cube::list_of_cnodes& c )
            {
                cube::list_of_cnodes new_cnodes = c;

// ----   ASYNC CALCULATION

                std::vector<std::future<void> > calculation_done;
                auto asynx_calculate_metric = [ ]( const cube::list_of_cnodes& c,  popcalculation::PerformanceTest* t )
                                              {
                                                  if ( t->isActive() )
                                                  {
                                                      t->apply( c );
                                                  }
                                              };

                std::for_each( all_tests.begin(), all_tests.end(), [ &c, &calculation_done, &asynx_calculate_metric ]( popcalculation::PerformanceTest* t )
                {
                    calculation_done.push_back(  std::async( std::launch::async, asynx_calculate_metric, c, t )  );
                } );
                std::for_each( calculation_done.begin(), calculation_done.end(), [ ]( std::future<void>& _f )
                {
                    _f.get(); // waiting for the result
                } );
// ---- DONE, NOW COLLECT RESULTS


                std::vector<double> _r;
                auto calculate_pop_metric = [ &_r ]( popcalculation::PerformanceTest* t )
                                            {
                                                if ( t->isActive() )
                                                {
                                                    double v = t->value();
                                                    _r.push_back( v );
                                                }
                                                else
                                                {
                                                    _r.push_back( std::numeric_limits<double>::quiet_NaN() );
                                                }
                                                std::cout << ".";
                                            };


                std::for_each( pop_metrics.crbegin(), pop_metrics.crend(), calculate_pop_metric );
                lcvalues.push_back( std::make_pair( c, _r ) );
            } );
            pop_values.push_back(
                std::make_pair(
                    inputs[ i ].first,
                    lcvalues
                    )
                );

            // --- GPU
            lcvalues.clear();
            std::for_each( list_of_fois.cbegin(), list_of_fois.cend(), [ &gpu_metrics, &lcvalues ]( const cube::list_of_cnodes& c )
            {
                std::vector<double> _r;
                auto calculate_gpu_metric = [ &_r ]( popcalculation::PerformanceTest* t )
                                            {
                                                if ( t->isActive() )
                                                {
                                                    double v = t->value();
                                                    _r.push_back( v );
                                                }
                                                else
                                                {
                                                    _r.push_back( std::numeric_limits<double>::quiet_NaN() );
                                                }
                                                std::cout << ".";
                                            };


                std::for_each( gpu_metrics.crbegin(), gpu_metrics.crend(), calculate_gpu_metric );
                lcvalues.push_back( std::make_pair( c, _r ) );
            } );
            gpu_values.push_back(
                std::make_pair(
                    inputs[ i ].first,
                    lcvalues
                    )
                );
            // --- IO
            lcvalues.clear();
            std::for_each( list_of_fois.cbegin(), list_of_fois.cend(), [ &io_metrics, &lcvalues ]( const cube::list_of_cnodes& c )
            {
                std::vector<double> _r;
                auto calculate_io_metric = [ &_r ]( popcalculation::PerformanceTest* t )
                                           {
                                               if ( t->isActive() )
                                               {
                                                   double v = t->value();
                                                   _r.push_back( v );
                                               }
                                               else
                                               {
                                                   _r.push_back( std::numeric_limits<double>::quiet_NaN() );
                                               }
                                               std::cout << ".";
                                           };


                std::for_each( io_metrics.crbegin(), io_metrics.crend(), calculate_io_metric );
                lcvalues.push_back( std::make_pair( c, _r ) );
            } );
            io_values.push_back(
                std::make_pair(
                    inputs[ i ].first,
                    lcvalues
                    )
                );

            // add metric

            lcvalues.clear();
            std::for_each( list_of_fois.cbegin(), list_of_fois.cend(), [ &add_metrics, &lcvalues ]( const cube::list_of_cnodes& c )
            {
                std::vector<double> _r;
                auto calculate_add_metric = [ &_r ]( popcalculation::PerformanceTest* t )
                                            {
                                                if ( t->isActive() )
                                                {
                                                    double v = t->value();
                                                    _r.push_back( v );
                                                }
                                                else
                                                {
                                                    _r.push_back( std::numeric_limits<double>::quiet_NaN() );
                                                }
                                                std::cout << ".";
                                            };


                std::for_each( add_metrics.crbegin(), add_metrics.crend(), calculate_add_metric );
                lcvalues.push_back( std::make_pair( c, _r ) );
            } );
            add_values.push_back(
                std::make_pair(
                    inputs[ i ].first,
                    lcvalues
                    )
                );


            // control metric
            std::vector< std::pair<cube::list_of_cnodes, std::vector< std::tuple< double, double, double> > > > c_lcvalues;
            c_lcvalues.clear();
            std::for_each( list_of_fois.cbegin(), list_of_fois.cend(), [ &control_metrics, &c_lcvalues ]( const cube::list_of_cnodes& c )
            {
                std::vector<std::tuple<double, double, double> > _r;
                auto calculate_control_metric = [ &_r ]( popcalculation::PerformanceTest* t )
                                                {
                                                    if ( t->isActive() )
                                                    {
                                                        double min_v = t->min_value();
                                                        double v = t->value();
                                                        double max_v = t->max_value();
                                                        _r.push_back( std::make_tuple( min_v, v, max_v ) );
                                                    }
                                                    else
                                                    {
                                                        _r.push_back( std::make_tuple( std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() ) );
                                                    }
                                                    std::cout << ".";
                                                };


                std::for_each( control_metrics.crbegin(), control_metrics.crend(), calculate_control_metric );
                c_lcvalues.push_back( std::make_pair( c, _r ) );
            } );
            std::cout << std::endl;
            control_values.push_back(
                std::make_pair(
                    inputs[ i ].first,
                    c_lcvalues
                    )
                );







            n_fois = std::max( n_fois, list_of_fois.size() );



            delete performance_analysis;
            delete cube;
        }
        catch ( const cube::RuntimeError& e )
        {
            delete cube;
            exit( EXIT_FAILURE );
        }
    }
    std::cout << std::endl << "-------------- Result -------------- " << std::endl;
    std::string header = "POP Metric";
    header.resize( 30, ' ' );

    std::vector<std::string> pop_output_lines     = pop_metric_names;
    std::vector<std::string> gpu_output_lines     = gpu_metric_names;
    std::vector<std::string> io_output_lines      = io_metric_names;
    std::vector<std::string> add_output_lines     = add_metric_names;
    std::vector<std::string> control_output_lines = control_metric_names;

    size_t n_pop_tests     = pop_output_lines.size();
    size_t n_gpu_tests     = gpu_output_lines.size();
    size_t n_io_tests      = io_output_lines.size();
    size_t n_add_tests     = add_output_lines.size();
    size_t n_control_tests = control_output_lines.size();
    control_output_lines.resize( 3 * n_control_tests );  // we printout 3 lines per test
    size_t n_cubes = inputs.size();

    for ( size_t nt = 0; nt < n_pop_tests; ++nt )
    {
        pop_output_lines[ nt ].resize( 30, ' ' );
    }
    for ( size_t nt = 0; nt < n_gpu_tests; ++nt )
    {
        gpu_output_lines[ nt ].resize( 30, ' ' );
    }
    for ( size_t nt = 0; nt < n_io_tests; ++nt )
    {
        io_output_lines[ nt ].resize( 30, ' ' );
    }
    for ( size_t nt = 0; nt < n_add_tests; ++nt )
    {
        add_output_lines[ nt ].resize( 30, ' ' );
    }
    for ( size_t nt = 0; nt < n_control_tests; ++nt )
    {
        control_output_lines[ 3 * nt ] += "; min";
        control_output_lines[ 3 * nt ].resize( 30, ' ' );

        control_output_lines[ 3 * nt + 1 ] += "               ";
        control_output_lines[ 3 * nt + 1 ] += "  avg";
        control_output_lines[ 3 * nt + 1 ].resize( 30, ' ' );

        control_output_lines[ 3 * nt + 2 ] += "               ";
        control_output_lines[ 3 * nt + 2 ] += "  max";
        control_output_lines[ 3 * nt + 2 ].resize( 30, ' ' );
    }


    for ( size_t nc = 0; nc < n_cubes; ++nc )
    {
        header.resize( 30 + ( nc + 1 ) * 25, ' ' );
        header += inputs[ nc ].first;
        std::cout << inputs[ nc ].second << " -> " << inputs[ nc ].first << std::endl;
    }
    std::cout << std::endl << analysis_name << std::endl;
    std::cout << "------------------------------------ " << std::endl;
    for ( size_t nf = 0; nf < n_fois; ++nf )
    {
        std::cout <<
            "Calculating for \"" <<
            fois_names[ 0 ][ nf ] <<
            '"' <<
            std::endl <<
            "------------------------------------ " <<
            std::endl << std::endl <<
            header <<
            std::endl <<
            "------------------------------------ " <<
            std::endl;

        for ( int nt = n_pop_tests - 1; nt >= 0; --nt )
        {
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                pop_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                pop_output_lines[ nt ] += std::to_string( pop_values[ nc ].second[ nf ].second[ nt ] );
                pop_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << pop_output_lines[ nt ] << std::endl;
        }
        std::cout <<
            "------------------------------------ " <<
            std::endl << std::endl <<
            "GPU Metric" <<
            std::endl <<
            "------------------------------------ " <<
            std::endl;
        for ( int nt = n_gpu_tests - 1; nt >= 0; --nt )
        {
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                gpu_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                gpu_output_lines[ nt ] += std::to_string( gpu_values[ nc ].second[ nf ].second[ nt ] );
                gpu_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << gpu_output_lines[ nt ] << std::endl;
        }
        std::cout <<
            "------------------------------------ " <<
            std::endl << std::endl <<
            "IO Metric" <<
            std::endl <<
            "------------------------------------ " <<
            std::endl;
        for ( int nt = n_io_tests - 1; nt >= 0; --nt )
        {
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                io_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                io_output_lines[ nt ] += std::to_string( io_values[ nc ].second[ nf ].second[ nt ] );
                io_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << io_output_lines[ nt ] << std::endl;
        }
        std::cout <<
            "------------------------------------ " <<
            std::endl << std::endl <<
            "Additional Metrics" <<
            std::endl <<
            "------------------------------------ " <<
            std::endl;

        for ( int nt = n_add_tests - 1; nt >= 0; --nt )
        {
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                add_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                add_output_lines[ nt ] += std::to_string( add_values[ nc ].second[ nf ].second[ nt ] );
                add_output_lines[ nt ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << add_output_lines[ nt ] << std::endl;
        }
        std::cout <<
            "------------------------------------ " <<
            std::endl << std::endl <<
            "FOA Quality Control Metrics" <<
            std::endl <<
            "------------------------------------ " <<
            std::endl;
        for ( int nt = n_control_tests - 1; nt >= 0; --nt )
        {
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                double range =  std::get<1>( control_values[ nc ].second[ nf ].second[ nt ] ) -
                               std::get<0>( control_values[ nc ].second[ nf ].second[ nt ] );
                double prange = std::fabs( range /  std::get<1>( control_values[ nc ].second[ nf ].second[ nt ] ) * 100 );
                control_output_lines[ 3 * nt ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                control_output_lines[ 3 * nt ] += std::to_string( std::get<0>( control_values[ nc ].second[ nf ].second[ nt ] ) );
                control_output_lines[ 3 * nt ] += std::string( ", " );
                control_output_lines[ 3 * nt ] += std::to_string( prange );
                control_output_lines[ 3 * nt ] += std::string( "%" );
                control_output_lines[ 3 * nt ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << control_output_lines[ 3 * nt ] << std::endl;
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                control_output_lines[ 3 * nt + 1 ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                control_output_lines[ 3 * nt + 1 ] += std::to_string( std::get<1>( control_values[ nc ].second[ nf ].second[ nt ] ) );
                control_output_lines[ 3 * nt + 1 ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << control_output_lines[ 3 * nt + 1 ] << std::endl;
            for ( size_t nc = 0; nc < n_cubes; ++nc )
            {
                double range =  std::get<2>( control_values[ nc ].second[ nf ].second[ nt ] ) -
                               std::get<1>( control_values[ nc ].second[ nf ].second[ nt ] );
                double prange = std::fabs( range /  std::get<1>( control_values[ nc ].second[ nf ].second[ nt ] ) * 100 );
                control_output_lines[ 3 * nt + 2 ].resize( 30 + ( nc + 1 ) * 25, ' ' );
                control_output_lines[ 3 * nt + 2 ] += std::to_string( std::get<2>( control_values[ nc ].second[ nf ].second[ nt ] ) ) + std::string( ", " ) + std::to_string( prange ) + std::string( "%" );
                control_output_lines[ 3 * nt + 2 ].resize( 30 + ( nc + 1 ) * 25 + 25, ' ' );
            }
            std::cout << control_output_lines[ 3 * nt + 2 ] << std::endl;
        }

        std::cout << std::endl << std::endl;
    }
    control_values.clear();
    cnode_names.clear();
    fois_names.clear();
    inputs.clear();
    pop_values.clear();
    gpu_values.clear();
    add_values.clear();
    io_values.clear();
    control_values.clear();

    pop_metric_names.clear();
    gpu_metric_names.clear();
    io_metric_names.clear();
    add_metric_names.clear();
    control_metric_names.clear();
    pop_output_lines.clear();
    gpu_output_lines.clear();
    io_output_lines.clear();
    add_output_lines.clear();
    control_output_lines.clear();


    exit( EXIT_SUCCESS );
}
