/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2016-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2014-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2014-2015                                                **
**  RWTH Aachen University, JARA-HPC                                       **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/*-------------------------------------------------------------------------*/
/**
 *  @file
 *  @ingroup CUBE_tools
 *  @brief   Server application for the client-server network layer.
 *
 *  The cube server will listen on a given port for incoming client
 *  connections to serve information read and computed from a Cube file
 *  local to the server.
 **/
/*-------------------------------------------------------------------------*/
#include "config.h"

#include <cerrno>
#include <csignal>
#include <cstdlib>
#include <getopt.h>
#include <iostream>
#include <sstream>
#include <string>
#include <sys/types.h>
#include <unistd.h>
#include <thread>
#include <future>
#include <vector>
#include <random>
#include <algorithm>

#include "CubeError.h"
#include "CubeNetworkRequest.h"
#include "CubeServerCallbackData.h"
#include "CubeServerConnection.h"
#include "CubePlatformsCompat.h"
#include "CubeSocket.h"
#include "CubeUrl.h"

#include "ServerWorker.h"
#define CUBELIB_DEBUG_MODULE_NAME CUBE_SERVER
#include <UTILS_Debug.h>

#ifdef HAVE_CUBELIB_DEBUG
#include <sstream>
#endif

#ifdef HAVE_DLOPEN
#include "CubePluginManager.h"
#endif

using namespace std;
using namespace cube;

string CUBE_SERVER_NAME;
// an array of 256 unique non-repeating animal names
static const string animalNames[ 248 ] =
{ "Beaver",            "Coyote",                           "Goose",                                          "Hawk",
  "Llama",             "Puma",                             "Quail",                                          "Raccoon",
  "Falcon",            "Finch",                            "Fox",                                            "Gazelle",
  "Lion",              "Ocelot",                           "Otter",                                          "Arctic Fox",
  "Rabbit",            "Tiger",                            "Vulture",                                        "Baboon",
  "Cheetah",           "Chipmunk",                         "Gorilla",                                        "Hyena",
  "Ibex",              "Jackal",                           "Jaguar",                                         "Kangaroo",
  "Mink",              "Mole",                             "Monkey",                                         "Ostrich",
  "Pelican",           "Rhinoceros",                       "Sheep",                                          "Skunk",
  "Sloth",             "Squirrel",                         "Vole",                                           "Wildebeest",
  "Alpaca",            "Badger",                           "Bearded Dragon",                                 "Grizzly Bear",
  "Buffalo",           "Capybara",                         "Crow",                                           "Deer",
  "Dingo",             "Donkey",                           "Dormouse",                                       "Duck",
  "Eagle",             "Elephant",                         "Emu",                                            "Flamingo",
  "Gibbon",            "Giraffe",                          "Gnu",                                            "Goat",
  "Hedgehog",          "Hippopotamus",                     "Horse",                                          "Hummingbird",
  "Impala",            "Leopard",                          "Lemur",                                          "Lynx",
  "Marmot",            "Marten",                           "Mongoose",                                       "Muskrat",
  "Opossum",           "Parrot",                           "Partridge",                                      "Penguin",
  "Pheasant",          "Pig",                              "Prairie Dog",                                    "Python",
  "Reindeer",          "Roadrunner",                       "Salamander",                                     "Serval",
  "Toucan",            "Turkey",                           "Turtle",                                         "Wallaby",
  "Walrus",            "Wolverine",                        "Wombat",                                         "Snake",
  "Woodchuck",         "Aardvark",                         "Antilope",                                       "Barracuda",
  "Basilisk",          "Bison",                            "Blue Jay",                                       "Tern",
  "Boar",              "Bobcat",                           "Camel",                                          "Caracal",
  "Catfish",           "Chameleon",                        "Civet",                                          "Cormorant",
  "Cougar",            "Cow",                              "Crane",                                          "Crocodile",
  "Dolphin",           "Dove",                             "Echidna",                                        "Ferret",
  "Frog",              "Gecko",                            "Goldfinch",                                      "Tapir",
  "Gopher",            "Hamster",                          "Hare",                                           "Heron",
  "Hornet",            "Hummingbird",                      "Iguana",                                         "Kiwa",
  "Koala",             "Kudu",                             "Lark",                                           "Leopard",
  "Lobster",           "Macaw",                            "Magpie",                                         "Meerkat",
  "Moorhen",           "Mule",                             "Newt",                                           "Osprey",
  "Otter",             "Owl",                              "Panda",                                          "Panther",
  "Parakeet",          "Peacock",                          "Platypus",                                       "Porcupine",
  "Rattlesnake",       "Raven",                            "Scorpion",                                       "Seahorse",
  "Seal",              "Shark",                            "Shrew",                                          "Snail",
  "Snipe",             "Starling",                         "Stork",                                          "Swallow",
  "Tortoise",          "Viper",                            "Warthog",                                        "Weasel",
  "Whale",             "Wildcat",                          "Wren",                                           "Armadillo",
  "Polar Bear",        "Swan",                             "Woodpecker",                                     "Pigeon",
  "Antelope",          "Caribou",                          "Elk",                                            "Hog",
  "Nutria",            "Gerbil",                           "Mice",                                           "Sea Lion",
  "Deer Mouse",        "Rats",                             "Chimpanzee",                                     "Orangutan",
  "Aye-Aye",           "Spider Monkey",                    "Howler Monkey",                                  "Binturong",
  "White-Tailed Deer", "Lamb",                             "Yak",                                            "Fennec Fox",
  "Wild Boar",         "Zebra",                            "Dromedary",                                      "Barn Owl",
  "Bactrian",          "Hippo",                            "Wild Dog",                                       "Horned Owl",
  "Grey Wolf",         "Red Wolf",                         "Maned Wolf",                                     "Peregrine Falcon",
  "Bush Dog",          "Dhole",                            "Bat",                                            "Flying Fox",
  "Screech Owl",       "Snowy Owl",                        "Burrowing Owl",                                  "Long-Eared Owl",
  "Short-Eared Owl",   "Barn Swallow",                     "House Wren",                                     "Mockingbird",
  "American Robin",    "Oriole",                           "Red-Winged Blackbird",                           "American Kestrel",
  "Cowbird",           "Black-Capped Chickadee",           "Bluebird",                                       "Sparrow",
  "Cardinal",          "House Finch",                      "Pine Siskin",                                    "Tufted Titmouse",
  "Grebe",             "Coot",                             "Bittern",                                        "White Pelican",
  "Loon",              "Gannet",                           "Kingfisher",                                     "Murrelet",
  "Sandpiper",         "Grouse",                           "Albatross",                                      "Egret",
  "Condor",            "Kiwi",                             "Ptarmigan",                                      "Plover",
  "Oystercatcher",     "Avocet",                           "Stilt",                                          "Gallinule" };



CubeServerVerbosityLevel CUBE_SERVER_VERBOSITY_LEVEL;

static const std::string
    VerbosityNames[ CUBE_SERVER_VERBOSITY_N ] =
{
    "mute",
    "launch",
    "full"
};


const string&
generate_cube_server_name()
{
    std::random_device              rd;
    std::mt19937                    eng( rd() );
    std::uniform_int_distribution<> distr( 0, 249 );
    return animalNames[ distr( eng ) ];
}

/// @brief Print command-line options and usage info
///
const string&
getUsage()
{
    stringstream portNo;
    portNo << Url::getDefaultPort();
    static const string USAGE =
        "Usage: cube_server [options]\n"
        " -h, -?     Print this help message.\n"
        " -n <name>  Use <name> as a name of the server in report messages.\n"
        " -p N       Bind socket on port N (default port: " + portNo.str() + ")\n"
        " -s <path>  Bind unix socket to path (only for use with CubeWebservice)\n"
        " -v <level> Verbosity <level>, one of \"mute\", \"launch\", \"full\" (default level: "  + VerbosityNames[ CUBE_SERVER_VERBOSITY_LAUNCH ] + ")\n"
        " -c         Close server when last taks is finished.\n\n"
        "Report bugs to <" PACKAGE_BUGREPORT ">\n";

    return USAGE;
}

vector<string>
split( const string& str, char delimiter )
{
    vector<string> result;

    auto endOfString = str.end();
    auto search      = [ delimiter ]( char c ){
                           return c == delimiter;
                       };
    auto idx = str.begin();
    while ( idx != endOfString )
    {
        idx = find_if_not( idx, endOfString, search );
        if ( idx != endOfString )
        {
            auto end = find_if( idx, endOfString, search );
            result.push_back( string( idx, end ) );
            idx = end;
        }
    }
    return result;
}

int
main( int   argc,
      char* argv[] )
{
    size_t portNo = Url::getDefaultPort();
    string unix_socket;
    CUBE_SERVER_VERBOSITY_LEVEL = CUBE_SERVER_VERBOSITY_LAUNCH;
    bool         closeServerWithEndOfTask = false;
    stringstream message;

    int         option_arg;
    std::string _level;
    // check for command line parameters
    while ( ( option_arg = getopt( argc, argv, "h?n:p:s:cv:" ) ) != -1 )
    {
        switch ( option_arg )
        {
            case 'h':
            case '?':
                cerr << getUsage() << endl;
                exit( EXIT_SUCCESS );
                break;
            case 'n':
                CUBE_SERVER_NAME = optarg;
                break;
            case 'v':
                _level                      = optarg;
                CUBE_SERVER_VERBOSITY_LEVEL = ( _level == VerbosityNames[ CUBE_SERVER_VERBOSITY_MUTE ] ) ? CUBE_SERVER_VERBOSITY_MUTE :
                                              (
                    ( _level == VerbosityNames[ CUBE_SERVER_VERBOSITY_LAUNCH ] ) ? CUBE_SERVER_VERBOSITY_LAUNCH : CUBE_SERVER_VERBOSITY_FULL
                                              );
                break;
            case 'c':
                closeServerWithEndOfTask = true;
                break;
            case 'p':
                portNo = atoi( optarg );
                break;
            case 's':
                unix_socket = optarg; // use unix socket instead of port
                break;
        }
    }

    if ( CUBE_SERVER_NAME.empty() )
    {
        CUBE_SERVER_NAME = ( getenv( "CUBE_SERVER_NAME" ) != nullptr ) ? getenv( "CUBE_SERVER_NAME" ) : generate_cube_server_name();
    }

#define CHECK_PLUGIN_IN_PATH( path ) \
    { \
        defaultPath = realpath( PKGDATADIR "/../../" path, nullptr ); \
        if ( defaultPath != nullptr ) \
        { \
            pathes.push_back( defaultPath ); \
        } \
        defaultPath = realpath( PKGDATADIR "/../../" path "/cube-plugins", nullptr ); \
        if ( defaultPath != nullptr ) \
        { \
            pathes.push_back( defaultPath ); \
        } \
    }


#ifdef HAVE_DLOPEN
    // check for plugins
    const char* pathEnv    = "CUBE_PLUGIN_DIR";
    const char* pluginPath = getenv( pathEnv );

    vector<string> pathes;
    if ( pluginPath )
    {
        pathes = split( string( pluginPath ), ':' );
    }
    char* defaultPath = nullptr;

    CHECK_PLUGIN_IN_PATH( "lib" )
    CHECK_PLUGIN_IN_PATH( "lib64" )
    CHECK_PLUGIN_IN_PATH( "lib32" )

    std::string pathStr;
    if ( defaultPath != nullptr )
    {
        pathes.push_back( defaultPath );
        // add "." as default plugin path, if it doesn't already exist
        if ( find_if( pathes.begin(), pathes.end(), [ ]( const string& str ){
            return str == ".";
        } ) == pathes.end() )
        {
            pathes.push_back( "." );
        }
        for ( size_t i = 0; i < pathes.size(); ++i )
        {
            pathStr += ( i > 0 ? ":" : "" ) + pathes[ i ];
        }
    }

    PluginManager* pm = PluginManager::getInstance();
    if ( CUBE_SERVER_VERBOSITY_LEVEL > CUBE_SERVER_VERBOSITY_MUTE )
    {
        cout << "plugin search path (" << pathEnv << "): " << pathStr << endl;
    }
    pm->loadPlugins( pathes, CUBE_SERVER_VERBOSITY_LEVEL );

#endif

    SocketPtr                       socket     = Socket::create();
    ServerConnection::Ptr           connection = nullptr;
    std::vector<std::future<bool> > connections_futures;

    try
    {
        if ( unix_socket.empty() )
        {
            connection = ServerConnection::create(  socket, portNo );
        }
        else
        {
            connection = ServerConnection::create( socket, unix_socket );
        }
    }
    catch ( NetworkError& theError )
    {
        cerr << "cube_server: " << theError.what() << endl;
        exit( EXIT_FAILURE );
    }
    if ( CUBE_SERVER_VERBOSITY_LEVEL > CUBE_SERVER_VERBOSITY_MUTE )
    {
        message << "[" << getpid() << "] Cube Server(" << CUBE_SERVER_NAME << "):"
                << connection->getInfoString();
        cerr << message.str() << endl;
        message.str( "" );
    }
    //     reset connection protocol
    connection->setProtocolVersion( 0 );
    while ( connection->isListening() )
    {
        // wait for incoming connection

        if ( CUBE_SERVER_VERBOSITY_LEVEL > CUBE_SERVER_VERBOSITY_MUTE )
        {
            message << "cube_server[" << getpid() << "](" << CUBE_SERVER_NAME << "): "
                    << connection->getInfoString() << " Waiting for connections on port " << portNo << ".";
            cerr << message.str() << endl;
            message.str( "" );
        }

        try
        {
            connection->accept();
            ServerConnection::Ptr clientConnection = std::shared_ptr<ServerConnection> ( new ServerConnection( connection ) );
            // creating task and handing over connection
            connections_futures.emplace_back( std::async( std::launch::async,  serveClient, clientConnection, closeServerWithEndOfTask ) );
        }
        catch ( const exception& err )
        {
            message << "cube_server[" << getpid() << "](" << CUBE_SERVER_NAME << "):"
                    << connection->getInfoString()  << err.what();
            cerr << message.str() << endl;
            message.str( "" );
            continue;
        }
    }
#ifdef HAVE_DLOPEN
    delete pm;
#endif

#if defined( CUBE_NETWORK_DEBUG )
    if ( CUBE_SERVER_VERBOSITY_LEVEL > CUBE_SERVER_VERBOSITY_MUTE )
    {
        message << "cube_server[" << getpid() << "(" << CUBE_SERVER_NAME << "):  Done.";
        cerr << message.str() << endl;
        message.str( "" );
    }
#endif
}
