/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2016-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef REQUEST_TASK_H
#define REQUEST_TASK_H

#include <iostream>
#include <sstream>
#include <string>

#include "CubeNetworkRequest.h"
#include "CubeServerCallbackData.h"
#include "CubeServerConnection.h"
#include "CubeNetworkRequest.h"
#include "ServerWorker.h"

extern std::string                    CUBE_SERVER_NAME;
extern cube::CubeServerVerbosityLevel CUBE_SERVER_VERBOSITY_LEVEL;
namespace cube
{
/**
 * @brief The RequestTask class holds the data of a cube request and the method work() to execute it and to
 * send the result to the client.
 */
class RequestTask
{
public:
    RequestTask( ServerConnection::Ptr connection,
                 NetworkRequest::Ptr   request,
                 ServerCallbackData*   data
                 ) : connection_( connection ), request_( request ), data_( data )
    {
        taskName = request_->getName();
        peerInfo = connection_->getInfoString();
    }

    void
    work()
    {
        request_->processRequest( data_ ); // calculate result

        std::stringstream message;
        try
        {
            request_->sendResponse( *connection_, data_ ); // send result to client

#if defined( CUBE_NETWORK_DEBUG )
            if ( CUBE_SERVER_VERBOSITY_LEVEL == CUBE_SERVER_VERBOSITY_FULL )
            {
                message <<  "cube_server[" << getpid() << "(" << CUBE_SERVER_NAME << ":" << peerInfo << "){Task: " << taskName << "}:  Sent response number #" << request_->getSequenceNumber();
                std::cerr << message.str() << std::endl;
                message.str( "" );
            }
#endif
        }
        catch ( NetworkError& theError )
        {
            message << "cube_server[" << getpid() << "(" << CUBE_SERVER_NAME << ":" << peerInfo << "){Task: " << taskName << "}:  ### Failed to send response.\n"
                    << "cube_server[" << getpid() << "(" << CUBE_SERVER_NAME << ":" << peerInfo << "):  ### " << theError.what();
            std::cerr << message.str() << std::endl;
            message.str( "" );
            connection_->disconnect();
        }
        catch ( const std::exception& theError )
        {
            message << "cube_server[" << getpid() << "(" << CUBE_SERVER_NAME << ":" << peerInfo << "){Task: " << taskName << "}:  ### Unexpected exception while handling request.\n"
                    << "cube_server[" << getpid() << "(" << CUBE_SERVER_NAME << ":" << peerInfo << "):  ### " << theError.what();
            std::cerr << message.str() << std::endl;
            message.str( "" );
        }
    }
private:
    ServerConnection::Ptr connection_;
    NetworkRequest::Ptr   request_;
    ServerCallbackData*   data_;
    std::string           taskName;
    std::string           peerInfo;
};

class ServerCloseException
{
public:
    ServerCloseException()
    {
    }
    ~ServerCloseException()
    {
    }
};
}
#endif
