/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



/**
 * \file cube4_dump.cpp
 * \brief Extracts data from the cube in various format
 *
 */
/******************************************

   Performance Algebra Operation: MERGE

 *******************************************/
#include "config.h"

#include <cstdlib>
#include <iostream>
#include <string>
#include <sstream>
#include <fstream>
#include <unistd.h>


#include "Cube.h"
#include "CubeCnode.h"
#include "CubeMachine.h"
#include "CubeMetric.h"
#include "CubeRegion.h"
#include "CubeServices.h"
#include "CubeTypes.h"
#include "algebra4.h"

#include "cube4_dump_Printer.h"
#include "cube4_dump_GnuPlotPrinter.h"
#include "cube4_dump_GnuPlot2Printer.h"
#include "cube4_dump_CSVPrinter.h"
#include "cube4_dump_CSV2Printer.h"
#include "cube4_dump_CSV3Printer.h"
#ifdef CUBE_DUMP_WITH_R
#include "cube4_dump_RPrinter.h"
#endif

#include "cube4_dump_CommandOption.h"

using namespace std;
using namespace cube;
using namespace services;


enum OutputFormat
{
    HUMAN_READABLE_FORMAT = 0,
    GNUPLOT_FORMAT        = 1,
    CSV_FORMAT            = 2,
    CSV2_FORMAT           = 3,
    CSV3_FORMAT           = 4,
    GNUPLOT2_FORMAT       = 5,
    R_FORMAT
};




/**
 * Main program.
 * - Check the calling parameters.
 * - For every filename it does
 * - Read the  .cube file. and saves it on alternate way either in 0-th,
   or 1th place in cube[] array.
 * - Calls cube_merge(...) with previous cube (merged) and the current one.
 * - Saves a created merged cube in either "-o output" or "merge.cube|.gz" file.
 * - end.
 */
int
main( int argc, char* argv[] )
{
    int             return_code = EXIT_SUCCESS;
    int             ch;
    vector <string> inputs;

    bool            selected_metric = false;
    bool            selected_cnode  = false;
    ThreadSelection selected_thread = THREADS_AS_ROW;

    bool                selected_header       = false;
    bool                selected_data         = false;
    bool                selected_flat_profile = false;
    bool                disable_clustering    = false;
    bool                disable_tasks_tree    = false;
    bool                show_ghost            = false;
    string              data;
    CalculationFlavour  mf     = cube::CUBE_CALCULATE_INCLUSIVE;
    CalculationFlavour  cf     = cube::CUBE_CALCULATE_EXCLUSIVE;
    bool                stored = false;
    std::string         metric;
    std::vector<string> metricv;
    std::vector<string> _metricv;
//     bool                  check_newmet = false;
    bool                  check_all = false;
    string                tempmet;
    unsigned              cnode_id;
    std::vector<unsigned> cnode_idv;
    unsigned              thread_id;
    std::vector<unsigned> thread_idv;
    std::string           cnode_cond_cubepl;
    bool                  cnode_cond  = false;
    bool                  show_coords = false;
    string                output      = "-";
    OutputFormat          format      = HUMAN_READABLE_FORMAT;
    std::vector<string>   selective_headers;
    std::vector<string>   selection_names = { "attributes", "mirrors", "metrics", "calltree", "systemtree", "topologies",  "files" };


    const string USAGE = "Usage: " + string( argv[ 0 ] ) + " [-m <metrics>|<new metrics>|all|<filename>] [-c <cnode ids>|all|leafs|roots|level>X|level=X|level<X|name=/regext/|<filename>] [-x incl|excl] [-z incl|excl|stored] [-t thread_id|aggr]  [-r] ] [-f name]  [-w] [-g]- [-s human|gnuplot|csv|csv2 ] [-h|-?] [-o output] <cube experiment>\n"
                         "\n"
                         "  -m <metrics>|<new metrics>|all|<filename>\n"
                         "                                        Select one or more of metrics (unique names) for data dump. \n"
                         "                                        \n"
                         "                                        By giving a CubePL expression one can define one or more new metrics by giving correspond formula\n"
                         "                                        If the expression prefixed with \"<name>:\", <name> is used as a unique name for the new metric\n"
                         "                                        <filename> - takes a CubePL expression from file <filename> and defines a derived metric with it \n"
                         "                                        all - all metrics will be printed.\n"
                         "                                        Combination of these three is possible.\n"
                         "  -c <cnode ids>|all|leafs|roots|level>X|level=X|level<X|name=/regexp/|<filename>\n"
                         "                                        Select one or more call paths to be printed out . \n"
                         "                                        <cnode ids> -  list or range of call path ids: 0,3,5-10,25\n"
                         "                                        all - all call paths are printed \n"
                         "                                        leafs - only call paths without children are printed \n"
                         "                                        roots - only root cnodes are printed \n"
                         "                                        level<X, level=X or level>X - only cnodes of the level more, equal or less than N are printed \n"
                         "                                        name=/regexp/ - only cnodes of with the name matching to the regular expression 'regexp' \n"
                         "                                        <filename> - takes a CubePL expression from file <filename> and evaluates it for every callpath. \n"
                         "                                                     If the result is non-zero - call path is included into the output.\n"
                         "  -x incl|excl                          Selects, if the data along the metric tree should be calculated as an inclusive or an exclusive value.\n"
                         "                                        (Default value: incl).\n"
                         "  -z incl|excl|stored                   Selects, if the data along the call tree should be calculated as an inclusive or an exclusive value.\n"
                         "                                        (Default value: excl).\n"
                         "  -t <thread id>|aggr                   Show data for one or more selected threads or aggregated over system tree. \n"
                         "  -r                                    Prints aggregated values for every region (flat profile), sorted by id.\n"
                         "  -f <name>                             Selects a stored data with the name <name> to display\n"
                         "  -d                                    Shows the coordinates for every topology as well.\n"
                         "  -y                                    Disables expansion of clusters, special handling of TASKS tree and shows bare stored meta structure.\n"
                         "  -w [<section>]                        Prints out the information about the structure of the cube (and CubePL memory, if --enable-debud used).\n"
                         "                                        By providing a section only the named section will be printed. Possible values are:\n"
                         "                                        `attributes`, `mirrors`, `metrics`, `calltree`, `systemtree`, `topologies`, or `files`. Multiple\n"
                         "                                        instances of the parameter are allowed.\n"
                         "  -g                                    Show ghost metrics (only in combination with -w).\n"
                         "  -o <filename>|-                       Uses a device or STDOUT for the output. If omit, STDOUT is used.\n"
                                                #ifdef CUBE_DUMP_WITH_R
                         "  -s human|gnuplot|gnuplot2|csv|csv2|csv3|R\n"
                         "                                        Uses either human readable form, GNUPLOT, CSV (three different layouts) format or binary R matrix for data export.\n"
                                                #else
                         "  -s human|gnuplot|gnuplot2|csv|csv2|csv3\n"
                         "                                        Uses either human readable form, GNUPLOT, CSV (three different layouts) format for data export.\n"
                                                #endif
                         "  -h|-?                                 Help; Output a brief help message.\n\n"
                         "Report bugs to <" PACKAGE_BUGREPORT ">\n";

    while ( ( ch = getopt( argc, argv, "s:o:f:z:x:m:c:t:rdw:ghy?" ) ) != -1 )
    {
        switch ( ch )
        {
            case 'h':
            case '?':
                cerr << USAGE << endl;
                exit( EXIT_SUCCESS );
                break;
            case 's':
                if ( std::strcmp( optarg, "gnuplot" ) == 0 )
                {
                    format = GNUPLOT_FORMAT;
                }
                else if ( std::strcmp( optarg, "gnuplot2" ) == 0 )
                {
                    format = GNUPLOT2_FORMAT;
                }
                else if ( std::strcmp( optarg, "csv2" ) == 0 )
                {
                    format = CSV2_FORMAT;
                }
                else if ( std::strcmp( optarg, "csv3" ) == 0 )
                {
                    format = CSV3_FORMAT;
                }
                else if ( std::strcmp( optarg, "csv" ) == 0 )
                {
                    format = CSV_FORMAT;
                }
                #ifdef CUBE_DUMP_WITH_R
                else if ( std::strcmp( optarg, "R" ) == 0 )
                {
                    format = R_FORMAT;
                }
                                #endif
                break;
            case 'o':
                output = optarg;
                break;
            case 'f':
                data          = optarg;
                selected_data = true;
                break;
            case 'm':
                selected_metric = true;
                tempmet         = optarg;
                // parseMetrics( optarg, metricv, new_metricv, check_newmet, check_all );
                break;
            case 'c':
                if ( std::strcmp( optarg, "all" ) == 0 )
                {
                    selected_cnode = false;
                }
                else if ( parseCallPaths_Threads( optarg, cnode_idv ) )
                {
                    selected_cnode = true;
                    ;
                }
                else if ( parseCubePlCnodeCondition( optarg, cnode_cond_cubepl ) )
                {
                    selected_cnode = true;
                    cnode_cond     = true;
                }
                else
                {
                    cerr << USAGE << "\nError: Wrong arguments.\n";
                    exit( EXIT_FAILURE );
                }
                break;
            case 't':
                if ( std::strcmp( optarg, "aggr" ) == 0 )
                {
                    selected_thread = AGGREGATED_THREADS;
                }
                else
                {
                    selected_thread = SELECTED_THREADS;
                    parseCallPaths_Threads( optarg, thread_idv );
                }
                break;
            case 'd':
                show_coords = true;
                break;
            case 'w':
                // initially assume no optional selection
                optind--;
                selected_header = true;
                // test if there is an optional selection and add its entry
                for ( size_t i = 0; i < selection_names.size(); i++ )
                {
                    if ( strcmp( optarg, selection_names[ i ].c_str() ) == 0 )
                    {
                        optind++;
                        selected_header = false;
                        selective_headers.push_back( optarg );
                        break;
                    }
                }
                break;
            case 'g':
                show_ghost = true;
                break;
            case 'y':
                disable_clustering = true;
                disable_tasks_tree = true;
                break;
            case 'r':
                selected_flat_profile = true;
                break;
            case 'x':
                if ( std::strcmp( optarg, "excl" ) == 0 )
                {
                    mf = cube::CUBE_CALCULATE_EXCLUSIVE;
                }
                else
                if ( std::strcmp( optarg, "incl" ) == 0 )
                {
                    mf = cube::CUBE_CALCULATE_INCLUSIVE;
                }
                else
                {
                    cerr << USAGE << "\nError: Wrong arguments.\n";
                    exit( EXIT_FAILURE );
                }
                break;
            case 'z':
                if ( std::strcmp( optarg, "stored" ) == 0 )
                {
                    stored = true;
                }
                else
                if ( std::strcmp( optarg, "excl" ) == 0 )
                {
                    cf = cube::CUBE_CALCULATE_EXCLUSIVE;
                }
                else
                if ( std::strcmp( optarg, "incl" ) == 0 )
                {
                    cf = cube::CUBE_CALCULATE_INCLUSIVE;
                }
                else
                {
                    cerr << USAGE << "\nError: Wrong arguments.\n";
                    exit( EXIT_FAILURE );
                }
                break;
            default:
                cerr << USAGE << "\nError: Wrong arguments.\n";
                exit( EXIT_FAILURE );
        }
    }

    if ( argc - optind < 1 )
    {
        cerr << USAGE << "\nError: Wrong arguments.\n";
        exit( EXIT_FAILURE );
    }

    for ( int i = optind; i < argc; i++ )
    {
        inputs.push_back( argv[ i ] );
    }

    fstream    _fout;
    ostream&   fout = cout;
    streambuf* buf  = NULL;
    if ( output != "-" )
    {
        _fout.open(  output.c_str(),  fstream::in | fstream::out | fstream::trunc );
        if ( !_fout.is_open() )
        {
            cerr << "Cannot open " << output << " to use as an output device. " << endl;
            exit( EXIT_FAILURE );
        }
        buf = fout.rdbuf();
        fout.rdbuf( _fout.rdbuf() );
    }

    int   num  = inputs.size();
    Cube* cube = nullptr;

    Printer* printer = NULL;


    // _______________________________________________________________________
    // num is the number of files

    for ( int i = 0; i < num; i++ )
    {
        cube = new Cube();
        try
        {
            cube->openCubeReport( inputs[ i ], disable_tasks_tree, disable_clustering );

            if ( cnode_cond )
            {
                cnodeSelection( cube, cnode_cond_cubepl,  cnode_idv );
            }


            if ( selected_metric )
            {
                parseMetrics( tempmet, _metricv, cube );
                parseNewMetric( _metricv, metricv, check_all, cube );
            }

            if ( format == GNUPLOT_FORMAT )
            {
                printer = new GnuPlotPrinter( fout, cube );
            }
            else if ( format == GNUPLOT2_FORMAT )
            {
                printer = new GnuPlot2Printer( fout, cube );
            }
            else
            if ( format == HUMAN_READABLE_FORMAT )
            {
                printer = new Printer( fout, cube );
            }
            else
            if ( format == CSV_FORMAT )
            {
                printer = new CSVPrinter( fout, cube );
            }
            else
            if ( format == CSV2_FORMAT )
            {
                printer = new CSV2Printer( fout, cube );
            }
            else
            if ( format == CSV3_FORMAT )
            {
                printer = new CSV3Printer( fout, cube );
            }
#ifdef CUBE_DUMP_WITH_R
            else
            if ( format == R_FORMAT )
            {
                // we're not exporting into file
                if ( output == "-" )
                {
                    cerr << "Output to R matrix requires specifying file to output"  << endl;
                    exit( EXIT_FAILURE );
                }
                printer = new RPrinter( output, cube );
            }
#endif
            std::vector<Metric*>   metrics = cube->get_metv();
            std::vector<Cnode*>    cnodes  = cube->get_cnodev();
            std::vector<Location*> threads = cube->get_locationv();
            std::vector<Region*>   regions = cube->get_regv();
            if ( selected_header )
            {
                printer->print_header(  data, show_ghost, selected_data, show_coords );
            }
            else if ( !selected_header && !selective_headers.empty() )
            {
                for ( std::vector<std::string>::iterator it = selective_headers.begin(); it != selective_headers.end(); ++it )
                {
                    if ( *it == selection_names[ 0 ] )
                    {
                        printer->print_header_cube_attributes();
                    }
                    else if ( *it == selection_names[ 1 ] )
                    {
                        printer->print_header_cube_mirrors();
                    }
                    else if ( *it == selection_names[ 2 ] )
                    {
                        printer->print_header_metric_dimension( show_ghost );
                    }
                    else if ( *it == selection_names[ 3 ] )
                    {
                        printer->print_header_calltree_dimension();
                    }
                    else if ( *it == selection_names[ 4 ] )
                    {
                        printer->print_header_system_dimension();
                    }
                    else if ( *it == selection_names[ 5 ] )
                    {
                        printer->print_header_topologies( show_coords );
                    }
                    else if ( *it == selection_names[ 6 ] )
                    {
                        printer->print_header_files( data, selected_data );
                    }
                    else
                    {
                        cerr << "Unknown section name `" << *it << "` for `-W <section_name>`!" << endl;
                        exit( EXIT_FAILURE );
                    }
                }
            }

            if ( selected_metric )
            {
                std::vector < Metric* > _metrics;
                if ( check_all ) // show all metrics
                {
                    _metrics = metrics;
                }
                else
                {
                    for ( size_t mv_iter = 0; mv_iter != metricv.size(); ++mv_iter )
                    {
                        metric = metricv[ mv_iter ];
                        _metrics.push_back( cube->get_met( metric ) );
                    }
                }

                std::vector < Cnode* > _cnodes;
                if ( selected_cnode )
                {
                    for ( size_t cv_iter = 0; cv_iter != cnode_idv.size(); ++cv_iter )
                    {
                        cnode_id = cnode_idv[ cv_iter ];
                        for ( vector<Cnode*>::iterator c_iter = cnodes.begin(); c_iter != cnodes.end(); ++c_iter )
                        {
                            if ( ( *c_iter )->get_id() == cnode_id )
                            {
                                _cnodes.push_back( *c_iter );
                            }
                        }
                    }
                }
                else
                {
                    _cnodes = cnodes;
                }
                std::vector < Thread* > _threads;
                if ( selected_thread == SELECTED_THREADS )
                {
                    for ( size_t tv_iter = 0; tv_iter != thread_idv.size(); ++tv_iter )
                    {
                        thread_id = thread_idv[ tv_iter ];
                        for ( vector<Thread*>::iterator t_iter = threads.begin(); t_iter != threads.end(); ++t_iter )
                        {
                            if ( ( *t_iter )->get_id() == thread_id )
                            {
                                _threads.push_back( *t_iter );
                            }
                        }
                    }
                }
                else
                {
                    _threads = threads;
                }

                if ( !_metrics.empty() )
                {
                    if ( selected_flat_profile )
                    {
                        printer->dump_flat_profile( _metrics, regions, _threads, mf, selected_thread );
                    }
                    else
                    {
                        printer->dump_data( _metrics, _cnodes, _threads, mf, cf, stored, selected_thread );
                    }
                }
            }
            delete cube;
        }
        catch ( const RuntimeError& error )
        {
            delete cube;
            cube = nullptr;
            cerr << error.what() << endl;
            return_code = EXIT_FAILURE;
        }
        delete printer;
    }
    if ( _fout.is_open() )
    {
        fout.rdbuf( buf );
        _fout.close();
    }
    return return_code;
}
