/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"

#ifdef CUBE_COMPRESSED
#  include "zfstream.h"
#endif

#include <cstdio>
#include <cstdlib>
#include <unistd.h>
#include <fstream>
#include <chrono>

#include "cube_topoassist.h"
#include "algebra4-internals.h"
#include "Question.h"

#include "CubeTypes.h"
#include "CubeSysres.h"
#include "CubeSystemTreeNode.h"
#include "CubeLocationGroup.h"
#include "CubeLocation.h"

using namespace std;
using namespace cube;


/*-------------------------------------------------------------------------*/
/**
 * @file    cube_topoassist.cpp
 * @brief   Assistant to manually add virtual topologies in cube files
 *
 * With this assistant, one is able to add an arbitrary virtual topology to a cube file.
 * So far, this topology might have up to three dimensions (it's a CUBE limitation, actually)
 * and the total number of possible coordinates must NOT be smaller than the number of processes/threads
 * of the specifed CUBE file.
 *
 * Coordinates are asked for in rank order, and inside every rank, in thread order.
 */
/*-------------------------------------------------------------------------*/




void
write_cube( Cube* outCube, string output_name )
{
    cout << endl << "Write " << output_name << "......";
    outCube->writeCubeReport( output_name );
    cout << "done." << endl;
}


void
create_topology( Cube* inCube )
{
    /**
     * Reads the process vector, sorts it into a map of rank->process and ask for coordinates for the new topology.
     * @param cubeObj The CUBE object to add a topology.
     */
    std::string              tempStrQuestion;
    std::string              tempStrAnswer;
    size_t                   num_elements = 1, dimtemp = 0;
    unsigned int             num_dims     = 0;
    std::vector<long>        dims;
    std::vector<bool>        periods;
    std::vector<std::string> nameDims;

    cout << "So far, only cartesian topologies are accepted." << endl;

    string name;
    question( string( "Name for new topology?" ), &name );

    // In a future expansion, I should support Graph topologies too

    question( "Number of Dimensions?", &num_dims );

    if ( num_dims > 3 )
    {
        cerr
            << "Currently, Cube will not show more than 3 dimensions."
            << endl;
    }

    bool areDimsNamed;
    bool isDimPeriodic;

    question( "Do you want to name the dimensions (axis) of this topology? (Y/N)", &areDimsNamed );

    for ( size_t i = 0; i < num_dims; i++ )
    {
        if ( areDimsNamed )
        {
            question( i, "Name for dimension %i", &tempStrAnswer );
            nameDims.push_back( tempStrAnswer );
        }

        do
        {
            question( i, "Number of elements for dimension %i", &dimtemp );
        }
        while ( dimtemp == 0 );

        num_elements = num_elements * dimtemp;
        dims.push_back( dimtemp );

        question( i, "Is dimension %i periodic?", &isDimPeriodic );
        periods.push_back( isDimPeriodic );
    } // End dimension definition stuff


    Cartesian* topo = inCube->def_cart( dims.size(), dims, periods );

    topo->set_name( name );
    topo->set_namedims( nameDims );

    // Define topology
    const vector<Process*>&  procv = inCube->get_procv();
    const vector<Location*>& thrdv = inCube->get_locationv();

    map<long, long> procv_ranks;
    for ( size_t i = 0; i < procv.size(); i++ )
    {
        procv_ranks[ procv[ i ]->get_rank() ] = i;
    }

    // Number of processes/threads should be same or smaller than the number of possible coordinates for them.
    if ( num_elements < thrdv.size() )
    {
        cout
            << "The number of possible coordinates with this topology is smaller then the number of threads ("
            << thrdv.size() << "). Cannot continue" << endl;
        exit( EXIT_FAILURE );
    }
    if ( num_elements > thrdv.size() )
    {
        cout << "Alert: The number of possible coordinates ("
             << num_elements
             << ") is bigger than the number of threads on the specified cube file ("
             << thrdv.size() << "). Some positions will stay empty." << endl;
    }
    std::cout << "Topology on THREAD level." << std::endl;
    Location* thrd;
    Process*  proc;

    // It will iterate over the process vector and get all the children

    for ( size_t j = 0; j < procv.size(); ++j )
    {
        proc = procv[ procv_ranks[ j ] ];
        for ( size_t k = 0; k < proc->num_children(); k++ )
        {
            thrd = proc->get_child( k );
            string q = thrd->get_name() + "'s (rank " + intToStr( proc->get_rank() ) +  ") coordinates in %i dimensions, separated by spaces";
            question( num_dims, q, &tempStrQuestion );
            stringstream ss( tempStrQuestion );
            long         l = 0;
            vector<long> values;
            for ( size_t i = 0; i < num_dims; i++ )
            {
                ss >> l;
                if ( l + 1 > dims[ i ] )
                {
                    cerr << "Invalid coordinate. Dimension " << i
                         << " is of size " << dims[ i ] << ", beginning in zero. You typed " << l
                         << ". Exiting." << endl;
                    exit( EXIT_FAILURE );
                }
                values.push_back( l );
            }
            inCube->def_coords( topo, thrd, values );
        }
    }
}

void
generate_topology( Cube* inCube )
{
    string tempStrQuestion, tempStrAnswer;

    cout << "Generating global cartesian topologies." << endl;

    bool   useDefaults;
    bool   useSingularDims = false;
    bool   createMultiple  = false;
    string name;
    question( "Do you want to use default parameters (Y/N)", &useDefaults );

    if ( useDefaults )
    {
        // Determine if we have multiple root nodes (machines) or first children, that are modules.
        if ( !inCube->get_root_stnv().empty() )
        {
            set < string > distinct_modules;
            for ( size_t i = 0; i < inCube->get_root_stnv().size(); i++ )
            {
                for ( size_t j = 0; j < inCube->get_root_stnv()[ i ]->get_children().size(); j++ )
                {
                    if ( inCube->get_root_stnv()[ i ]->get_child( j )->get_class() == "module" )
                    {
                        distinct_modules.insert( inCube->get_root_stnv()[ i ]->get_child( j )->get_name() );
                    }
                }
            }
            if ( distinct_modules.size() > 1 || inCube->get_root_stnv().size() > 1 )
            {
                createMultiple = true;
            }
        }
    }
    else
    {
        question( "Do you want to show dimensions of size 1 (Y/N)", &useSingularDims );
        question( "Do you want separate topologies on either machine or module level (Y/N)", &createMultiple );
    }
    // Generating topologies
    // Assumption: all applications have at least one CPU thread
    chrono::high_resolution_clock::time_point t1 = chrono::high_resolution_clock::now();
    gen_top( *inCube, CUBE_LOCATION_TYPE_CPU_THREAD, useSingularDims, createMultiple, false  );
    chrono::high_resolution_clock::time_point t2 = chrono::high_resolution_clock::now();
    cout << "\nCPU topology generated in "
         << chrono::duration_cast<chrono::duration<double> >( t2 - t1 ).count() << " seconds." << endl;

    // Only generate GPU topologies if there is at least one GPU stream
    const vector<Location*>& locationv = inCube->get_locationv();
    for ( size_t i = 0; i < locationv.size(); i++ )
    {
        if ( locationv[ i ]->get_type() == CUBE_LOCATION_TYPE_GPU )
        {
            bool force_old_style_streams = false;
            if ( locationv[ i ]->get_parent()->get_type() == CUBE_LOCATION_GROUP_TYPE_PROCESS )
            {
                // Pre Score-P 8.0 streams as children of the process
                force_old_style_streams =  true;
            }
            t1 = chrono::high_resolution_clock::now();
            gen_top( *inCube, CUBE_LOCATION_TYPE_GPU, useSingularDims, createMultiple, force_old_style_streams );
            t2 = chrono::high_resolution_clock::now();
            cout << "\nGPU topology generated in "
                 << chrono::duration_cast<chrono::duration<double> >( t2 - t1 ).count() << " seconds." << endl;
            break;
        }
    }
}

void
show_topologies( vector<Cartesian*> topologies )
{
    /**
     * Displays information of topologies: name, # of dimensions, their names and total threads.
     * Used to rename a topology or a topology's dimensions at rename_topology and rename_dimensions
     * @param topologies The topology vector from cube file with get_cartv().
     */

    vector<string> names, namedims;
    size_t         num_threads = 1;

    cout << "This CUBE has " << topologies.size() << " topologie(s)." << endl;

    // Iterates over every topology and displays their names, number of dimensions, number of coordinates on each etc.
    for ( size_t i = 0; i < topologies.size(); i++ )
    {
        names.push_back( topologies[ i ]->get_name() );
        namedims = topologies[ i ]->get_namedims();
        if ( names[ i ].empty() )
        {
            cout << i << ". " << "Unnamed topology, ";
        }
        else
        {
            cout << i << ". " <<  names[ i ] << ", ";
        }
        cout << ( topologies[ i ] )->get_ndims() << " dimensions: ";
        for ( int j = 0; j < topologies[ i ]->get_ndims(); j++ )
        {
            if ( !namedims.empty() )
            {
                cout << namedims[ j ] << ": ";
            }
            cout << topologies[ i ]->get_dimv()[ j ] << ( ( j + 1 < topologies[ i ]->get_ndims() ) ? " x " : "," );
            num_threads = num_threads * topologies[ i ]->get_dimv()[ j ];
        }
        std::cout << " total = " << num_threads << " threads. ";
        if ( namedims.empty() )
        {
            cout << "Dimensions are not named.";
        }
        cout << endl;
        num_threads = 1; // reset counter.
    }
}

void
rename_topology( Cube* inCube )
{
    /**
     * Traverses cube's topologies, shows the current names of them and offer to name/rename them
     * @param cubeObj The CUBE object to be changed.
     */
    vector<Cartesian*> topologies = inCube->get_cartv();
    show_topologies( topologies );
    unsigned int index_topo;
    string       new_name;
    question( "\nTopology to [re]name?", &index_topo );
    question( "New name: ", &new_name );
    topologies[ index_topo ]->set_name( new_name );
    cout << "Topology successfully [re]named." << endl << endl;
}

void
remove_topology( Cube* inCube )
{
    /**
     * Traverses cube's topologies, shows the current names of them and offer to remove them
     * @param cubeObj The CUBE object to be changed.
     */
    vector<Cartesian*> topologies = inCube->get_cartv();
    show_topologies( topologies );
    unsigned int index_topo;
    question( "\nTopology to remove?", &index_topo );
    inCube->drop_cart( index_topo );
    cout << "Topology successfully removed." << endl << endl;
}

void
rename_dimensions( Cube* inCube )
{
    /**
     * Traverses cube's topologies, shows the current names of them and offer to name/rename dimensions of one of them
     * @param cubeObj The CUBE object to be changed.
     */
    // GET_NAMEDIMS
    vector<Cartesian*> topologies = inCube->get_cartv();
    show_topologies( topologies );
    unsigned int             index_topo;
    std::vector<std::string> new_namedims;
    std::string              new_name;
    question( "\nIn which topology do you want to [re]name dimensions?", &index_topo );
    for ( int dim = 0; dim < topologies[ index_topo ]->get_ndims(); dim++ )
    {
        question( dim, "New name for dimension %i : ", &new_name );
        new_namedims.push_back( new_name );
        new_name = "";
    }
    if ( !topologies[ index_topo ]->set_namedims( new_namedims ) )
    {
        cout << "Could not rename dimensions.";
        exit( EXIT_FAILURE );
    }
    else
    {
        cout << "Dimensions successfully [re]named" << endl << endl;
    }
}

int
main( int argc, char** argv )
{
    int    rename_topology_flag   = 0;
    int    rename_dimensions_flag = 0;
    int    create_topology_flag   = 0;
    int    generate_topology_flag = 0;
    int    remove_topology_flag   = 0;
    string cube_name;
    string cube_output_name = "topologies.cubex";
    int    c; // option to name existing topology or dimensions
    opterr = 1;

    string usage( "Usage: cube_topoassist [opts] <cube experiment>\n \
Command-line switches:\n \
-c : create a new topology in the existing CUBE file.\n \
-g : generate global topologies in the existing CUBE file.\n \
-n : [re]name an existing topology\n \
-r : remove an existing topology\n \
-d : name dimensions of a topology \n \
-o : output CUBE file (default: topologies.cubex) \n \
\nThe output is a topo.cube[.gz] file in the current directory.\n\n \
Report bugs to <" PACKAGE_BUGREPORT ">\n" );

    // Use last parameter as input file if possible
    if ( !access( argv[ argc - 1 ], F_OK ) )
    {
        cube_name = argv[ argc - 1  ];
    }

    if ( cube_name.empty() )
    {
        cerr << "ERROR: Missing file name or wrong argument order. " << endl << usage;
        exit( EXIT_FAILURE );
    }

    // Parses command-line options
    while ( ( c = getopt( argc, argv, "ndcgo:r" ) ) != -1 )
    {
        switch ( c )
        {
            case 'n':
                rename_topology_flag++;
                break;
            case 'd':
                rename_dimensions_flag++;
                break;
            case 'c':
                create_topology_flag++;
                break;
            case 'g':
                generate_topology_flag++;
                break;
            case 'o':
                cube_output_name = optarg;
                break;
            case 'r':
                remove_topology_flag++;
                break;
            case '?':
                cerr << usage;
                exit( EXIT_SUCCESS );
            default:
                exit( EXIT_FAILURE );
        }
    }
    // No options
    if ( rename_topology_flag == 0 &&   \
         rename_dimensions_flag == 0 && \
         generate_topology_flag == 0 && \
         create_topology_flag == 0 &&   \
         remove_topology_flag == 0 )
    {
        cout <<  usage << endl;
        exit( EXIT_FAILURE );
    }

    // Read CUBE file
    Cube* inCube  = new Cube();
    Cube* outCube = NULL;

    cout << "Reading " << cube_name << ".\nPlease wait... \n" << flush;
    try
    {
        inCube->openCubeReport( cube_name );
        outCube = new Cube( *inCube, CUBE_DEEP_COPY );
    }
    catch ( ... )
    {
        delete inCube;
        exit( EXIT_FAILURE );
    }

    cout << "Done." << endl;


    if ( rename_topology_flag )
    {
        rename_topology( outCube );
    }
    if ( rename_dimensions_flag )
    {
        rename_dimensions( outCube );
    }
    if ( create_topology_flag )
    {
        create_topology( outCube );
    }

    if ( generate_topology_flag )
    {
        generate_topology( outCube );
    }
    if ( remove_topology_flag )
    {
        remove_topology( outCube );
    }


    write_cube( outCube, cube_output_name );
    delete inCube;
    delete outCube;
    exit( EXIT_SUCCESS );
}
