/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2009-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2009-2011                                                **
**  RWTH Aachen University, JARA-HPC                                       **
**                                                                         **
**  Copyright (c) 2009-2016                                                **
**  TU Dresden, Zentrum fuer Informationsdienste und Hochleistungsrechnen  **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  Technische Universitaet Muenchen, Germany                              **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  University of Oregon, Eugene, USA                                      **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany      **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/**
 * @file
 *
 * Declares classes for the representation and analysis of library dependencies
 */

#ifndef CUBELIB_CONFIG_DATA_HPP
#define CUBELIB_CONFIG_DATA_HPP

#include <map>
#include <deque>
#include <string>
#include <library_data.hpp>

/**
 * A class to store and analyze the dependencies from libraries. To generate
 * the dependecies, Makefile variables are parsed and header files are generated,
 * see build-config/common/Makefile-rpaths.inc.am. Note that we take care of
 * omitting known linker and loader search paths in GetLDFlags(), GetLTLDFlags()
 * and GetRpathFlags().
 */
class cubelib_config_data
{
    // ------------------------------------- Public types
public:

    /**
     * Constructs the library dependecies
     */
    cubelib_config_data();

    /**
     * Destructor
     */
    virtual
    ~cubelib_config_data();

    /**
     * Returns the string containing the @a input_libs and its dependencies.
     * @param input_libs  A list of libraries, that should be linked.
     */
    std::string
    GetLibraries( const std::deque<std::string>& inputLibs );

    /**
     * Returns the duplicate-free string containing the -L library path flags
     * for the @a input_libs and its dependencies. Known linker search paths
     * are omitted.
     * @param libs     A list of library names.
     */
    std::string
    GetLDFlags( const std::deque<std::string>& libs );

    /**
     * Returns the duplicate-free string containing the -L and -R flags for
     * libtool consumption for the @a input_libs and their dependencies.
     * Known linker and loader search paths are omitted.
     * If LD_RUN_PATH is set at execution time of cubelib-config, its
     * directories are appended as -R<dir>.
     * @param libs     A list of library names.
     */
    std::string
    GetLTLDFlags( const std::deque<std::string>& libs );

    /**
     * Returns the duplicate-free string containing the rpath flags
     * for the @a libs and its dependencies (usually -Wl,-rpath -Wl,<dir>).
     * Known loader search paths are omitted.
     * If LD_RUN_PATH is set at execution time of cubelib-config, its
     * directories are appended.
     * @param libs      A list of library names.
     */
    std::string
    GetRpathFlags( const std::deque<std::string>& libs );

    /**
     * Construct rpath construction flags
     */
    void
    prepare_rpath_flags();

    /** Make string with compiler or linker flags compatible to CUDA
     *  compiler requirements.
     *
     *  @param str              String to be processed.
     *
     *  @return Returns string with compiler or linker flags that can be
     *          passes to CUDA compiler.
     */
    std::string
    prepare_string( const std::string& str );

    // ------------------------------------- Public members
    std::string m_rpath_head;
    std::string m_rpath_delimiter;
    std::string m_rpath_tail;

    std::string m_cc;
    std::string m_cxx;
    std::string m_cppflags;
    std::string m_incldir;
    bool        m_use_rpath_flag;
    std::string m_rpath_flag_cc;
    std::string m_wl_flag;
    std::string m_aix_libpath;
    std::string m_sys_lib_search_path;
    std::string m_sys_lib_dlsearch_path;

    std::map< std::string, LibraryData> m_library_objects;

private:
    /**
     * Remove known linker search paths from @p input.
     */
    std::deque<std::string>
    remove_linker_search_paths( const std::deque<std::string>& input ) const;

    /**
     * Remove known loader search paths from @p input.
     */
    std::deque<std::string>
    remove_loader_search_paths( const std::deque<std::string>& input ) const;

    /**
     * Helper function to make sure @a libs exist in m_libdir_objects,
     * die otherwise.
     * @param libs a container of library names
     */
    void
    assert_lib_objects_exist( const std::deque<std::string>& libs ) const;

    /**
     * Helper function to collect a container of all libdirs needed for
     * linking all @a libs. Intended to be used from GetLDFlags(),
     * GetLTLDFlags(), and GetRpathFlags().
     */
    std::deque<std::string>
    get_libdirs( const std::deque<std::string>& libs );

    /**
     * Return the contents of the environment variable LD_RUN_PATH as
     * container of strings, ommitting system linker search paths,
     * non-absolute paths and those containing whitespace.
     */
    std::deque<std::string>
    get_ld_run_path();
};



cubelib_config_data*
cubelib_config_get_frontend_data();


#endif /* CUBELIB_CONFIG_DATA_HPP */
