/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_POP_COMMUNICATION_EFFICENCY_TEST_H
#define CUBELIB_POP_COMMUNICATION_EFFICENCY_TEST_H

#include <string>
#include "PerformanceTest.h"
#include "POPSerialisationTest.h"
#include "POPTransferTest.h"

namespace mpianalysis
{
class POPParallelEfficiencyTest;

class POPCommunicationEfficiencyTest : public popcalculation::PerformanceTest
{
    friend class POPParallelEfficiencyTest;

private:
    cube::Metric*         max_comp_time;
    cube::Metric*         max_runtime;
    POPSerialisationTest* pop_ser;
    POPTransferTest*      pop_transeff;
    bool                  scout_cubex;
    bool                  original_scout_cubex;

    cube::list_of_metrics lmax_comp_time;

    double
    calculateForScout( const cube::list_of_cnodes& cnodes  ) const;

protected:

    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "Communication Efficiency (CommE) is the maximum across all processes of the ratio between useful computation time and total runtime.\n\n"
            "Mathematical definition:\n"
            "\tCommE = maximum across processes (computation time / total runtime)\n\n"
            "CommE identifies inefficient code by revealing situations where a program spends an excessive amount of time\n"
            "communicating instead of performing useful computations.\n\n"
            "It is derived from two core sub-metrics, each addressing a distinct cause of communication overhead:\n"
            "\t1. Serialization Efficiency: Measures time processes remain idle at synchronization points (e.g., barriers)\n"
            "\t   due to stragglers or serialization delays.\n"
            "\t2. Transfer Efficiency: Evaluates the network bandwidth utilization during data transfers.\n\n"
            "Combined formula:\n"
            "\tCommE = Serialization Efficiency x Transfer Efficiency\n\n"
            "These sub-metrics are obtained through Scalasca trace analysis, which automatically detects\n"
            "serialization bottlenecks and inefficient communication patterns.";

        std::string not_active_text =
            "The Communication Efficiency metric is a fundamental POP metric and is included\n"
            "in all Score-P/Scalasca measurements by default.\n\n"
            "Limitation: If the Cube report originates from a tool other than Score-P/Scalasca, it may lack the\n"
            "'Time' metric, making Communication Efficiency analysis impossible.";

        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;


public:
    POPCommunicationEfficiencyTest(
        cube::CubeProxy*,
        POPSerialisationTest* pop_ser,
        POPTransferTest*      pop_transef );

    virtual
    ~POPCommunicationEfficiencyTest()
    {
    };

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false   );


    virtual
    double
    analyze( const cube::list_of_cnodes& cnodes,
             cube::LocationGroup*        _lg = nullptr ) const;

// ------ overview tests ---------

    bool
    isActive() const;

    bool
    isIssue() const;

    virtual
    std::list<PerformanceTest*>
    getPrereqs()
    {
        std::list<PerformanceTest*> prereqs;
        if ( pop_ser != nullptr && pop_transeff != nullptr )
        {
            prereqs.push_back( pop_transeff );
            prereqs.push_back( pop_ser );
        }
        return prereqs;
    }
};
};
#endif // ADVISER_RATING_WIDGET_H
