/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_POP_HYBRID_IMBALANCE_TEST_ADD_H
#define CUBELIB_POP_HYBRID_IMBALANCE_TEST_ADD_H

#include <string>
#include "PerformanceTest.h"


namespace hybaddanalysis
{
class POPHybridProcessEfficiencyTestAdd;
class POPHybridImbalanceTestAdd : public popcalculation::PerformanceTest
{
    friend class POPHybridProcessEfficiencyTestAdd;
private:
    cube::Metric* pop_avg_omp;
    cube::Metric* pop_avg_ser_comp;
    cube::Metric* max_omp_serial_comp_time;
    cube::Metric* max_runtime;

    cube::list_of_metrics lavg_omp_metrics;
    cube::list_of_metrics lavg_ser_metrics;
    cube::list_of_metrics lmax_runtime_metrics;

protected:


    virtual
    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "Computation Load Balance measures the degree to which computational work is evenly distributed\n"
            "across processes. It can be calculated using the following formula:\n"
            "\n"
            "Computation Load Balance = \n"
            "    [max(runtime) - max(time in OpenMP regions + serial computation time) + \n"
            "     avg(time in OpenMP regions) + avg(serial computation time)] / max(runtime)\n"
            "\n"
            "Where:\n"
            "- Average times are computed as weighted arithmetic means, accounting for:\n"
            "  - Different numbers of threads per process\n"
            "  - Process-specific execution characteristics\n"
            "\n"
            "When all processes have identical thread counts, the averages simplify to ordinary\n"
            "arithmetic means calculated across all processes.\n"
            "\n"
            "Key observations:\n"
            "- Values near 1 indicate excellent load balance\n"
            "- Values near 0 reveal significant load imbalance\n"
            "- The term max(time in OpenMP + serial) represents the worst-case computational workload";

        std::string not_active_text =
            "Computation Load Balance analysis requires precise timing metrics that are only available\n"
            "when Cube reports are generated by Score-P/Scalasca with complete instrumentation.\n"
            "\n"
            "The metric depends on:\n"
            "- Process-level timing information (runtime, OpenMP region times)\n"
            "- Thread-level timing data (especially in hybrid environments)\n"
            "- Serial computation timing for each process\n"
            "\n"
            "When reports come from alternative tools, critical timing data may be:\n"
            "- Missing entirely (e.g., 'time' metric)\n"
            "- Aggregated at incorrect granularity (e.g., process-level instead of thread-level)\n"
            "- Incomplete (e.g., only wall-clock time without parallel region breakdowns)\n"
            "\n"
            "Without these essential components, accurate POP analysis cannot proceed, including:\n"
            "- Load balance assessment\n"
            "- Parallel efficiency calculations\n"
            "- Work distribution optimization recommendations";

        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;

public:
    POPHybridImbalanceTestAdd( cube::CubeProxy* );

    virtual
    ~POPHybridImbalanceTestAdd()
    {
    };

    virtual
    double
    analyze( const cube::list_of_cnodes& cnodes,
             cube::LocationGroup*        _lg = nullptr ) const;




// ------ overview tests ---------

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false   );

    bool
    isActive() const;

    bool
    isIssue() const;
};
};
#endif // POPHybrid_IMBALANCE_TEST_H
