/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>

#include "POPHybridImbalanceTestAdd.h"
#include "POPCalculation.h"


using namespace hybaddanalysis;

POPHybridImbalanceTestAdd::POPHybridImbalanceTestAdd( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName(  " + + Computation Load Balance" );
    setWeight( 1 );   // need to be adjusted
    max_omp_serial_comp_time = cube->getMetric( "max_omp_serial_comp_time" );
    if ( max_omp_serial_comp_time == nullptr )
    {
        adjustForTest( cube );
    }
    max_omp_serial_comp_time = cube->getMetric( "max_omp_serial_comp_time" );
    if ( max_omp_serial_comp_time == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }

    pop_avg_omp      = cube->getMetric( "max_omp_time" );
    pop_avg_ser_comp = cube->getMetric( "ser_comp_time" );
    max_runtime      = cube->getMetric( "max_runtime" );


    cube::list_of_sysresources lsysres = getRootsOfSystemTree();
    cube::metric_pair          metric;
    metric.first  = max_omp_serial_comp_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );

    metric.first  = pop_avg_omp;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lavg_omp_metrics.push_back( metric );

    metric.first  = pop_avg_ser_comp;
    metric.second = cube::CUBE_CALCULATE_EXCLUSIVE;
    lavg_ser_metrics.push_back( metric );


    metric.first  = max_runtime;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_runtime_metrics.push_back( metric );
}


double
POPHybridImbalanceTestAdd::analyze( const cube::list_of_cnodes& cnodes,
                                    cube::LocationGroup*          ) const
{
    if ( max_omp_serial_comp_time == nullptr )
    {
        return 0.;
    }
    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );

    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( lavg_omp_metrics,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );

    cube::value_container inclusive_values3;
    cube::value_container exclusive_values3;
    cube->getSystemTreeValues( lavg_ser_metrics,
                               cnodes,
                               inclusive_values3,
                               exclusive_values3 );

    cube::value_container inclusive_values4;
    cube::value_container exclusive_values4;
    cube->getSystemTreeValues( lmax_runtime_metrics,
                               cnodes,
                               inclusive_values4,
                               exclusive_values4 );

    double avg_max_omp_ser_value = inclusive_values1[ 0 ]->getDouble();
    double max_runtime_value     = inclusive_values4[ 0 ]->getDouble();

    const std::vector<cube::LocationGroup*>& _lgs              = cube->getLocationGroups();
    double                                   _avg_ser_comp_sum = 0.;
    double                                   _avg_omp_sum      = 0.;
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        double _v1 =  inclusive_values3[ ( *iter )->get_sys_id() ]->getDouble();
        _avg_ser_comp_sum +=   _v1 * popcalculation::POPCalculation::get_num_thread_children( ( *iter ) );
        double _v2 =  inclusive_values2[ ( *iter )->get_sys_id() ]->getDouble();
        _avg_omp_sum +=   _v2 * popcalculation::POPCalculation::get_num_thread_children( ( *iter ) );
    }


    size_t cpu_locs_num     = get_number_of_cpu_locations();
    double avg_ser_comp_sum = _avg_ser_comp_sum / cpu_locs_num;
    double avg_omp_sum      = _avg_omp_sum / cpu_locs_num;

    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values3.begin(), inclusive_values3.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values3.begin(), exclusive_values3.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values4.begin(), inclusive_values4.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values4.begin(), exclusive_values4.end(),  [ ]( cube::Value* element ){
        delete element;
    } );

    return ( max_runtime_value + avg_omp_sum + avg_ser_comp_sum  - avg_max_omp_ser_value ) / ( max_runtime_value  );
}



void
POPHybridImbalanceTestAdd::applyCnode( const cube::list_of_cnodes& cnodes,
                                       const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    if ( max_omp_serial_comp_time == nullptr )
    {
        return;
    }
    setValue(  analyze( cnodes ) );
}


const std::string&
POPHybridImbalanceTestAdd::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
POPHybridImbalanceTestAdd::isActive() const
{
    return max_omp_serial_comp_time != nullptr;
};

bool
POPHybridImbalanceTestAdd::isIssue() const
{
    return false;
};

void
POPHybridImbalanceTestAdd::adjustForTest( cube::CubeProxy* cube ) const
{
    add_comp_time( cube );
    add_max_omp_and_ser_execution( cube );
}
