/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_POP_HYBRID_COMMUNICATION_EFFICENCY_TEST_ADD_H
#define CUBELIB_POP_HYBRID_COMMUNICATION_EFFICENCY_TEST_ADD_H

#include <string>
#include "PerformanceTest.h"
#include "POPHybridSerialisationTestAdd.h"
#include "POPHybridTransferTestAdd.h"

namespace hybaddanalysis
{
class POPHybridProcessEfficiencyTestAdd;
class POPHybridCommunicationEfficiencyTestAdd : public popcalculation::PerformanceTest
{
    friend class POPHybridProcessEfficiencyTestAdd;
private:
    cube::Metric*                  pop_commeff;
    cube::Metric*                  max_omp_serial_comp_time;
    cube::Metric*                  max_runtime;
    POPHybridSerialisationTestAdd* pop_ser;
    POPHybridTransferTestAdd*      pop_transeff;
    bool                           scout_cubex;
    bool                           original_scout_cubex;


    cube::list_of_metrics lmax_omp_ser_comp_time;

    double
    calculateForScout( const cube::list_of_cnodes& cnodes  ) const;

    void
    add_comm_eff( cube::CubeProxy* ) const;



protected:

    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "MPI Communication Efficiency (CommE) can be calculated using the following formula:\n"
            "\tCommE = (max(time in OpenMP regions + serial computation time)) / max(runtime)\n\n"
            "CommE quantifies communication overhead by measuring the fraction of total runtime spent in \n"
            "communication rather than productive computation. When CommE is low, significant time is wasted\n"
            "on inter-process communication.\n\n"
            "Communication inefficiencies arise from two primary sources, each measured by separate metrics:\n"
            "\t1. Serialisation overhead - time processes wait at communication points for synchronisation\n"
            "\t2. Data transfer inefficiency - time spent moving large data volumes relative to network capacity\n\n"
            "These sub-components combine multiplicatively to produce the overall Communication Efficiency:\n"
            "\tCommE = Serialisation Efficiency x Transfer Efficiency\n\n"
            "To derive these metrics, Scalasca trace analysis must be performed to:\n"
            "\t- Identify all synchronization points and their waiting times\n"
            "\t- Measure actual data transfer volumes and transfer rates\n"
            "\t- Detect inefficient communication patterns (e.g., load imbalance during MPI_Allreduce)";

        std::string not_active_text =
            "MPI Communication Efficiency (CommE) requires precise timing metrics that are not available \n"
            "when Cube reports are generated by tools other than Score-P/Scalasca. \n\n"
            "Critical missing components include:\n"
            "\t- Process timing information (for runtime calculations)\n"
            "\t- Communication event traces (for serialization/transfer analysis)\n\n"
            "Without these, complete POP analysis cannot be performed, \n"
            "including:\n"
            "\t- Communication Efficiency calculation\n"
            "\t- Serialization efficiency assessment\n"
            "\t- Transfer efficiency measurement";

        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;


public:
    POPHybridCommunicationEfficiencyTestAdd(
        cube::CubeProxy*,
        POPHybridSerialisationTestAdd* pop_ser,
        POPHybridTransferTestAdd*      pop_transef );

    virtual
    ~POPHybridCommunicationEfficiencyTestAdd()
    {
    };

    virtual
    double
    analyze( const cube::list_of_cnodes& cnodes,
             cube::LocationGroup*        _lg = nullptr ) const;

// ------ overview tests ---------

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false  );

    bool
    isActive() const;

    bool
    isIssue() const;

    virtual
    std::list<PerformanceTest*>
    getPrereqs()
    {
        std::list<PerformanceTest*> prereqs;
        if ( pop_ser != nullptr && pop_transeff != nullptr )
        {
            prereqs.push_back( pop_transeff );
            prereqs.push_back( pop_ser );
        }
        return prereqs;
    }
};
};
#endif // ADVISER_RATING_WIDGET_H
