/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_IO_EFFICIENCY_TEST_H
#define CUBELIB_IO_EFFICIENCY_TEST_H

#include <string>
#include "PerformanceTest.h"
#include "POPPosixIOTimeTest.h"
#include "POPMpiIOTimeTest.h"

namespace popcalculation
{
class POPIOEfficiencyTest : public PerformanceTest
{
private:
    POPPosixIOTime* posix_io;
    POPMpiIOTime*   mpi_io;

    void
    calculate();


protected:

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false   );


    virtual
    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "IO Efficiency measures the proportion of execution time consumed by input/output operations rather than \n"
            "productive computations. It quantifies the performance impact of all I/O activities during program execution.\n"
            "\nThis metric is composed of two distinct components:\n"
            "\t- POSIX IO Efficiency: The fraction of time spent in standard POSIX I/O operations\n"
            "\t- MPI IO Efficiency: The fraction of time spent in MPI collective and point-to-point I/O operations\n"
            "\nThe overall IO Efficiency is calculated using:\n"
            "\tIO Efficiency = 1 - (1 - POSIX IO Efficiency) x (1 - MPI IO Efficiency)\n\n"
            "This composite metric provides a comprehensive evaluation of how I/O operations affect the overall \n"
            "performance of parallel applications, highlighting potential bottlenecks in data handling operations.";

        std::string not_active_text =
            "The IO Efficiency metric requires precise timing information specifically for I/O operations.\n"
            "It is only available in Score-P/Scalasca measurements where:\n"
            "\t- Either MPI I/O operations are present and properly instrumented, OR\n"
            "\t- POSIX I/O calls are detected and timed within the execution trace\n"
            "\t- The fundamental Time metric is fully collected with appropriate I/O call instrumentation\n\n"
            "When utilizing Cube Reports from alternative profiling tools or when:\n"
            "\t- The essential Time metric is incomplete or not properly correlated with I/O calls\n"
            "\t- No I/O-specific instrumentation (MPI IO or POSIX IO) is present\n"
            "\t- The required call stack information for I/O operations is unavailable\n"
            "the IO Efficiency calculation cannot be performed.\n"
            "In such cases, the metric will be reported as Not a Number (NAN), as the necessary \n"
            "instrumentation data for accurately measuring I/O time is either missing or incompatible.\n"
            "This limitation prevents proper analysis of I/O-related performance characteristics.";
        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;


public:
    POPIOEfficiencyTest( POPPosixIOTime*,
                         POPMpiIOTime* );

    virtual
    ~POPIOEfficiencyTest()
    {
    };

    bool
    isActive() const;

    bool
    isIssue() const;


    virtual
    std::list<PerformanceTest*>
    getPrereqs()
    {
        std::list<PerformanceTest*> prereqs;
        if ( posix_io != nullptr && mpi_io != nullptr )
        {
            prereqs.push_back( posix_io );
            prereqs.push_back( mpi_io );
        }
        return prereqs;
    }
};
};
#endif // ADVISER_RATING_WIDGET_H
