/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_POP_COMPUTATION_TIME_H
#define CUBELIB_POP_COMPUTATION_TIME_H

#include <string>
#include "PerformanceTest.h"


namespace popcalculation
{
class POPComputationTime : public PerformanceTest
{
private:
    cube::Metric* comp;

protected:

    void
    applyCnode( const cube::list_of_cnodes& cnodes,
                const bool                  direct_calculation = false   );

    virtual
    const std::string&
    getCommentText() const;

    inline
    virtual
    std::string
    getHelpUrl() const
    {
        std::string active_text =
            "Computation Time represents the duration spent executing actual computational work in the application.\n"
            "It is defined as the portion of execution time that excludes all parallelization, communication,\n"
            "I/O, and accelerator operations. Specifically, it excludes:\n"
            "\t- MPI communication and synchronization calls\n"
            "\t- OpenMP parallel regions\n"
            "\t- All I/O operations\n"
            "\t- Shared Memory (SHMEM) communications\n"
            "\t- Service library calls\n"
            "\t- GPU accelerated code (CUDA, OpenCL, OpenACC)\n"
            "\t- Instrumentation overhead from library wrapping\n"
            "\t- Any other explicitly parallel or offloaded execution\n\n"
            "This metric captures the genuine CPU computation time, providing a baseline\n"
            "for evaluating the efficiency of all other parallel and accelerator components.";

        std::string not_active_text =
            "The Computation Time metric requires the fundamental Time metric to be properly collected and maintained\n"
            "throughout the entire execution timeline.\n"
            "This metric is fully supported in all Score-P/Scalasca measurements as it forms the foundation\n"
            "of the performance analysis infrastructure.\n\n"
            "When Cube Reports are generated by tools other than Score-P/Scalasca or when:\n"
            "\t- The essential Time metric is incomplete or missing\n"
            "\t- There is no standardized instrumentation for computation time detection\n"
            "\t- The required call path information is unavailable\n"
            "the Computation Time metric cannot be accurately determined.\n"
            "In such cases, the metric will be reported as Not a Number (NAN), as the necessary timing\n"
            "and instrumentation data is either absent or incompatible with proper Computation Time calculation.\n"
            "This limitation prevents comprehensive performance analysis and efficiency evaluation.";

        return isActive() ? active_text : not_active_text;
    }

    virtual
    void
    adjustForTest( cube::CubeProxy* cube ) const;

public:
    POPComputationTime( cube::CubeProxy* );

    virtual
    ~POPComputationTime()
    {
    };
// ------ overview tests ---------


    virtual
    inline
    bool
    isPercent() const
    {
        return false;
    }

    bool
    isActive() const;

    bool
    isIssue() const;

    virtual
    inline
    bool
    isNormalized() const // returns true, if range goes from 0 to 1. false -> otherwise
    {
        return false;
    }

    inline
    virtual
    std::string
    units() const
    {
        return "sec";
    }
};
};
#endif // POP_ipc_TEST_H
