/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <future>
#include <thread>
#include "BSPOPHybridOMPLoadBalanceEfficiencyTest.h"

using namespace bscanalysis;

BSPOPHybridOMPLoadBalanceEfficiencyTest::BSPOPHybridOMPLoadBalanceEfficiencyTest(   cube::CubeProxy*               cube,
                                                                                    BSPOPHybridLoadBalanceTest*    _par_lb_eff,
                                                                                    BSPOPHybridMPILoadBalanceTest* _mpi_lb_eff
                                                                                    ) :
    popcalculation::PerformanceTest( cube ),
    par_lb_eff( _par_lb_eff ),
    mpi_lb_eff( _mpi_lb_eff )
{
    setName( " * OpenMP Load Balance Efficiency" );
    setWeight( 1 );   // need to be adjusted
    if ( par_lb_eff == nullptr || mpi_lb_eff == nullptr  ||
         ( !par_lb_eff->isActive() && !mpi_lb_eff->isActive() )
         )
    {
        setWeight( 0.2 );
        setValue( 0. );
        return;
    }
}


void
BSPOPHybridOMPLoadBalanceEfficiencyTest::calculate()
{
    if ( par_lb_eff == nullptr || mpi_lb_eff == nullptr  ||
         ( !par_lb_eff->isActive() && !mpi_lb_eff->isActive()  ) )
    {
        return;
    }
    double par_lb_eff_value = par_lb_eff->value();
    double mpi_lb_eff_value = mpi_lb_eff->value();
    setValue( ( par_lb_eff->isActive() ? par_lb_eff_value : 1. ) /
              ( mpi_lb_eff->isActive() && ( mpi_lb_eff_value > std::numeric_limits<double>::min() ) ? mpi_lb_eff_value : 1. ) );
}

static
void
hyb_lb_task_full_( BSPOPHybridLoadBalanceTest* _par_lb_eff,
                   const cube::list_of_cnodes& cnodes )
{
    if ( _par_lb_eff )
    {
        _par_lb_eff->applyCnode( cnodes );
    }
}

static
void
hyb_mpi_lb_task_full_( BSPOPHybridMPILoadBalanceTest* _mpi_lb_eff,
                       const cube::list_of_cnodes&    cnodes )
{
    if ( _mpi_lb_eff )
    {
        _mpi_lb_eff->applyCnode( cnodes );
    }
}




void
BSPOPHybridOMPLoadBalanceEfficiencyTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                                     const bool                   )
{
    auto hyb_lb_task_result     = std::async( std::launch::async, hyb_lb_task_full_, par_lb_eff, cnodes );
    auto hyb_mpi_lb_task_result = std::async( std::launch::async, hyb_mpi_lb_task_full_, mpi_lb_eff, cnodes );

    hyb_lb_task_result.get();
    hyb_mpi_lb_task_result.get();

    calculate();
}


const std::string&
BSPOPHybridOMPLoadBalanceEfficiencyTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
BSPOPHybridOMPLoadBalanceEfficiencyTest::isActive() const
{
    return par_lb_eff != nullptr &&
           mpi_lb_eff != nullptr &&
           (
        ( par_lb_eff->isActive() )  || ( mpi_lb_eff->isActive() ) )
    ;
};

bool
BSPOPHybridOMPLoadBalanceEfficiencyTest::isIssue() const
{
    return false;
};



void
BSPOPHybridOMPLoadBalanceEfficiencyTest::adjustForTest( cube::CubeProxy* cube ) const
{
    ( void )cube;
}
