/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>

#include "BSPOPHybridMPITransferTest.h"


using namespace bscanalysis;

BSPOPHybridMPITransferTest::BSPOPHybridMPITransferTest( cube::CubeProxy* cube ) : popcalculation::PerformanceTest( cube )
{
    setName( " * * * MPI Transfer Efficiency" );
    setWeight( 1 );   // need to be adjusted
    max_total_time       = cube->getMetric( "max_total_time_hyb" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        adjustForTest( cube );
    }
    max_total_time       = cube->getMetric( "max_total_time_hyb" );
    max_total_time_ideal = cube->getMetric( "max_total_time_ideal_hyb" );

    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        setWeight( 0.1 );
        setValue( 0. );
        return;
    }
    cube::metric_pair metric;
    metric.first  = max_total_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );


    metric.first  = max_total_time_ideal;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_total_time_ideal.push_back( metric );
}

void
BSPOPHybridMPITransferTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                        const bool                  direct_calculation  )
{
    ( void )direct_calculation; // not used here
    setValue(  analyze( cnodes ) );
}

double
BSPOPHybridMPITransferTest::analyze( const cube::list_of_cnodes& cnodes,
                                     cube::LocationGroup* ) const
{
    if ( max_total_time_ideal == nullptr || max_total_time == nullptr )
    {
        return 0.;
    }

    cube::value_container inclusive_values1;
    cube::value_container exclusive_values1;
    cube->getSystemTreeValues( lmetrics,
                               cnodes,
                               inclusive_values1,
                               exclusive_values1 );

    cube::value_container inclusive_values2;
    cube::value_container exclusive_values2;
    cube->getSystemTreeValues( lmax_total_time_ideal,
                               cnodes,
                               inclusive_values2,
                               exclusive_values2 );

    const std::vector<cube::LocationGroup*>& _lgs                         = cube->getLocationGroups();
    double                                   max_total_time_ideal_value_d = 0.;
    double                                   max_total_time_value_d       = 0.;
    for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
    {
        if ( ( *iter )->get_type() != cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
        {
            continue;
        }
        max_total_time_value_d       +=  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
        max_total_time_ideal_value_d +=  inclusive_values2[ ( *iter )->get_sys_id() ]->getDouble();
    }

    std::for_each( inclusive_values1.begin(), inclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values1.begin(), exclusive_values1.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( inclusive_values2.begin(), inclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );
    std::for_each( exclusive_values2.begin(), exclusive_values2.end(),  [ ]( cube::Value* element ){
        delete element;
    } );


    return max_total_time_ideal_value_d / max_total_time_value_d;
}


const std::string&
BSPOPHybridMPITransferTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
BSPOPHybridMPITransferTest::isActive() const
{
    return ( max_total_time_ideal != nullptr ) && ( max_total_time != nullptr );
};

bool
BSPOPHybridMPITransferTest::isIssue() const
{
    return false;
};

void
BSPOPHybridMPITransferTest::adjustForTest( cube::CubeProxy* cube ) const
{
    if ( scout_metrics_available( cube ) )
    {
        add_max_total_time_ideal( cube );
        add_max_total_time( cube );
    }
}



void
BSPOPHybridMPITransferTest::add_max_total_time_ideal( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "max_total_time_ideal_hyb" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal total time in ideal network, hybrid",
            "max_total_time_ideal_hyb",
            "DOUBLE",
            "sec",
            "",
            BSPOP_TRANSFER_EFF_METRIC_URL,
            "Maximal total time in ideal network, ( execution - transfer_time_mpi )",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution() - metric::transfer_time_mpi()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}


void
BSPOPHybridMPITransferTest::add_max_total_time( cube::CubeProxy* ) const
{
    add_execution_time( cube );
    cube::Metric* _met = cube->getMetric( "max_total_time_hyb" );
    if ( _met == nullptr )
    {
        _met = cube->defineMetric(
            "Maximal total time, hybrid",
            "max_total_time_hyb",
            "DOUBLE",
            "sec",
            "",
            BSPOP_TRANSFER_EFF_METRIC_URL,
            "Maximal total time, max( execution )",
            nullptr,
            cube::CUBE_METRIC_PREDERIVED_INCLUSIVE,
            "metric::execution()",
            "",
            "",
            "",
            "max(arg1, arg2)",
            true,
            cube::CUBE_METRIC_GHOST
            );
        if ( _met != nullptr )
        {
            _met->setConvertible( false );
        }
        _met->def_attr( "origin", "advisor" );
    }
}
