/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <list>
#include "BSPOPHybridAuditAnalysis.h"
#include "PerformanceTest.h"


using namespace cube;
using namespace bscanalysis;



BSPOPHybridAuditPerformanceAnalysis::BSPOPHybridAuditPerformanceAnalysis( cube::CubeProxy* _cube ) : popcalculation::PerformanceAnalysis( _cube )
{
    stalled_resources = new popcalculation::POPStalledResourcesTest( cube );
    ipc               = new popcalculation::POPIPCTest( cube );
    wall              = new popcalculation::POPWallTimeTest( cube );
    no_wait_ins       = new popcalculation::POPNoWaitINSTest( cube );
    comp              = new popcalculation::POPComputationTime( cube );
    gpu_comp          = new popcalculation::POPGPUComputationTime( cube );
    posix_io          = new popcalculation::POPPosixIOTime( cube );
    mpi_io            = new popcalculation::POPMpiIOTime( cube );
    io_eff            = new popcalculation::POPIOEfficiencyTest( posix_io, mpi_io );

    mpi_ser_eff      = new BSPOPHybridMPISerialisationTest( cube );
    mpi_transfer_eff = new BSPOPHybridMPITransferTest( cube );
    mpi_comm_eff     = new BSPOPHybridMPICommunicationEfficiencyTest( cube );
    mpi_lb_eff       = new BSPOPHybridMPILoadBalanceTest( cube, mpi_comm_eff );
    mpi_pareff       = new BSPOPHybridMPIParallelEfficiencyTest( cube );


    hyb_comm_eff = new BSPOPHybridCommunicationEfficiencyTest( cube );
    hyb_lb_eff   = new BSPOPHybridLoadBalanceTest( cube, hyb_comm_eff );
    hyb_par_eff  = new BSPOPHybridParallelEfficiencyTest( cube );

    omp_comm_eff = new BSPOPHybridOMPCommunicationEfficiencyTest( cube, hyb_comm_eff, mpi_comm_eff  );
    omp_lb_eff   = new BSPOPHybridOMPLoadBalanceEfficiencyTest( cube, hyb_lb_eff, mpi_lb_eff );
    omp_par_eff  = new BSPOPHybridOMPParallelEfficiencyTest( cube, hyb_par_eff, mpi_pareff );

    gpu_comm_eff = new popcalculation::POPGPUCommunicationEfficiencyTest( cube );
    gpu_lb_eff   = new popcalculation::POPGPUImbalanceTest( cube );
    gpu_par_eff  = new popcalculation::POPGPUParallelEfficiencyTest( gpu_lb_eff, gpu_comm_eff );

    popcalculation::PerformanceTest::finalizePrepsForTest( _cube );

    max_ipc = ipc->getMaximum();
}


BSPOPHybridAuditPerformanceAnalysis::~BSPOPHybridAuditPerformanceAnalysis()
{
    delete gpu_comm_eff;
    delete gpu_lb_eff;
    delete gpu_par_eff;

    delete stalled_resources;
    delete ipc;
    delete wall;
    delete no_wait_ins;
    delete comp;
    delete gpu_comp;
    delete posix_io;
    delete mpi_io;
    delete io_eff;

    delete mpi_transfer_eff;
    delete mpi_ser_eff;
    delete mpi_comm_eff;
    delete mpi_lb_eff;
    delete mpi_pareff;

    delete omp_comm_eff;
    delete omp_lb_eff;
    delete omp_par_eff;

    delete hyb_comm_eff;
    delete hyb_lb_eff;
    delete hyb_par_eff;
}


std::list<popcalculation::PerformanceTest*>
BSPOPHybridAuditPerformanceAnalysis::getAllTestsForCalculation()
{
    std::list<popcalculation::PerformanceTest*> to_return;
    to_return.push_back( wall );
    to_return.push_back( stalled_resources );
    to_return.push_back( ipc );
    to_return.push_back( no_wait_ins );
    to_return.push_back( comp );
    to_return.push_back( gpu_comp );
    to_return.push_back( io_eff );
    to_return.push_back( gpu_par_eff );
    to_return.push_back( omp_comm_eff );
    to_return.push_back( omp_lb_eff );
    to_return.push_back( omp_par_eff );
    return to_return;
}



std::list<popcalculation::PerformanceTest*>
BSPOPHybridAuditPerformanceAnalysis::getPOPTests()
{
    std::list<popcalculation::PerformanceTest*> to_return;
    to_return.push_back( hyb_par_eff );
    to_return.push_back( hyb_lb_eff );
    to_return.push_back( hyb_comm_eff );
    to_return.push_back( mpi_pareff );
    to_return.push_back( mpi_lb_eff );
    to_return.push_back( mpi_comm_eff );
    to_return.push_back( mpi_ser_eff );
    to_return.push_back( mpi_transfer_eff );
    to_return.push_back( omp_par_eff );
    to_return.push_back( omp_lb_eff );
    to_return.push_back( omp_comm_eff );
    return to_return;
}

std::list<popcalculation::PerformanceTest*>
BSPOPHybridAuditPerformanceAnalysis::getGPUTests()
{
    std::list<popcalculation::PerformanceTest*> to_return;
    to_return.push_back( gpu_par_eff );
    to_return.push_back( gpu_lb_eff );
    to_return.push_back( gpu_comm_eff );
    return to_return;
}

std::list<popcalculation::PerformanceTest*>
BSPOPHybridAuditPerformanceAnalysis::getIOTests()
{
    std::list<popcalculation::PerformanceTest*> to_return;
    to_return.push_back( io_eff );
    to_return.push_back( posix_io );
    to_return.push_back( mpi_io );
    return to_return;
}

std::list<popcalculation::PerformanceTest*>
BSPOPHybridAuditPerformanceAnalysis::getAdditionalTests()
{
    std::list<popcalculation::PerformanceTest*> to_return;
    to_return.push_back( stalled_resources );
    to_return.push_back( ipc );
    to_return.push_back( no_wait_ins );
    to_return.push_back( comp );
    to_return.push_back( gpu_comp );
    return to_return;
}

std::list<popcalculation::PerformanceTest*>
BSPOPHybridAuditPerformanceAnalysis::getControlTests()
{
    std::list<popcalculation::PerformanceTest*> to_return;
    to_return.push_back( wall );
    return to_return;
}

// ------ overview tests ---------

bool
BSPOPHybridAuditPerformanceAnalysis::isActive() const
{
    return true;
}

std::string
BSPOPHybridAuditPerformanceAnalysis::getAnchorHowToMeasure()
{
    std::string text =
        "This is one approach to extend POP (see: <cube_pop_metrics -? mpi>) metrics for hybrid (MPI+OpenMP) applications. It provides three types of efficiencies, i.e.,\n"
        "\t# Hybrid Parallel Efficiency reveals the inefficiency in processes and thread utilisation and can be broken down into \n"
        "\t\tHybrid Load Balance Efficiency and Hybrid Communication Efficiency\n"
        "\t# MPI Parallel Efficiency reveals the inefficiency in MPI processes and can be broken down into \n"
        "\t\tMPI Load Balance Efficiency and MPI Communication Efficiency\n"
        "\t# OpenMP Parallel Efficiency reveals the inefficiency in OpenMP thread usage and can be broken down into \n"
        "\t\tOpenMP Load Balance Efficiency and OpenMP Communication Efficiency\n";
    return text;
}
