/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023                                                     **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>

/*-------------------------------------------------------------------------*/
/**
 *  @file
 *  @ingroup CUBE_lib.network
 *  @brief   Declaration of the class cube::PluginRequest which implements
 *           cube server side plugins
 **/
/*-------------------------------------------------------------------------*/

#include <iostream>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <regex>

#include "CubeIoProxy.h"
#include "CubePlatformsCompat.h"
#include "CubeClientCallbackData.h"
#include "CubeClientConnection.h"
#include "CubePluginRequest.h"
#include "CubeServerCallbackData.h"
#include "CubeServerConnection.h"
#include "cube_network_types.h"
#include "CubeServerPlugin.h"
#include "CubePluginManager.h"

using namespace std;
using namespace cube;


/// --- Request identification -------------------------------------------

string
PluginRequest::getName() const
{
    return "PluginRequest";
}


NetworkRequest::id_t
PluginRequest::getId() const
{
    return PLUGIN_REQUEST;
}

/// --- Construction & destruction ---------------------------------------
///

NetworkRequestPtr
PluginRequest::create()
{
    string                     pluginName;
    std::vector<unsigned char> data, answer; // dummy on server side
    return std::shared_ptr<PluginRequest>( new PluginRequest( pluginName, data, answer ) );
}

NetworkRequestPtr
PluginRequest::create( const string& pluginName, const std::vector< unsigned char>& data,  std::vector< unsigned char>& answer )
{
    return std::shared_ptr<PluginRequest>( new PluginRequest( pluginName, data, answer ) );
}

void
PluginRequest::sendRequestPayload( ClientConnection& connection, ClientCallbackData* ) const
{
    connection << mPluginName;
    connection << ( uint64_t )mData.size();
    for ( unsigned char c : mData )
    {
        connection << c;
    }
}

void
PluginRequest::receiveRequestPayload( ServerConnection& connection, ServerCallbackData* data )
{
    unsigned char c;
    uint64_t      size;
    connection >> mPluginName;
    connection >> size;
    for ( uint64_t i = 0; i < size; i++ )
    {
        connection >> c;
        mData.push_back( c );
    }
}

void
PluginRequest::sendResponsePayload( ServerConnection& connection, ServerCallbackData* data ) const
{
    //cerr << "PluginRequest:: received " << this->mPluginName << this->mData << endl;
    std::vector<unsigned char> answer;
#ifdef HAVE_DLOPEN
    CubeServerPlugin* plugin = PluginManager::getInstance()->getPlugin( mPluginName );
    if ( plugin )
    {
        answer = plugin->send( this->mData, data->getCube() );
    }
    if ( !plugin )
    {
        cerr << "plugin " << mPluginName << " not found" << endl;
    }
#else
    cerr << "server side plugins are not supported" << endl;
#endif

    connection << ( uint64_t )answer.size();
    for ( unsigned char c : answer )
    {
        connection << c;
    }
}

void
PluginRequest::receiveResponsePayload( ClientConnection& connection, ClientCallbackData* )
{
    // receive answer from server plugin
    unsigned char c;
    uint64_t      size;
    connection >> size;
    for ( uint64_t i = 0; i < size; i++ )
    {
        connection >> c;
        mAnswer.push_back( c );
    }
}
