/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2025                                                     **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



/**
 * \file CubeParanoidStrategy.cpp
 * \brief In this strategy, rows are kept in memory as long as they are "in use".
 */

#include "config.h"
#include <iostream>
#include "CubeParanoidStrategy.h"

using namespace cube;

ParanoidStrategy::ParanoidStrategy( bool permissionToFreeAll )
    : m_permissionToFreeAll( permissionToFreeAll )
{
    // Constructor initializes members. No complex logic needed here.
}

ParanoidStrategy::~ParanoidStrategy()
{
    // Destructor. No dynamic memory to free within the strategy itself,
    // as map elements are automatically managed.
}

void
ParanoidStrategy::addRow( const cube::cnode_id_t& rowId, bool& readAllRows, std::vector<cnode_id_t>& rowsToRemove )
{
    std::lock_guard<std::mutex> lock( m_mutex );
    readAllRows = false; // This strategy doesn't inherently trigger a "read all"

    // If the row is not in our map (meaning its usage count is 0), it needs to be loaded.
    // Increment its usage counter.
    m_rowsUsageParanoid[ rowId ]++;

    // This strategy does not spontaneously remove rows on an add, so rowsToRemove is cleared.
    rowsToRemove.clear();
}

void
ParanoidStrategy::removeRows( std::vector<cnode_id_t>& rowsWantToRemove, std::vector<cnode_id_t>& rowsToRemove )
{
    std::lock_guard<std::mutex> lock( m_mutex );
    rowsToRemove.clear(); // Clear the output vector for this call

    for ( const cnode_id_t& rowId : rowsWantToRemove )
    {
        auto it = m_rowsUsageParanoid.find( rowId );
        if ( it != m_rowsUsageParanoid.end() )
        {
            it->second--; // Decrement the usage counter

            if ( it->second <= 0 )
            {
                rowsToRemove.push_back( rowId ); // Mark for removal if counter drops to 0 or below
                m_rowsUsageParanoid.erase( it ); // Remove from map
            }
        }
    }
}

bool
ParanoidStrategy::permissionToFreeAll()
{
    // Only grant permission to free all if the strategy is configured to allow it
    // AND there are no rows currently in use (all counters are zero or map is empty).
    std::lock_guard<std::mutex> lock( m_mutex );
    return m_permissionToFreeAll && m_rowsUsageParanoid.empty();
}

void
ParanoidStrategy::forcedFreeAll()
{
    std::lock_guard<std::mutex> lock( m_mutex );
    m_rowsUsageParanoid.clear(); // Clear all counters, effectively freeing all rows
}

void
ParanoidStrategy::needRows( std::vector<cnode_id_t>& rowsToAdd, std::vector<cnode_id_t>& rowsToRemoveFirst )
{
    std::lock_guard<std::mutex> lock( m_mutex );
    rowsToRemoveFirst.clear(); // Clear the output vector for rows to be removed.

    // Process rows to add: increment their counters.
    for ( const cnode_id_t& rowId : rowsToAdd )
    {
        m_rowsUsageParanoid[ rowId ]++;
    }

    // In this strategy, `needRows` doesn't inherently cause rows to be removed
    // unless they were already due for removal (counter <= 0), which would
    // be handled by a prior `removeRows` call or by the specific `removeRow` logic.
    // Since this method's purpose is to signal *needed* rows, it implies they will
    // be in memory. The removal logic is primarily driven by `removeRows`.
}
