/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_SYSRES_SET_EVALUATION_H
#define CUBELIB_SYSRES_SET_EVALUATION_H 0

#include "CubeNullaryEvaluation.h"
#include "CubeStringConstantEvaluation.h"
#include "Cube.h"
using namespace std;

namespace cube
{
class SysresSetEvaluation : public NullaryEvaluation
{
protected:
    sysres_kind        skind;
    cube::Cube*        cube;
    GeneralEvaluation* sysres_id;
    GeneralEvaluation* property;
    GeneralEvaluation* attribute_name;
    GeneralEvaluation* value;



public:
    SysresSetEvaluation( sysres_kind        _skind,
                         cube::Cube*        _cube,
                         GeneralEvaluation* _id,
                         GeneralEvaluation* _property,
                         GeneralEvaluation* _attribute_name,
                         GeneralEvaluation* _value ) : skind( _skind ), sysres_id( _id ), property( _property ), attribute_name( _attribute_name ), value( _value )
    {
        cube = _cube;
    };
    virtual
    ~SysresSetEvaluation();



    inline
    virtual
    void
    setRowSize( size_t size )
    {
        row_size = size;
        sysres_id->setRowSize( size );
        property->setRowSize( size );
        value->setRowSize( size );
        if ( attribute_name != nullptr )
        {
            attribute_name->setRowSize( size );
        }
    }


    inline
    virtual
    void
    set_metric_id( uint32_t _id )
    {
        met_id = _id;
        sysres_id->set_metric_id( _id );
        property->set_metric_id( _id );
        value->set_metric_id( _id );
        if ( attribute_name != nullptr )
        {
            attribute_name->set_metric_id( _id );
        }
    }



    inline
    virtual
    void
    set_verbose_execution( bool _v )
    {
        GeneralEvaluation::set_verbose_execution( _v );
        sysres_id->set_verbose_execution( _v );
        property->set_verbose_execution( _v );
        value->set_verbose_execution( _v );
        if ( attribute_name != nullptr )
        {
            attribute_name->set_verbose_execution( _v );
        }
    }

    inline
    virtual
    double
    eval() const
    {
        StringConstantEvaluation* prop = dynamic_cast<StringConstantEvaluation*>( property );
        StringConstantEvaluation* val  = dynamic_cast<StringConstantEvaluation*>( value );

        StringConstantEvaluation* attr_name       = ( attribute_name != nullptr ) ? dynamic_cast<StringConstantEvaluation*>( value ) : nullptr;
        string                    _prop           = ( prop  != nullptr ) ? prop->strEval() : "";
        string                    _value          = ( val != nullptr ) ? val->strEval() : "";
        string                    _attribute_name = ( attr_name != nullptr ) ? attr_name->strEval() : "";

        size_t id = static_cast<size_t>( sysres_id->eval() );
        switch ( skind )
        {
            case CUBE_SYSTEM_TREE_NODE:


                if ( id <  cube->get_stnv().size() )
                {
                    cube::SystemTreeNode* _stn = cube->get_stnv()[ id ];
                    if ( _prop == "attribute" )
                    {
                        _stn->def_attr( _attribute_name, _value );
                    }
                    else
                    {
                        cerr << "Supported only \"attribute\". Property \"" << _prop << "\" is ignored" << endl;
                    }
                }
                else
                {
                    cerr << "Cannot find system tree node with id=" << id << endl;
                }
                break;
            case CUBE_LOCATION_GROUP:
                if ( id <  cube->get_location_groupv().size() )
                {
                    cube::LocationGroup* _lg = cube->get_location_groupv()[ id ];
                    if ( _prop == "attribute" )
                    {
                        _lg->def_attr( _attribute_name, _value );
                    }
                    else
                    {
                        cerr << "Supported only \"attribute\". Property \"" << _prop << "\" is ignored" << endl;
                    }
                }
                else
                {
                    cerr << "Cannot find system tree node with id=" << id << endl;
                }
                break;
            case CUBE_LOCATION:
                if ( id <  cube->get_locationv().size() )
                {
                    cube::Location* _loc = cube->get_locationv()[ id ];
                    if ( _prop == "attribute" )
                    {
                        _loc->def_attr( _attribute_name, _value );
                    }
                    else
                    {
                        cerr << "Supported only \"attribute\". Property \"" << _prop << "\" is ignored" << endl;
                    }
                }
                else
                {
                    cerr << "Cannot find system tree node with id=" << id << endl;
                }
                break;
        }
        return 0;
    }


    inline
    virtual
    double
    eval( const Cnode*,
          const CalculationFlavour,
          const Sysres*,
          const CalculationFlavour )  const
    {
        eval();
        return 0.;
    };

    inline
    virtual
    double
    eval( const Cnode*,
          const CalculationFlavour ) const
    {
        eval();
        return 0.;
    };


    inline
    virtual
    double
    eval( double,
          double ) const
    {
        eval();
        return 0.;
    };

    inline
    virtual
    double*
    eval_row( const Cnode*,
              const CalculationFlavour ) const
    {
        eval();
        return nullptr;
    };




    inline
    virtual
    double
    eval( const list_of_cnodes&,
          const list_of_sysresources& ) const
    {
        eval();
        return 0.;
    };

    virtual
    double*
    eval_row( const list_of_cnodes&,
              const list_of_sysresources& ) const
    {
        eval();
        return nullptr;
    };




    inline
    virtual
    void
    print() const
    {
        if ( skind == CUBE_SYSTEM_TREE_NODE )
        {
            std::cout << "cube::stn::set(";
        }
        if ( skind == CUBE_LOCATION_GROUP )
        {
            std::cout << "cube::locationgroup::set(";
        }
        if ( skind == CUBE_LOCATION )
        {
            std::cout << "cube::locationg::set(";
        }
        sysres_id->print();
        std::cout << ",";
        property->print();
        std::cout << ",";
        if ( attribute_name != nullptr )
        {
            attribute_name->print();
            std::cout << ",";
        }
        value->print();
        std::cout << ");";
    };

    // fills an array with the list of metrics which are directly needed for the calculation.
    // data metrics -> nope operation, derived -> only metrics, which do appear in CubePL expression
    virtual
    inline
    void
    fillReqMetrics( std::vector<cube::Metric*>& v )
    {
        GeneralEvaluation::fillReqMetrics( v );
        value->fillReqMetrics( v );
        property->fillReqMetrics( v );
    }

    // fills an array with the list of metrics which are directly needed for the calculation.
    // data metrics -> nope operation, derived -> all metrics, which do appear in CubePL expression (also submetrics)
    virtual
    inline
    void
    fillAllReqMetrics( std::vector<cube::Metric*>& v )
    {
        GeneralEvaluation::fillAllReqMetrics( v );
        value->fillAllReqMetrics( v );
        property->fillAllReqMetrics( v );
    }
};
};

#endif
