/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2015                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBELIB_METRIC_SET_EVALUATION_H
#define CUBELIB_METRIC_SET_EVALUATION_H 0

#include "CubeNullaryEvaluation.h"
#include "CubeStringConstantEvaluation.h"
using namespace std;

namespace cube
{
class MetricSetEvaluation : public NullaryEvaluation
{
protected:
    cube::Metric*      metric;
    GeneralEvaluation* property;
    GeneralEvaluation* attribute_name;
    GeneralEvaluation* value;



public:
    MetricSetEvaluation( cube::Metric*      _met,
                         GeneralEvaluation* _property,
                         GeneralEvaluation* _value ) : metric( _met ), property( _property ), attribute_name( nullptr ), value( _value )
    {
    };

    MetricSetEvaluation( cube::Metric*      _met,
                         GeneralEvaluation* _property,
                         GeneralEvaluation* _attribute_name,
                         GeneralEvaluation* _value ) : metric( _met ), property( _property ), attribute_name( _attribute_name ), value( _value )
    {
    };
    virtual
    ~MetricSetEvaluation();



    inline
    virtual
    void
    setRowSize( size_t size )
    {
        row_size = size;
        property->setRowSize( size );
        value->setRowSize( size );
        if ( attribute_name != nullptr )
        {
            attribute_name->setRowSize( size );
        }
    }


    inline
    virtual
    void
    set_metric_id( uint32_t _id )
    {
        met_id = _id;
        property->set_metric_id( _id );
        value->set_metric_id( _id );
        if ( attribute_name != nullptr )
        {
            attribute_name->set_metric_id( _id );
        }
    }



    inline
    virtual
    void
    set_verbose_execution( bool _v )
    {
        GeneralEvaluation::set_verbose_execution( _v );
        property->set_verbose_execution( _v );
        value->set_verbose_execution( _v );
        if ( attribute_name != nullptr )
        {
            attribute_name->set_verbose_execution( _v );
        }
    }

    inline
    virtual
    double
    eval() const
    {
        if ( metric == nullptr )
        {
            return 0.;
        }
        StringConstantEvaluation* prop = dynamic_cast<StringConstantEvaluation*>( property );
        StringConstantEvaluation* val  = dynamic_cast<StringConstantEvaluation*>( value );

        StringConstantEvaluation* attr_name       = ( attribute_name != nullptr ) ? dynamic_cast<StringConstantEvaluation*>( value ) : nullptr;
        string                    _prop           = ( prop  != nullptr ) ? prop->strEval() : "";
        string                    _value          = ( val != nullptr ) ? val->strEval() : "";
        string                    _attribute_name = ( attr_name != nullptr ) ? attr_name->strEval() : "";
        if ( _prop == "value" )
        {
            metric->set_val( _value );
        }
        else if ( _prop == "attribute" )
        {
            metric->def_attr( _attribute_name, _value );
        }
        else
        {
            cerr << "Supported only \"value\" and \"attribute\". Property \"" << _prop << "\" is ignored" << endl;
        }
        return 0.;
    }


    inline
    virtual
    double
    eval( const Cnode*,
          const CalculationFlavour,
          const Sysres*,
          const CalculationFlavour )  const
    {
        eval();
        return 0.;
    };

    inline
    virtual
    double
    eval( const Cnode*,
          const CalculationFlavour ) const
    {
        eval();
        return 0.;
    };


    inline
    virtual
    double
    eval( double,
          double ) const
    {
        eval();
        return 0.;
    };

    inline
    virtual
    double*
    eval_row( const Cnode*,
              const CalculationFlavour ) const
    {
        eval();
        return nullptr;
    };




    inline
    virtual
    double
    eval( const list_of_cnodes&,
          const list_of_sysresources& ) const
    {
        eval();
        return 0.;
    };

    virtual
    double*
    eval_row( const list_of_cnodes&,
              const list_of_sysresources& ) const
    {
        eval();
        return nullptr;
    };




    inline
    virtual
    void
    print() const
    {
        if ( metric == nullptr ) // no metric -> no print
        {
            return;
        }
        std::cout << "cube::metric::set::" << metric->get_uniq_name() << "(";
        property->print();
        std::cout << ",";
        value->print();
        std::cout << ");";
    };

    // fills an array with the list of metrics which are directly needed for the calculation.
    // data metrics -> nope operation, derived -> only metrics, which do appear in CubePL expression
    virtual
    inline
    void
    fillReqMetrics( std::vector<cube::Metric*>& v )
    {
        GeneralEvaluation::fillReqMetrics( v );
        value->fillReqMetrics( v );
        property->fillReqMetrics( v );
    }

    // fills an array with the list of metrics which are directly needed for the calculation.
    // data metrics -> nope operation, derived -> all metrics, which do appear in CubePL expression (also submetrics)
    virtual
    inline
    void
    fillAllReqMetrics( std::vector<cube::Metric*>& v )
    {
        GeneralEvaluation::fillAllReqMetrics( v );
        value->fillAllReqMetrics( v );
        property->fillAllReqMetrics( v );
    }
};
};

#endif
