/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBE_SEMAPHORE
#define CUBE_SEMAPHORE

#include <mutex>
#include <condition_variable>
namespace cube
{
class CubeSemaphore
{
public:

    explicit CubeSemaphore( size_t count ) : count( count )
    {
    }

    size_t
    getCount() const
    {
        return count;
    };

    bool
    isFree() const
    {
        return count > 1;
    };
    void
    lock()        // call before critical section
    {
        std::unique_lock<std::mutex> lock( mutex );
        condition_variable.wait( lock, [ this ] {
                return count != 0;     // written out for clarity, could just be return (count != 0);
            } );
        --count;
    }
    void
    unlock()         // call after critical section
    {
        std::unique_lock<std::mutex> lock( mutex );
        ++count;
        condition_variable.notify_one();
    }

private:
    std::mutex              mutex;
    std::condition_variable condition_variable;
    size_t                  count;
};
};

#endif
