/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2016-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2014                                                     **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2014                                                     **
**  RWTH Aachen University, JARA-HPC                                       **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/*-------------------------------------------------------------------------*/
/**
 *  @file
 *  @ingroup CUBE_lib.network
 *  @brief   Declaration of the class OpenCubeRequest
 **/
/*-------------------------------------------------------------------------*/


#ifndef CUBE_OPENALGORITHMREQUEST_H
#define CUBE_OPENALGORITHMREQUEST_H

#include <string>
#include <vector>
#include "CubeOpenCubeRequest.h"
#include "CubeProxy.h"

namespace cube
{
/*-------------------------------------------------------------------------*/
/**
 *  @class   cube::OpenAlgorithmRequest
 *  @ingroup CUBE_lib.network
 *  @brief   Network request to open a remote Cube file.
 **/
/*-------------------------------------------------------------------------*/

class OpenAlgorithmRequest
    : public cube::OpenCubeRequest
{
public:
    /// Virtual Destructor
    virtual
    ~OpenAlgorithmRequest();

    /// @brief Client-side factory method.
    ///
    /// @param fileName
    ///     Fully-qualified name of the Cube file
    /// @return
    ///     Pointer to request
    ///
    static NetworkRequestPtr
    create( Algorithm                       algo,
            const std::vector<std::string>& fileNames,
            const std::vector<std::string>& options );

    /// @brief Server-side factory method.
    ///
    /// @return
    ///     Pointer to request
    ///
    static NetworkRequestPtr
    create();

    /// @brief Get the name of this request type.
    ///
    /// @return
    ///     String representation of request name.
    ///
    virtual std::string
    getName() const;

    /// @brief Get the unique network request ID.
    ///
    /// @return
    ///     Unique request Id defined in %cube::NetworkRequestId
    ///
    virtual id_t
    getId() const;

    /// @brief Server-side callback to execute after reception.
    ///
    /// @param data
    ///     Callback data structure.
    ///
    virtual void
    processRequest( ServerCallbackData* data );

protected:
    /// @brief Stream information from client to server.
    ///
    /// @param[in,out] connection
    ///     Reference to connection on client.
    /// @param[in,out] data
    ///     Pointer to callback data.
    ///
    virtual void
    sendRequestPayload( ClientConnection&   connection,
                        ClientCallbackData* data ) const;

    /// @brief Receive information from client on server.
    ///
    /// @param[in,out] connection
    ///     Reference to connection on client.
    /// @param[in,out] data
    ///     Pointer to callback data.
    ///
    virtual void
    receiveRequestPayload( ServerConnection&   connection,
                           ServerCallbackData* data );


private:
    /// @brief Server-side constructor hidden from public access.
    ///
    OpenAlgorithmRequest();

    /// @brief Client-side constructor hidden from public access.
    ///
    OpenAlgorithmRequest( Algorithm                       algo,
                          const std::vector<std::string>& fileNames,
                          const std::vector<std::string>& options );

    Algorithm                mAlgo;
    std::vector<std::string> mFileNames;
    std::vector<std::string> mOptions;
};
}      /* namespace cube */

#endif /* !CUBE_OPENALGORITHMREQUEST_H */
