/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2016-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2015                                                     **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2015                                                     **
**  RWTH Aachen University, JARA-HPC                                       **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


/*-------------------------------------------------------------------------*/
/**
 *  @file
 *  @ingroup CUBE_lib.base
 *  @brief   Declaration of the class cube::CubeIoProxy
 **/
/*-------------------------------------------------------------------------*/


#ifndef CUBE_IOPROXY_H
#define CUBE_IOPROXY_H

#include <string>
#include <vector>

#include "CubeProxy.h"


namespace cube
{
class Cnode;
class Cube;
class IdIndexMap;
class Metric;
class Sysres;
class SystemTreeNode;
class Value;

/*-------------------------------------------------------------------------*/
/**
 *  @class   cube::CubeIoProxy
 *  @ingroup CUBE_lib.base
 *  @brief   Implementation of the CubeProxy interface using local I/O calls.
 **/
/*-------------------------------------------------------------------------*/
class CubeIoProxy
    : public CubeProxy
{
public:
    /// @brief Default constructor
    ///
    CubeIoProxy();

    /// @brief constructor to be used for internally created cube object
    CubeIoProxy( Cube* cube );

    /// @brief Virtual destructor
    ///
    virtual
    ~CubeIoProxy();

    /// @brief Open a report based on filename.
    ///
    /// Open Cube report specified via filename according to the base
    /// URL specified during creation.
    ///
    /// @param filename
    ///     Path and filename
    ///
    virtual void
    openReport( const std::string& filename ) override;

    /// @brief Open the report specified in initial URL.
    ///
    /// Opens the Cube report specified in the initial URL used to create
    /// the CubeProxy object.
    ///
    virtual void
    openReport() override;

    /// @brief Close report currently open.
    ///
    virtual void
    closeReport() override;

    /// @since 4.9.0
    /// @brief server side plugin communication, not useful for IOProxy
    virtual std::vector<unsigned char>
    sendToPlugin( const std::string&, const std::vector<unsigned char>& ) override
    {
        return std::vector<unsigned char>();
    }

    virtual
    std::string
    get_statistic_name() override;

    // / drop specific row in all metrics
    virtual
    void
    dropAllRowsInMetric( cube::Metric* met ) override;


    /// @brief Save data to a new Cube report.
    ///
    /// @param filename
    ///     Filename of the new Cube report.
    ///
    virtual void
    saveCopy( const std::string& filename ) override;

    virtual std::vector<char>
    getMiscData( const std::string& name ) override;

    /// @}
    /// @name Metrics
    /// @{

    /// @brief Define a new metric.
    ///
    /// @param disp_name
    ///     Name as displayed in the UI
    /// @param uniq_name
    ///     Unique internal name as used in derived metrics
    /// @param dtype
    ///     Name of datatype
    /// @param uom
    ///     Unit of measure
    /// @param val
    ///     TODO
    /// @param url
    ///     URL of online description
    /// @param descr
    ///     Description of the metric as displayed in the UI
    /// @param parent
    ///     Pointer to parent metric (NULL for root metrics)
    /// @param type_of_metric
    ///     Method to compute metric
    /// @param expression
    ///     CubePL expression to compute metric
    /// @param init_expression
    ///     CubePL expression used to initialize metric
    /// @param aggr_plus_expression
    ///     CubePL expression to compute plus aggregation
    /// @param aggr_minus_expression
    ///     CubePL expression to compute minus aggregation
    /// @param aggr_aggr_expression
    ///     CubePL expression to compute aggregation over system tree
    /// @param row_wise
    ///     Row-wise storage of data
    /// @param visibility
    ///     Visibility in UI
    ///
    /// @return Pointer to metric object.
    ///
    virtual Metric*
    defineMetric( const std::string& disp_name,
                  const std::string& uniq_name,
                  const std::string& dtype,
                  const std::string& unit_of_measure,
                  const std::string& val,
                  const std::string& url,
                  const std::string& descr,
                  Metric*            parent,
                  TypeOfMetric       type_of_metric = CUBE_METRIC_EXCLUSIVE,
                  const std::string& expression = "",
                  const std::string& init_expression = "",
                  const std::string& aggr_plus_expression = "",
                  const std::string& aggr_minus_expression = "",
                  const std::string& aggr_aggr_expression = "",
                  const bool         row_wise = true,
                  VizTypeOfMetric    visibility = CUBE_METRIC_NORMAL ) override;

    /// @brief Retrieve a defined metric by its unique name
    ///
    /// @param uniq_name
    ///     Unique string name of metric
    /// @return
    ///     Pointer to metric object, NULL if metric not found.
    ///
    virtual Metric*
    getMetric( const std::string& uniq_name ) const override;

    /// @brief Get a vector of all metrics
    ///
    /// @return
    ///     Vector of metrics.
    ///
    virtual const std::vector< Metric* >&
    getMetrics() const override;

    /// @brief Get a vector of all root metrics
    ///
    /// @return
    ///     Vector of root metrics.
    ///
    virtual const std::vector< Metric* >&
    getRootMetrics() const override;

    /// @brief Get a vector of all non-visible (ghost) metrics
    ///
    /// @return
    ///     Vector of non-visible metrics.
    ///
    virtual const std::vector< Metric* >&
    getGhostMetrics() const override;

    /// @brief Get the data for all trees in the metric dimension.
    ///
    /// This call returns all data to fully expand all trees in the metric
    /// dimension.
    ///
    /// @param[in] cnode_selections
    ///     The selected cnodes.
    /// @param[in] sysres_selections
    ///     The selected system resources.
    /// @param[out] inclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     inclusive values. The container will be cleared before new data
    ///     is added.
    /// @param[out] exclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     exclusive values. The container will be cleared before new data
    ///     is added.
    ///
    virtual void
    getMetricTreeValues( const list_of_cnodes&       cnode_selections,
                         const list_of_sysresources& sysres_selections,
                         value_container&            inclusive_values,
                         value_container&            exclusive_values ) override;

    /// @brief Get the data for all trees in the metric dimension.
    ///
    /// This call returns all data to fully expand all trees in the metric
    /// dimension.
    ///
    /// @param[in] region_selections
    ///     The selected region.
    /// @param[in] sysres_selections
    ///     The selected system resources.
    /// @param[out] inclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     inclusive values. The container will be cleared before new data
    ///     is added.
    /// @param[out] exclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     exclusive values. The container will be cleared before new data
    ///     is added.
    ///
    virtual void
    getMetricTreeValues( const list_of_regions&      region_selections,
                         const list_of_sysresources& sysres_selections,
                         value_container&            inclusive_values,
                         value_container&            exclusive_values ) override;

    /// @brief Query inclusive and exclusive values of a Metric subtree.
    ///
    /// @param cnode_selections
    ///     The selected cnodes
    /// @param sysres_selections
    ///     The selected system resources
    /// @param root_metric
    ///     Root of the subtree
    /// @param depth
    ///     Depth of the subtree
    /// @param metric_id_indices
    ///     Associative container with id to index mappingY
    /// @param inclusive_values
    ///     List of inclusive values
    /// @param exclusive_values
    ///     List of exclusive values
    ///
    virtual void
    getMetricSubtreeValues( const list_of_cnodes&       cnode_selections,
                            const list_of_sysresources& sysres_selections,
                            Metric&                     root_metric,
                            size_t                      depth,
                            IdIndexMap&                 metric_id_indices,
                            value_container*            inclusive_values,
                            value_container*            exclusive_values ) override;


    /// @brief Query inclusive and exclusive values of a Metric subtree.
    ///
    /// @param region_selections
    ///     The selected region
    /// @param sysres_selections
    ///     The selected system resources
    /// @param root_metric
    ///     Root of the subtree
    /// @param depth
    ///     Depth of the subtree
    /// @param metric_id_indices
    ///     Associative container with id to index mappingY
    /// @param inclusive_values
    ///     List of inclusive values
    /// @param exclusive_values
    ///     List of exclusive values
    ///
    virtual void
    getMetricSubtreeValues( const list_of_regions&      regions_selections,
                            const list_of_sysresources& sysres_selections,
                            Metric&                     root_metric,
                            size_t                      depth,
                            IdIndexMap&                 metric_id_indices,
                            value_container*            inclusive_values,
                            value_container*            exclusive_values ) override;

    /// @}
    /// @name Call dimension
    /// @{

    /// @brief Get a vector of all call-tree nodes
    ///
    /// @return
    ///     Vector of cnodes
    ///
    virtual const std::vector< Cnode* >&
    getCnodes() const override;

    /// @brief Get a vector of the root elements of artificial cnode subtrees, which are cut of the main tree
    ///
    /// @return
    ///     Vector of cnodes
    ///
    virtual const std::vector<Cnode*>&
    getArtificialCnodes() const override;

    /// @brief Get a vector of all root cnodes.
    ///
    /// @return
    ///     Vector of root cnodes.
    ///
    virtual const std::vector< Cnode* >&
    getRootCnodes() const override;

    Value*
    calculateValue( const list_of_metrics&      metric_selections,
                    const list_of_cnodes&       cnode_selections,
                    const list_of_sysresources& sysres_selections ) override;

    /// @deprecated, use calculateValue
    virtual void
    getCallpathTreeValues( const list_of_metrics&      metric_selections,
                           const list_of_sysresources& sysres_selections,
                           value_container&            inclusive_values,
                           value_container&            exclusive_values ) override;

    /// @deprecated, use calculateValue
    virtual void
    getCallpathSubtreeValues( const list_of_metrics&      metric_selections,
                              const list_of_sysresources& sysres_selections,
                              Cnode&                      root_cnode,
                              size_t                      depth,
                              IdIndexMap&                 cnode_id_indices,
                              value_container*            inclusive_values,
                              value_container*            exclusive_values ) override;

    /// @brief Set given Cnode as new root Cnode.
    ///
    /// @param cnode
    ///     New root Cnode.
    ///
    virtual void
    rerootCnode( Cnode& cnode ) override;

    /** @deprecated */
    virtual void
    pruneCnode( Cnode& cnode ) override;

    /// @brief Remove all children of a given Cnode
    ///
    /// @param cnode
    ///     Cnode to remove children from.
    ///
    virtual void
    setCnodeAsLeaf( Cnode& cnode ) override;

    /// @brief Get a vector of regions.
    ///
    /// @return
    ///     Vector of regions.
    ///
    virtual const std::vector< cube::Region* >&
    getRegions() const override;

    /** @deprecated wrong subroutine values for derived metrics */
    virtual void
    getFlatTreeValues( const list_of_metrics&      metric_selections,
                       const list_of_sysresources& sysres_selections,
                       value_container&            inclusive_values,
                       value_container&            exclusive_values,
                       value_container&            difference_values ) override;

    /// @}
    /// @name System dimension
    /// @{

    /// @brief Get the data for a tree metric.
    ///
    /// This call enables the GUI to query metrics individually, so the GUI
    /// can enable access to an available root metric while other metrics of
    /// the tree are still being aggregated.
    ///
    /// @param[in] metric_selections
    ///     The selected metrics.
    /// @param[in] cnode_selections
    ///     The selected cnodes.
    /// @param[out] inclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     inclusive values. The container will be cleared before new data
    ///     is added.
    /// @param[out] exclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     exclusive values. The container will be cleared before new data
    ///     is added.
    ///
    virtual void
    getSystemTreeValues( const list_of_metrics& metric_selections,
                         const list_of_cnodes&  cnode_selections,
                         value_container&       inclusive_values,
                         value_container&       exclusive_values ) override;

    /// @brief Get the data for a tree metric.
    ///
    /// This call enables the GUI to query metrics individually, so the GUI
    /// can enable Access to an available root metric while other metrics of
    /// the tree are still being aggregated.
    ///
    /// @param[in] metric_selections
    ///     The selected metrics.
    /// @param[in] region_selections
    ///     The selected region.
    /// @param[out] inclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     inclusive values. The container will be cleared before new data
    ///     is added.
    /// @param[out] exclusive_values
    ///     Reference to a container to be used for the data returned as
    ///     exclusive values. The container will be cleared before new data
    ///     is added.
    ///
    virtual void
    getSystemTreeValues( const list_of_metrics& metric_selections,
                         const list_of_regions& region_selections,
                         value_container&       inclusive_values,
                         value_container&       exclusive_values ) override;

    /// @brief Get a vector of locations.
    ///
    /// @return
    ///     Vector of locations.
    ///
    virtual const std::vector< Location* >&
    getLocations() const override;

    /// @brief Get a vector of all location groups.
    ///
    /// @return
    ///     Vector of location groups
    ///
    virtual const std::vector< LocationGroup* >&
    getLocationGroups() const override;

    /// @brief Get a vector of all system-tree nodes
    ///
    /// @return
    ///     Vector of system-tree nodes
    ///
    virtual const std::vector< SystemTreeNode* >&
    getSystemTreeNodes() const override;

    /// @brief Get a vector of only the system-tree root nodes
    ///
    /// @return
    ///     Vector of system-tree root nodes
    ///
    virtual const std::vector< SystemTreeNode* >&
    getRootSystemTreeNodes() const override;

    /// @brief Get a vector of system resources
    ///
    /// @return
    ///     Vector of system resources
    ///
    virtual const std::vector< Sysres* >&
    getSystemResources() const override;

    /// @brief Get the specific Cartesian topology.
    ///
    /// @param cartesianId
    ///     Unique ID of Cartesian topology.
    /// @return
    ///     Reference to topology object.
    ///
    virtual const Cartesian&
    getCartesian( size_t cartesianId ) const override;

    /// @brief Get the number of Cartesian topologies defined.
    ///
    /// @return
    ///     Number of Cartesian topologies defined
    ///
    virtual size_t
    getNumCartesians() const override;

    /// @brief Get a vector of Cartesian topologies.
    ///
    /// @return
    ///     Topology vector.
    ///
    virtual const std::vector< Cartesian* >&
    getCartesians() const override;

    /// @}
    /// @name Online Documentation
    /// @{

    /// @brief Define a new mirror for the Metrics used in the Cube report.
    ///
    /// @param url
    ///     URL of the metric documentation.
    ///
    virtual void
    defineMirror( const std::string& url ) override;

    /// @brief Get a list of URLs as mirrors for documentation
    ///
    /// @return
    ///     List of URLs of documentation mirrors
    ///
    const std::vector< std::string >&
    getMirrors() override;

    /// @}
    /// @name CubePL
    /// @{

    /// @brief Query validity of CubePL driver
    ///
    /// @return
    ///     true, if CubePL driver is set. false, otherwise.
    ///
    virtual bool
    hasCubePlDriver() const override;

    /// @brief Get a active CubePL driver.
    ///
    /// @return
    ///     The CubePL driver
    ///
    virtual cubeplparser::CubePLDriver&
    getCubePlDriver() override;

    /// @brief Query validity of CubePL memory manager.
    ///
    /// @return
    ///     true, if CubePL memory manager is set, false, otherwise.
    ///
    virtual bool
    hasCubePlMemoryManager() const override;

    /// @brief Get the current CubePL memory manager.
    ///
    /// @return
    ///     CubePL memory manager
    ///
    virtual CubePLMemoryManager&
    getCubePlMemoryManager() override;

    /// @}
    /// @name Attributes
    /// @{

    /// @brief Define a string attribute.
    ///
    /// @param key
    ///     Unique attribute key
    /// @param value
    ///     Attribute value
    ///
    /// @note If an attribute of the same name is already defined,
    ///     it will be overwritten.
    ///
    virtual void
    defineAttribute( const std::string& key,
                     const std::string& value ) override;

    /// @brief Retrieve the value for a given attribute
    ///
    /// @param key
    ///     Unique attribute key
    /// @return
    ///     Attribute value, empty string for unknown keys.
    ///
    virtual std::string
    getAttribute( const std::string& key ) const override;

    /**
     * Retrieve the value for a given attribute
     * @param key Unique attribute key
     * @return Attribute value, empty string for unknown keys.
     */
    virtual const std::map< std::string, std::string >&
    getAttributes() const override;

    /// @}
    /// @name Memory settings
    /// @{

    /// @brief Set global memory strategy for the underlying Cube file
    ///
    /// @param strategy
    ///     Memory strategy
    ///
    virtual void
    setMemoryStrategy( CubeStrategy strategy ) override;

    /// @}
    /// @name Miscellaneous
    /// @{

    /// @brief Get the operation status of the current operation running.
    ///
    /// @return
    ///     Progress status structure
    ///
    virtual const ProgressStatus&
    getOperationProgressStatus() const override;

    virtual bool
    isCubePlExpressionValid( std::string& _to_test,
                             std::string& error_message ) override;

    virtual int
    getCubelibVersionNumber() override;

    /// --- IOProxy specific functions

    /// @brief Return internal Cube object.
    ///
    /// @return
    ///     Cube object
    ///
    virtual Cube*
    getCubeObject() const override
    {
        return mCube;
    };


private:
    /// @}

    /// @brief Internal pointer to Cube object
    ///
    /// @note declared mutable as Cube interface is not entirely const-correct.
    ///
    mutable Cube* mCube;
};
}      /* namespace cube */

#endif /* ! CUBE_IOPROXY_H */
