/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <QtPlugin>
#include <QInputDialog>
#include "ItemMarkerPlugin.h"
#include "PluginServices.h"
#include "TreeItemMarker.h"
#include "DefaultMarkerLabel.h"
#include "SystemTree.h"
#include "CubeLocation.h"
#include "CubeMetric.h"
#include "DefaultCallTree.h"
#include "Globals.h"
#include "TabManager.h"
#include "TreeView.h"

/**
 * If the plugin doesn't load, start cube with -verbose to get detailed information.
 * If the error message "Plugin verification data mismatch" is printed, check if the
 * same compiler and Qt version have been used.
 */

using namespace cubepluginapi;
using namespace itemmarkerplugin;
using namespace std;

bool
ItemMarkerPlugin::cubeOpened( PluginServices* service )
{
    this->service = service;
    connect( service, &PluginServices::contextMenuIsShown,
             this, &ItemMarkerPlugin::contextMenuIsShown );

    markerLabel = new DefaultMarkerLabel( "Tree item marker" );
    marker      = service->getTreeItemMarker( "Tree item marker", QList<QPixmap>(), false, markerLabel );


    relationLabel = new DefaultMarkerLabel( "Related tree items" );
    QList<QPixmap> icons;
    icons.append( QPixmap( ":/images/relation.png" ) );
    relationMarker = service->getTreeItemMarker( "Related tree items", icons, false, relationLabel );
    QList<QPixmap> icons2;
    icons2.append( QPixmap( ":/images/jump-arrow-small.png" ) );
    relatedTabLabel  = new DefaultMarkerLabel( "related tab" );
    relatedTabMarker = service->getTreeItemMarker( "Related tab", icons2, false, relatedTabLabel );

    QList<QPixmap> gicons;
    gicons.append( QPixmap( ":/images/ghost.png" ) );
    ghostMarker = service->getTreeItemMarker( "ghost metric", gicons, true );
    voidMarker  = service->getTreeItemMarker( "void metric", QList<QPixmap>(), true );

    // mark corresponding mpi item, if cuda item is selected
    initRelationMarker();
    connect( service, &PluginServices::treeItemIsSelected,
             this, &ItemMarkerPlugin::showRelationMarker );

    // mark call site items
    connect( service, &PluginServices::treeItemIsSelected,
             this, &ItemMarkerPlugin::showCallSiteMarker );
    connect( service, &PluginServices::tabActivated,
             this, &ItemMarkerPlugin::tabChanged );

    service->addSettingsHandler( this );

    markExpertMetrics();

    return true; // initialisation is ok => plugin should be shown
}

void
ItemMarkerPlugin::cubeClosed()
{
    service->disconnect();

    creatorItems.clear();
    markedItems.clear();
    markedRelationItems.clear();

    callSiteItems.clear();
    markedCallSiteRelationItems.clear();
}

void
ItemMarkerPlugin::markExpertMetrics()
{
    for ( TreeItem* metricItem :  service->getTreeItems( METRIC ) )
    {
        cube::Metric* metric = static_cast<cube::Metric*> ( metricItem->getCubeObject() );
        if ( metric->isGhost() )
        {
            service->addMarker( metricItem, ghostMarker );
        }
        else if ( std::strcmp( metric->get_val().c_str(), "VOID" ) == 0 )
        {
            service->addMarker( metricItem, voidMarker );
        }
    }
}

/*
 * Special case task tree:
 * If a kernel item in the task tree is selected
 * 1. the corresponding kernel item in the default call tree is marked
 * 2. the corresponding task item in the default call tree is selected
 * The selection in the call tree (2.) causes deletion of the relation marker (1.), so this function marks
 * the previous marker again.
 */
void
ItemMarkerPlugin::tabChanged( cubepluginapi::DisplayType )
{
    // changed tab to default task tree
    if ( !previousRelatedTabItems.isEmpty() )
    {
        for ( TreeItem* item : previousRelatedTabItems )
        {
            service->removeMarker( item, relationMarker );
            service->removeMarker( item, relatedTabMarker );
            service->addMarker( item, relationMarker );
            service->addMarker( item, relatedTabMarker );
            markedCallSiteRelationItems.append( item );
        }
        service->updateTreeView( DEFAULTCALLTREE );
        previousRelatedTabItems.clear();
    }
}

/** mark tree items pairs with the same call side id (experimental Score-P feature, id is part of the function parameter) */
void
ItemMarkerPlugin::showCallSiteMarker()
{
    cubegui::DefaultCallTree* callTree = dynamic_cast<cubegui::DefaultCallTree*>( cubegui::Globals::getTabManager()->getTree( DEFAULTCALLTREE ) );
    if ( !callTree )
    {
        return;
    }
    previousRelatedTabItems = relatedTabItems;
    relatedTabItems.clear();

    for ( TreeItem* item : markedCallSiteRelationItems )
    {
        service->removeMarker( item, relationMarker );
        service->removeMarker( item, relatedTabMarker );
    }
    markedCallSiteRelationItems.clear();
    for ( TreeItem* selection : service->getSelections( CALL ) )
    {
        QList<TreeItem*> allChildren;
        allChildren.append( selection );
        if ( !selection->isTopLevelItem() && !selection->getName().contains( "KERNELS" ) )
        {
            // check selection and all children of the selected items for relationship to other items
            QList<TreeItem*> tmp = allChildren;
            while ( !tmp.isEmpty() )
            {
                TreeItem* current = tmp.takeFirst();
                allChildren.append( current );
                tmp.append( current->getChildren() );
            }
        }

        for ( TreeItem* item : allChildren )
        {
            bool             hasRelatedTab = false;
            QList<TreeItem*> relatedItems  = callTree->getRelatedCallSiteItems( item );
            QStringList      allSites;
            for ( TreeItem* relatedItem : relatedItems ) // collect all related items
            {
                if ( relatedItem && !markedCallSiteRelationItems.contains( relatedItem ) )
                {
                    QString markerLabel =  "related to: " + relatedItem->getName();
                    if ( item->getTree() != relatedItem->getTree() )
                    {
                        markerLabel  += " in \"" + relatedItem->getTree()->getLabel() + "\" tab";
                        hasRelatedTab = true;
                    }
                    allSites.append( markerLabel );
                    markedCallSiteRelationItems.append( relatedItem );
                    service->addMarker( relatedItem, relationMarker );

                    markerLabel = "related to: " + item->getName();
                    if ( item->getTree() != relatedItem->getTree() )
                    {
                        relatedTabItems.append( relatedItem );
                        service->addMarker( relatedItem, relatedTabMarker );
                        markerLabel += " in \"" + item->getTree()->getLabel() + "\" tab";
                        relatedTabLabel->setLabel( item, "related item in \"" + relatedItem->getTree()->getLabel() + "\" tab" );
                        relatedTabLabel->setLabel( relatedItem, "related item in \"" + item->getTree()->getLabel() + "\" tab" );
                    }
                    relationLabel->setLabel( relatedItem, markerLabel );
                }
                relatedTabItems.append( item );
            }
            if ( !relatedItems.empty() && !markedCallSiteRelationItems.contains( item ) ) // set marker, if related items exist
            {
                markedCallSiteRelationItems.append( item );
                service->addMarker( item, relationMarker );
                relationLabel->setLabel( item, allSites.join( "\n- " ) );

                if ( hasRelatedTab )
                {
                    service->addMarker( item, relatedTabMarker );
                }
            }
        }
    }
    if ( !markedCallSiteRelationItems.isEmpty() )
    {
        previousRelatedTabItems.clear();
    }

    service->updateTreeView( DEFAULTCALLTREE );
}

/** search for related MPI tree item for each accelerator item and save them to relatedItems */
void
ItemMarkerPlugin::initRelationMarker()
{
    QList<TreeItem*>          acceleratorItems;
    QHash<QString, TreeItem*> processMap;

    for ( TreeItem* item : service->getActiveTree( SYSTEM )->getItems() )
    {
        cube::LocationGroup* location = dynamic_cast<cube::LocationGroup*>( item->getCubeObject() );
        if ( location )
        {
            if ( location->get_type() == cube::CUBE_LOCATION_GROUP_TYPE_ACCELERATOR )
            {
                acceleratorItems.append( item );
            }
            else if ( location->get_type() == cube::CUBE_LOCATION_GROUP_TYPE_PROCESS )
            {
                processMap[ item->getName() ] = item;
            }
        }
    }
    // fill relatedItems: acceleratorItem -> MPI item
    for ( TreeItem* acceleratorItem : acceleratorItems )
    {
        cube::LocationGroup*               accelerator = dynamic_cast<cube::LocationGroup*>( acceleratorItem->getCubeObject() );
        std::map<std::string, std::string> attrs       = accelerator->get_attrs();
        if ( attrs.size() != 0 )
        {
            map<string, string>::const_iterator ai;
            for ( ai = attrs.begin(); ai != attrs.end(); ++ai )
            {
                QString str = ai->second.c_str();
                if ( !str.isEmpty() )
                {
                    TreeItem* creatorItem = processMap[ str ];
                    if ( creatorItem )
                    {
                        creatorItems[ acceleratorItem ] = creatorItem;
                        childItems[ creatorItem ].append( acceleratorItem );
                        for ( TreeItem* child : acceleratorItem->getChildren() )
                        {
                            creatorItems[ child ] = creatorItem;
                        }
                        relationLabel->setLabel( creatorItem, "creates " + acceleratorItem->getName() );
                        relationLabel->setLabel( acceleratorItem, "is created by " + creatorItem->getName() );
                    }
                }
            }
        }
    }
}

/** mark related MPI tree item, if an accelerator item is selected */
void
ItemMarkerPlugin::showRelationMarker()
{
    for ( TreeItem* item : markedRelationItems )
    {
        service->removeMarker( item, relationMarker );
    }
    markedRelationItems.clear();
    for ( TreeItem* item : service->getSelections( SYSTEM ) )
    {
        TreeItem* relatedItem = creatorItems[ item ];
        if ( relatedItem && !markedRelationItems.contains( relatedItem ) )
        {
            markedRelationItems.append( relatedItem );
            service->addMarker( relatedItem, relationMarker );
        }
        else
        {
            for ( TreeItem* relatedItem : childItems[ item ] )
            {
                if ( !markedRelationItems.contains( relatedItem ) )
                {
                    markedRelationItems.append( relatedItem );
                    service->addMarker( relatedItem, relationMarker );
                }
            }
        }
        if ( markedRelationItems.size() > 0 )
        {
            if ( !markedRelationItems.contains( item ) )
            {
                markedRelationItems.append( item );
                service->addMarker( item, relationMarker );
            }
        }
    }
    service->updateTreeView( SYSTEMTREE );
}

/**
 * @brief DemoPlugin::contextMenuIsShown is called, if the user clicks with right mouse button on a tree
 * item.
 */
void
ItemMarkerPlugin::contextMenuIsShown( DisplayType type, TreeItem* item )
{
    contextItem = item;

    if ( item )
    {
        if ( !markedItems.contains( item ) )
        {
            QAction* contextAction = service->addContextMenuItem( type, tr( "Mark this item" ) );
            connect( contextAction, &QAction::triggered, this, &ItemMarkerPlugin::setMarker );
        }
        else
        {
            QAction* contextAction = service->addContextMenuItem( type, tr( "Remove marker" ) );
            connect( contextAction, &QAction::triggered, this, &ItemMarkerPlugin::removeMarker );
            contextAction = service->addContextMenuItem( type, tr( "Set marker label" ) );
            connect( contextAction, &QAction::triggered, this, &ItemMarkerPlugin::editMarker );
        }
    }
    if ( ( type == CALL ) && !markedCallSiteRelationItems.empty() )
    {
        QAction* contextAction = service->addContextMenuItem( type, tr( "Select marked items" ) );
        connect( contextAction, &QAction::triggered, this, &ItemMarkerPlugin::selectMarked );
    }
}

void
ItemMarkerPlugin::removeMarker()
{
    service->removeMarker( contextItem, marker );
    service->updateTreeView( contextItem->getDisplayType() );
    markedItems.removeAll( contextItem );
}

void
ItemMarkerPlugin::setMarker()
{
    service->addMarker( contextItem, marker );
    service->updateTreeView( contextItem->getDisplayType() );
    markedItems.append( contextItem );
}

void
ItemMarkerPlugin::editMarker()
{
    bool    ok;
    QString text = QInputDialog::getText( service->getParentWidget(), tr( "Set marker label" ),
                                          tr( "Insert label:" ), QLineEdit::Normal, markerLabel->getLabel( contextItem ), &ok );
    if ( ok && !text.isEmpty() )
    {
        markerLabel->setLabel( contextItem, text );
    }
}

void
ItemMarkerPlugin::selectMarked()
{
    if ( markedCallSiteRelationItems.empty() )
    {
        return;
    }
    Tree*              tree = markedCallSiteRelationItems.first()->getTree();
    cubegui::TreeView* view = cubegui::Globals::getTabManager()->getView( tree );
    view->selectItems( markedCallSiteRelationItems );
}

/** set a version number, the plugin with the highest version number will be loaded */
void
ItemMarkerPlugin::version( int& major, int& minor, int& bugfix ) const
{
    major  = 1;
    minor  = 0;
    bugfix = 0;
}

/** unique plugin name */
QString
ItemMarkerPlugin::name() const
{
    return "TreeItem Marker";
}

QString
ItemMarkerPlugin::getHelpText() const
{
    return tr( "This Plugin marks related items in the call, task and system tree. It also allows the user to mark tree items manually using the context menu." );
}

void
ItemMarkerPlugin::loadExperimentSettings( QSettings& settings )
{
    QList<QVariant> list      = settings.value( "marked" ).toList();
    QList<QVariant> labelList = settings.value( "markedLabel" ).toList();

    for ( int i = 0; i < list.size(); i++ )
    {
        TreeItem* item  = TreeItem::convertQVariantToTreeItem( list.at( i ) );
        QString   label = labelList.at( i ).toString();
        if ( item )
        {
            contextItem = item;
            setMarker();
            markerLabel->setLabel( contextItem, label );
        }
    }
}

void
ItemMarkerPlugin::saveExperimentSettings( QSettings& settings )
{
    QList<QVariant> list;
    QStringList     labelList;
    foreach( TreeItem * item, markedItems )
    {
        QVariant data = item->convertToQVariant();
        list.append( data );
        labelList.append( markerLabel->getLabel( item ) );
    }
    settings.setValue( "marked", QVariant( list ) );
    settings.setValue( "markedLabel", QVariant( labelList ) );
}
