/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2022                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include "ViolinPlot.h"
#include "StatisticalInformation.h"
#include "Globals.h"

#include <iostream>
#include <fstream>
#include <sstream>
#include <limits>

#include <QFontMetrics>
#include <QVBoxLayout>
#include <QPushButton>
#include <QMouseEvent>
#include <QDialog>
#include <QString>
#include <QPen>
#include <QDebug>
#include <math.h>
#include <QPainter>
#include <QEasingCurve>
#include <QPropertyAnimation>
#include <QPointF>
#include <QPainterPath>

using namespace std;
using namespace cubegui;
using namespace system_statistics;


ViolinPlot::ViolinPlot( QWidget* parent, QDialog* info )
    : Chart( parent ), firstPosition( 0, 0 )
{
    minimumValue = 0.0;
    maximumValue = 0.0;
    tooltip      = nullptr;
    infoDialog   = info;
}

/**
   set the data of a boxplot which consists only of one StatisticalInformation
   existing StatisticalInformations will be deleted
   @param violinStatistics values to generate a violinPlot
   @param stat basic statistical values for absolute and relative value mode
 */
void
ViolinPlot::set( const ViolinStatistics& violinStatistics, const StatisticPair& stat, double startValue )
{
    item       = violinStatistics;
    statistics = stat;

    minimumValue = startValue;
    maximumValue = item.getStatistics().getMaximum();

    setYRange( minimumValue, maximumValue );
}

void
ViolinPlot::drawChart( QPainter& painter )
{
    if ( item.getStatistics().getCount() == 0 )
    {
        painter.drawText( rect(), Qt::AlignCenter, tr( "Not enough values to display." ) );
        return;
    }
    drawViolinPlot( painter );
}

std::vector<AxisLabel>
ViolinPlot::generateRightAxisValues()
{
    return generateStatisticAxis( statistics.getCurrent(), statistics.getAbsolute() );
}

void
ViolinPlot::drawViolinPlot( QPainter& painter )
{
    const std::vector<double>& kernelDensity      = item.getKernelDensity();
    double                     kernel_density_min = item.getKernelDensityMin();
    double                     kernel_density_max = item.getKernelDensityMax();

    //item.selectKernel( used_kernel, used_width, used_order, CalculateHeight( item.getStatistics().getMinimum() ), CalculateHeight( item.getStatistics().getMaximum() ), item.getStatistics().getMinimum(), item.getStatistics().getMaximum() );
    //    item.make_width( used_width, CalculateHeight( item.getStatistics().getMinimum() ), CalculateHeight( item.getStatistics().getMaximum() ), item.getStatistics().getMinimum(), item.getStatistics().getMaximum() );
    //item.optimalKernel(CalculateHeight( item.getStatistics().getMinimum() ), CalculateHeight( item.getStatistics().getMaximum() ), item.getStatistics().getMinimum(), item.getStatistics().getMaximum());

    QColor penColor = palette().color( QPalette::WindowText );
    QPen   defaultPen( penColor );
    QPen   redPen( Qt::red );
    QPen   dashPen( defaultPen );
    dashPen.setStyle( Qt::DashLine );
    QPen thickPen;
    thickPen.setWidth( 3 );
    thickPen.setColor( Qt::blue );

    // draw lines for median, mean ...
    painter.setPen( thickPen );
    painter.drawLine( startX, getY( item.getStatistics().getMedian() ), chartWidth + startX, getY( item.getStatistics().getMedian() ) );
    painter.setPen( dashPen );
    painter.drawLine( startX, getY( item.getStatistics().getMean() ), chartWidth + startX, getY( item.getStatistics().getMean() ) );
    painter.setPen( redPen );
    painter.drawLine( startX, getY( item.getStatistics().getQ1() ), chartWidth + startX,  getY( item.getStatistics().getQ1() )  );
    painter.drawLine( startX, getY( item.getStatistics().getQ3() ), chartWidth + startX,  getY( item.getStatistics().getQ3() )  );

    // draw violing plot
    int    chartWidthThick = chartWidth - 2; // reduce by 2 pixels to prevent overlapping because of the thickness of the pen
    int    middle          = startX + 1 + chartWidthThick / 2;
    double stretchX        = ( chartWidthThick / 2.0 ) / ( kernel_density_max - kernel_density_min );
    int    x_current       = middle + ( kernelDensity.at( 0 ) - kernel_density_min ) * stretchX;
    int    opx_current     = middle - ( kernelDensity.at( 0 ) - kernel_density_min ) * stretchX;
    int    y_current       =  getY( item.getStatistics().getMinimum() );

    // draw a connection between min and max
    painter.setPen( dashPen );
    painter.drawLine( middle, getY( item.getStatistics().getMaximum() ), middle,
                      getY( item.getStatistics().getMinimum() ) );

    painter.setPen( thickPen );
    painter.drawLine( x_current, y_current, opx_current, y_current ); // line from left to right side of the violin plot
    for ( unsigned j = 1; j < kernelDensity.size(); j++ )
    {
        int x_next   = middle + ( kernelDensity.at( j ) - kernel_density_min ) * stretchX;
        int y_next   = getY( item.getNewdata().at( j ) );
        int opx_next = middle - ( kernelDensity.at( j ) - kernel_density_min ) * stretchX;
        painter.drawLine( x_current, y_current, x_next, y_next );     // right half of the violin plot
        painter.drawLine( opx_current, y_current, opx_next, y_next ); // left half
        y_current   = y_next;
        x_current   = x_next;
        opx_current = opx_next;
    }
    painter.drawLine( x_current, y_current, opx_current, y_current ); // line from left to right side of the violin plot

    painter.setPen( defaultPen );
}

void
ViolinPlot::mousePressEvent( QMouseEvent* event )
{
    Chart::mousePressEvent( event );
    if ( event->button() == Qt::RightButton )
    {
        QPoint mousePos = this->mapToGlobal( event->pos() );
        tooltip = Chart::showStatisticToolTip( this, mousePos, statistics.toHtml() );
    }
    mousePressPos = event->pos();
}


QString
ViolinPlot::getAreaDescription() const
{
    double                        y1   = getValue( selectedArea.y() + selectedArea.height() );
    double                        y2   = getValue( selectedArea.y() );
    const StatisticalInformation& stat = item.getStatistics();
    return QString::number( stat.countRange( y1, y2 ) ) + " elements / " +  QString::number( stat.getCount() );
}

void
ViolinPlot::mouseReleaseEvent( QMouseEvent* event )
{
    Chart::mouseReleaseEvent( event );
    if ( event->button() == Qt::LeftButton )
    {
        const int minimumRange = 5;  // minimum number of pixels for range selection
        if ( std::abs( event->pos().y() - mousePressPos.y() ) < minimumRange )
        {
            Chart::showStatisticWindow( this, tr( "Statistics info" ),
                                        statistics.toHtml(), infoDialog );
        }
    }
    if ( tooltip )
    {
        tooltip->close();
        delete tooltip;
        tooltip = nullptr;
    }
}

#ifdef SELECT_KERNELS
//slot for the order slider
void
ViolinPlot::orderSLider( int value )
{
    if ( 1 == stacker->currentIndex() )
    {
        ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
        if ( _vp != NULL )
        {
            _vp->selectOrder( static_cast<TaylorOrder>( value ) );
            _vp->update();
        }
    }
}
//slot for width_slider
void
ViolinPlot::widthSlider( int value )
{
    if ( 1 == stacker->currentIndex() )
    {
        ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
        if ( _vp != NULL )
        {
            _vp->makeWidth( value );
            _vp->update();
        }
    }
}


//functions for the selector
void
ViolinPlot::UniK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( UNIFORM );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::TriangK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( TRIANGULAR );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::EpaK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( EPANECHNIKOV );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::QuarK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( QUARTIC );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::TriwK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( TRIWEIGHT );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::TcubeK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( TRICUBE );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::GaussK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( GAUSSIAN );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::CosK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( COSINE );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::LogK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( LOGISTIC );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::SigK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( SIGMOID );
                _vp->update();
            }
        }
    }
}
void
ViolinPlot::SilvK( bool ok )
{
    if ( ok )
    {
        if ( 1 == stacker->currentIndex() )
        {
            ViolinPlot* _vp = dynamic_cast<ViolinPlot*>( stacker->currentWidget() );
            if ( _vp != NULL )
            {
                _vp->selectKernel( SILVERMAN );
                _vp->update();
            }
        }
    }
}
#endif
