/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <cassert>
#include "Compatibility.h"
#include <cstdio>
#include <fstream>
#include <QMessageBox>
#include <QFileDialog>
#include <QDialog>
#include <QVBoxLayout>
#include <QRadioButton>
#include <QCheckBox>
#include <QApplication>
#include <QFontDialog>
#include <QMenuBar>
#include <QStyle>
#include <QLabel>
#include <QScrollBar>
#include <QScrollArea>
#include <QStringList>
#include "Compatibility.h"
#ifdef HAS_QREGULAR_EXPRESSION
#include <QRegularExpression>
#define REGULAR_EXPRESSION QRegularExpression
#else
#include <QRegExp>
#define REGULAR_EXPRESSION QRegExp
#endif

#include "TreeItem.h"
#include "EditorPlugin.h"
#include "CPPSyntaxHighlighter.h"
#include "FortranSyntaxHighlighter.h"
#include "PythonSyntaxHighlighter.h"
#include "EditorConfig.h"

using namespace cubepluginapi;
using namespace editor_plugin;
using namespace std;

// OPARI_FNAME: workaround for gfortran bug, see #805
#define OPARI_FNAME ".input.prep.opari"

void
EditorPlugin::version( int& major, int& minor, int& bugfix ) const
{
    major  = 1;
    minor  = 0;
    bugfix = 0;
}

QString
EditorPlugin::name() const
{
    return "Source Code Viewer";
}

QString
EditorPlugin::getHelpText() const
{
    return tr( "Source code viewer with syntax highlighting and basic editing functionality" );
}

bool
EditorPlugin::cubeOpened( PluginServices* service )
{
    this->service = service;
    service->addSettingsHandler( this );
    selectedItem = nullptr;

    createWidgets();

    // --- create actions -----------------------------------------------------------------------
    switchSource = new QAction( QString(), editorWidget ); // select code editor font
    connect( switchSource, &QAction::triggered, this, [ this ]() {
        setSourceType( sourceType == SourceType::CallSite ? SourceType::FunctionDefinition : SourceType::CallSite );
        showSourceCode();
    } );

    copyPath = new QAction( tr( "Copy path" ), editorWidget ); // select code editor font
    connect( copyPath, &QAction::triggered, this, [ this ]() {
        QApplication::clipboard()->setText( this->source.fileName() );
    } );

    font = new QAction( tr( "Set Font..." ), editorWidget ); // select code editor font
    connect( font, SIGNAL( triggered() ), this, SLOT( onChangeFont() ) );

    textEditSaveAction = new QAction( tr( "Save" ), editorWidget );
    connect( textEditSaveAction, SIGNAL( triggered() ), this, SLOT( onSaveFile() ) );
    textEditSaveAsAction = new QAction( tr( "Save as" ), editorWidget );
    connect( textEditSaveAsAction, SIGNAL( triggered() ), this, SLOT( onSaveFileAs() ) );

    readOnlyAction = new QAction( tr( "Read only" ), editorWidget );
    readOnlyAction->setCheckable( true );
    readOnlyAction->setChecked( true );
    connect( readOnlyAction, SIGNAL( toggled( bool ) ), this, SLOT( onToggleReadOnly( bool ) ) );

    QAction* chooseEditor = new QAction( tr( "Set external editor" ), editorWidget );
    connect( chooseEditor, SIGNAL( triggered() ), this, SLOT( onChooseEditor() ) );

    findAction = new QAction( tr( "&Find" ), editorWidget );
    findAction->setShortcutContext( Qt::WidgetWithChildrenShortcut );
    findAction->setShortcut( QKeySequence( Qt::CTRL + Qt::Key_F ) );
    editorWidget->addAction( findAction );
    findAction->setToolTip( tr( "Find string" ) );
    connect( findAction, SIGNAL( triggered( bool ) ), this, SLOT( startSearch() ) );

    externalAction = new QAction( tr( "Open in &external editor" ), this );
    connect( externalAction, SIGNAL( triggered( bool ) ), this, SLOT( openExternalEditor() ) );
    externalUserAction = new QAction( tr( "" ), this );
    connect( externalUserAction, SIGNAL( triggered( bool ) ), this, SLOT( openDefinedExternalEditor() ) );
    externalUserAction->setVisible( false );

    resetUserPathAction = new QAction( tr( "&Reset user defined path" ), this );
    connect( resetUserPathAction, SIGNAL( triggered( bool ) ), this, SLOT( resetUserPath() ) );
    resetUserPathAction->setEnabled( false );
    // -------------------------------------------------------------------------------------
    connect( service, SIGNAL( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ),
             this, SLOT( contextMenuIsShown( cubepluginapi::DisplayType, cubepluginapi::TreeItem* ) ) );

    QMenu* menu = service->enablePluginMenu();
    menu->addAction( font );
    menu->addAction( textEditSaveAction );
    menu->addAction( textEditSaveAsAction );
    menu->addAction( readOnlyAction );
    menu->addAction( chooseEditor );

    QAction* separator = new QAction( "", nullptr );
    separator->setSeparator( true );
    textEdit->addToContextMenu( separator );
    textEdit->addToContextMenu( switchSource );
    textEdit->addToContextMenu( copyPath );
    textEdit->addToContextMenu( resetUserPathAction );
    separator = new QAction( "", nullptr );
    separator->setSeparator( true );
    textEdit->addToContextMenu( separator );
    textEdit->addToContextMenu( findAction );
    textEdit->addToContextMenu( externalAction );
    textEdit->addToContextMenu( externalUserAction );

    service->addTab( SYSTEM, this, OTHER_PLUGIN_TAB );

    onToggleReadOnly( readOnlyAction->isChecked() );

    tempDir = new QTemporaryDir();
    if ( !tempDir->isValid() )
    {
        cerr << "EditorPlugin:: could not create temporary directory" << endl;
    }

    return true;
}


void
EditorPlugin::contextMenuIsShown( cubepluginapi::DisplayType type, cubegui::TreeItem* item )
{
    if ( type == CALL && item != nullptr )
    {
        QAction* action = service->addContextMenuItem( CALL, tr( "Show source code" ) );
        connect( action, &QAction::triggered, this, [ this ](){
            toFront();
        } );
    }
}

void
EditorPlugin::toFront()
{
    showSourceCode();
    service->toFront( this );
}

void
EditorPlugin::createWidgets()
{
    mainWidget   = new QStackedWidget();
    editorWidget = new QWidget();
    textEdit     = new SourceCodeEditor();

    // -------------------------------------------------------------------------------------
    searchWidget = new QWidget();
    searchWidget->setVisible( false );
    findPrevButton = new QPushButton( QApplication::style()->standardIcon( QStyle::SP_ArrowLeft ), "" );
    findPrevButton->setToolTip( tr( "Find previous" ) );
    connect( findPrevButton, SIGNAL( clicked( bool ) ), this, SLOT( searchBackward() ) );
    findNextButton = new QPushButton( QApplication::style()->standardIcon( QStyle::SP_ArrowRight ), "" );
    findNextButton->setToolTip( tr( "Find next" ) );
    connect( findNextButton, SIGNAL( clicked( bool ) ), this, SLOT( searchForward() ) );
    findEdit = new QLineEdit();
    connect( findEdit, SIGNAL( textChanged( QString ) ), this, SLOT( search( const QString & ) ) );

    QHBoxLayout* buttonLayout = new QHBoxLayout();
    QPushButton* hideButton   = new QPushButton(  QApplication::style()->standardIcon( QStyle::SP_DockWidgetCloseButton ), "" );
    hideButton->setToolTip( tr( "Close search widget" ) );
    connect( hideButton, SIGNAL( clicked( bool ) ), searchWidget, SLOT( hide() ) );

    buttonLayout->setContentsMargins( 0, 0, 0, 0 );
    buttonLayout->addWidget( new QLabel( tr( "Find" ) ) );
    buttonLayout->addWidget( findEdit );
    buttonLayout->addWidget( findPrevButton );
    buttonLayout->addWidget( findNextButton );
    buttonLayout->addWidget( hideButton );
    searchWidget->setLayout( buttonLayout );
    // -------------------------------------------------------------------------------------

    QVBoxLayout* layout = new QVBoxLayout();
    layout->addWidget( textEdit );
    layout->addWidget( searchWidget );

    editorWidget->setLayout( layout );
    editorWidget->setMinimumSize( 50, 50 );
    textEdit->setMinimumSize( 50, 50 );

    // -------------------------------------------------------------------------------------
    openFile = new QPushButton( tr( "Open Source File" ) ); // to be used in getSourceFile
    connect( openFile, SIGNAL( pressed() ), this, SLOT( openFileDialog() ) );
    openFileLabel = new QLabel( tr( "file" ) );

    QWidget*     buttonWidget = new QWidget();
    QHBoxLayout* l            = new QHBoxLayout();
    buttonWidget->setLayout( l );
    openFile->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
    l->addWidget( openFile );
    l->addStretch();
    QWidget* child = new QWidget();
    layout = new QVBoxLayout();
    child->setLayout( layout );
    layout->addSpacerItem( new QSpacerItem( 0, 200, QSizePolicy::Minimum, QSizePolicy::Preferred ) );
    layout->addWidget( openFileLabel );
    layout->addWidget( buttonWidget );
    layout->setSizeConstraint( QLayout::SetMinimumSize ); // required for resizing if label content changes

    openWidget = new QScrollArea();
    openWidget->setHorizontalScrollBarPolicy( Qt::ScrollBarAsNeeded );
    openWidget->setWidget( child );

    mainWidget->addWidget( editorWidget );
    mainWidget->addWidget( openWidget );
}

void
EditorPlugin::cubeClosed()
{
    delete tempDir;
    tempDir = nullptr;
}

bool
EditorPlugin::onChooseEditor()
{
    bool ok = EditorConfig::configureEditor( editorWidget, externalEditors, externalEditor );
    if ( ok )
    {
        updateActions();
    }
    return ok;
}

void
EditorPlugin::updateActions()
{
    bool enabled = !readOnlyAction->isChecked() && !source.isEmpty();
    textEditSaveAction->setEnabled( enabled );
    textEditSaveAsAction->setEnabled( enabled );
    if ( !externalEditor.isEmpty() && externalEditors.contains( externalEditor ) )
    {
        externalUserAction->setText( tr( "Open in " ) + externalEditor );
        externalUserAction->setVisible( true );
    }
    else
    {
        externalUserAction->setVisible( false );
    }
}

void
EditorPlugin::closeSourceView()
{
    /*
       service->removeTab( this );
       delete mainWidget;
       mainWidget = 0;
     */
}

QStringList
splitCommand( QString str )
{
    return str.split( " " );
}

// open dialog to choose an external editor and open the file on success
void
EditorPlugin::openExternalEditor()
{
    setSourceInfo();
    if ( source.isEmpty() )
    {
        return;
    }
    if ( onChooseEditor() )
    {
        openDefinedExternalEditor();
    }
}

// open file in previously defined editor
void
EditorPlugin::openDefinedExternalEditor()
{
    setSourceInfo();
    if ( source.isEmpty() )
    {
        return;
    }

    QStringList commands = externalEditors.value( externalEditor );
    if ( commands.isEmpty() )
    {
        return;
    }

    commands = commands.replaceInStrings( "%LINE%", QString::number( source.startLine() ) );
    commands = commands.replaceInStrings( "%SOURCE%", source.fileName() );

    if ( !deamonStarted )
    {
        QStringList startDeamon = commands.at( 0 ).split( " " );
        if ( !startDeamon.isEmpty() )
        {
            QProcess* process    = new QProcess();
            QString   executable = startDeamon.takeFirst();
            process->start( executable, startDeamon );
            process->waitForFinished();
            if ( process->exitCode() == 0 )
            {
                deamonStarted = true;
            }
            delete process;
        }
    }
    QStringList open = commands.at( 1 ).split( " " );
    if ( !open.isEmpty() )
    {
        QProcess* process    = new QProcess();
        QString   executable = open.takeFirst();
        process->start( executable, open );
        connect( process, SIGNAL( finished( int ) ), this, SLOT( deleteProcess() ) );
    }
}

void
EditorPlugin::deleteProcess()
{
    QProcess* process = ( QProcess* )this->sender();
    process->deleteLater();
}

static QString
replacePath( const QString& path, const QString& origPath, const QString& userPath )
{
    QString replaced = path;
    if ( origPath.length() > 0 )
    {
        replaced.replace( origPath, userPath );
    }
    else
    {
        replaced = userPath + path;
    }
    return replaced;
}

void
EditorPlugin::openFileDialog()
{
    QString path;

    if ( !userPath.isEmpty() )                       // user has already inserted a new source directory
    {
        path = userPath;
    }
    else // take the path of the cube file as default directory
    {
        QString cubeFile = service->getCubeFileName();
        cubeFile.remove( "file://" );
        cubeFile.remove( "cube://" );
        path = QFileInfo( cubeFile ).absoluteDir().absolutePath();
    }

    // show a dialog to enter the path to the source file
    QString     fileName    = QFileInfo( source.originalLocation() ).fileName();
    QString     dialogTitle = "Select path to " + fileName;
    QFileDialog dialog( service->getParentWidget(), dialogTitle, path );
    dialog.setFileMode( QFileDialog::ExistingFiles );

    connect( &dialog, &QFileDialog::directoryEntered, &dialog, [ &dialog, fileName ]( const QString& dir ){
        if ( QFile( dir + "/" + fileName ).exists() ) // exit dialog if source file has been found
        {
            dialog.close();
        }
    } );
    dialog.exec();

    fileName = dialog.directory().absolutePath() + "/" + fileName;

    // check which part of the original path has to be replaced
    if ( !fileName.isEmpty() )
    {
        if ( source.originalLocation().contains( OPARI_FNAME ) )
        {
            QFileInfo origF( source.originalLocation() );
            QFileInfo userF( fileName );
            source.setOriginalLocation( origF.absolutePath() + QDir::separator() + userF.fileName() );
        }
        // the path of the source files has changed -> remember old and new path
        QStringList originalL = source.originalLocation().trimmed().split( '/' );
        QStringList userPathL = fileName.split( '/' );
        QStringList inCommon;
        while ( originalL.length() > 0 && userPathL.length() > 0 )
        {
            QString last1 = originalL.takeLast();
            QString last2 = userPathL.takeLast();
            if ( last1 != last2 )
            {
                break;
            }
            inCommon.prepend( last1 );
        }
        if ( inCommon.size() > 0 )
        {
            QString common = inCommon.join( "/" );
            origPath = source.originalLocation().trimmed();
            origPath.remove( common );
            userPath = fileName;
            userPath.remove( common );
            addPathReplacement( origPath, userPath );
        }
        else
        {
            origPath = "";
            userPath = "";
        }
        showSourceCode();
    }
}

void
EditorPlugin::addPathReplacement( const QString& orig, const QString& user )
{
    QStringList newPath;
    newPath << orig << user;
    foreach( const QStringList &path, sourcePathes )
    {
        QString origPath = path.first();
        if ( orig == origPath )
        {
            sourcePathes.removeOne( path );
            break;
        }
    }
    sourcePathes.push_front( newPath );
}

void
EditorPlugin::removePathReplacement( const QString& orig )
{
    foreach( const QStringList &path, sourcePathes )
    {
        QString origPath = path.first();
        if ( orig == origPath )
        {
            sourcePathes.removeOne( path );
            resetUserPathAction->setEnabled( false );
            break;
        }
    }
}



/**
   Checks if the given (original) source file exists. If it doesn't exit and the user has already inserted another location, then
   the filename is adapted to the new path.
 */
QString
EditorPlugin::getSourceFile( const QString& sourceFile )
{
    QString fileName = sourceFile.trimmed();

    bool isRemote = service->getCube()->get_cubename().find( "cube://" ) != std::string::npos;

    if ( isRemote ) // if remote URL, send request to read source from server
    {
        // sourcefiles from server are saved in temporary directory
        fileName = tempDir->path() + QDir::separator() + QFileInfo( fileName ).fileName();

        if ( !QFileInfo( fileName ).isReadable() )
        {
            string                     fn = sourceFile.trimmed().toStdString();
            std::vector<unsigned char> filename( fn.begin(), fn.end() );

            std::vector<unsigned char> answer = service->sendToPlugin( "FileServerPlugin", filename );

            std::string sourcef { answer.begin(), answer.end() };
            if ( sourcef.length() > 0 )
            {
                std::ofstream ofs( fileName.toStdString(), std::ofstream::out );
                for ( unsigned char c : answer )
                {
                    ofs << c;
                }
                ofs.close();
            }
        }
    }

    if ( !QFileInfo( fileName ).isReadable() )
    {
        if ( !userPath.isEmpty() ) // path replacement for this experiment exists
        {
            fileName = replacePath( fileName, origPath, userPath );
        }
        else // try path replacements from global settings
        {
            QFile file, fileLower;
            foreach( QStringList path, sourcePathes )
            {
                QString origPath = path.takeFirst();
                QString userPath = path.takeFirst();
                fileName = replacePath( sourceFile.trimmed(), origPath, userPath );

                QString fileNameLower = fileName;
                fileNameLower.replace( QString( OPARI_FNAME ) + ".F", ".f" );
                fileName.replace( OPARI_FNAME, "" );

                file.setFileName( fileName );
                fileLower.setFileName( fileNameLower );
                if ( file.exists() || fileLower.exists() )
                {
                    this->origPath = origPath;
                    this->userPath = userPath;
                    fileName       = file.exists() ? file.fileName() : fileLower.fileName();
                    break;
                }
            }
        }

        if ( source.originalLocation().contains( OPARI_FNAME ) )
        {
            QString fileNameLower = fileName;
            fileNameLower.replace( QString( OPARI_FNAME ) + ".F", ".f" );
            QFile file( fileNameLower );
            if ( file.exists() )
            {
                fileName = fileNameLower;
            }
            else
            {
                fileName.replace( OPARI_FNAME, "" );
            }
        }
    }
    QFile file( fileName );
    if ( !file.exists() || !file.open( QFile::ReadOnly | QFile::Text ) )
    {
        return "";
    }
    else   // file exists
    {
        return fileName;
    }
}

void
EditorPlugin::setSourceInfo()
{
    QString                  originalFileName, fileName;
    int                      startLine, endLine;
    cubepluginapi::TreeItem* infoItem = selectedItem;

    functionSource.invalidate();
    callSiteSource.invalidate();
    if ( !infoItem )
    {
        return;
    }

    // get function definition source info from current tree item
    infoItem->getSourceInfo( originalFileName, startLine, endLine );

    // if info is not available, search for source info in all parent items
    while ( originalFileName.isEmpty() && infoItem->getParent() )
    {
        infoItem = infoItem->getParent();
        infoItem->getSourceInfo( originalFileName, startLine, endLine );
    }

    fileName = getSourceFile( originalFileName );

    // only source of parent region found => search for current item in parent region
    if ( !fileName.isEmpty() && startLine >= 0 && infoItem != selectedItem )
    {
        QFile file( fileName );
        bool  found  = false;
        int   lineNr = 0;
        if ( file.open( QIODevice::ReadOnly ) )
        {
            QTextStream in( &file );
            while ( !in.atEnd() && ( ++lineNr < startLine ) )
            {
                in.readLine();
            }
            while ( !in.atEnd() && ( ++lineNr < endLine ) )
            {
                QString line = in.readLine();
                if ( line.contains( selectedItem->getName() ) )
                {
                    found = true;
                    lineNr--;
                    break;
                }
            }
            file.close();
        }
        if ( found )
        {
            startLine = lineNr;
            endLine   = lineNr;
        }
    }
    if ( !originalFileName.isEmpty() )
    {
        functionSource = SourceInfo( originalFileName, fileName, startLine, endLine );
    }

    // get call site source information for call tree items
    if ( selectedItem->getTreeType() == DEFAULTCALLTREE )
    {
        cube::Cnode* cnode = static_cast<cube::Cnode*>( selectedItem->getCubeObject() );
        if ( cnode )
        {
            originalFileName = QString::fromStdString( cnode->get_mod() );
            if ( !originalFileName.isEmpty() )
            {
                fileName       = getSourceFile( originalFileName );
                callSiteSource = SourceInfo( originalFileName, fileName, cnode->get_line() );
            }
        }
    }

    SourceType displayedType = SourceType::FunctionDefinition;
    if ( sourceType == SourceType::CallSite && !callSiteSource.originalLocation().isEmpty() )
    {
        source        = callSiteSource;
        displayedType = SourceType::CallSite;
    }
    else
    {
        source = functionSource;
    }

    if ( source.isEmpty() )
    {
        QString fileName = source.originalLocation();
        // file doesn't exist -> show message in main widget and add button to call a file dialog
        openFile->setVisible( true );
        QString message;
        if ( fileName.isEmpty() )
        {
            message = tr( "No source information available" );
            openFile->setVisible( false );
        }
        else
        {
            message = tr( "File \"%1\" cannot be opened." ).arg( fileName );
            message.append( tr( "\n\nDo you want to open another file?" ) );
        }
        openFileLabel->setText( message );
        mainWidget->setCurrentIndex( 1 );
        mainWidget->setToolTip( "" );
    }
    else // source file exists
    {
        QString fileName = source.fileName();
        service->setMessage( tr( "Source file: " ) + fileName );
        mainWidget->setCurrentIndex( 0 );
        resetUserPathAction->setEnabled( true );

        QString tooltip = tr( "Source: " );
        tooltip += displayedType == SourceType::CallSite ? tr( "call site" ) : tr( "function definition" );
        tooltip += tr( " of \"" ) + selectedItem->getName() + "\"";
        if ( !callSiteSource.originalLocation().isEmpty() ) // call site info is available
        {
            tooltip += tr( "\nUse context menu to switch between call site and function definition" );
            switchSource->setEnabled( true );
        }
        else
        {
            switchSource->setEnabled( false );
        }
        QString text = ( displayedType == SourceType::CallSite ) ? tr( "Show function definition source code" ) : tr( "Show call site source code" );
        switchSource->setText( text );
        tooltip += "\n" + fileName;
        tooltip += userPath.isEmpty() ? "" : tr( "\n (original location: " ) + origPath + ")";

        mainWidget->setToolTip( tooltip );
    }
}

static bool
regExpMatches( REGULAR_EXPRESSION regExp, const QString& source )
{
#ifdef HAS_QREGULAR_EXPRESSION
    return regExp.match( source ).hasMatch();
#else
    return regExp.indexIn( source ) != -1;
#endif
}

void
EditorPlugin::showSourceCode()
{
    setSourceInfo();
    if ( source.isEmpty() )
    {
        return;
    }

    QFile file( source.fileName() );
    file.open( QIODevice::ReadOnly );
    QTextStream in( &file );
    textEdit->setText( in.readAll() );

    textEdit->setFont( fontSourceCode );
    textEdit->markRegion( source.startLine(), source.endLine() );

    // todo: line numbers

    REGULAR_EXPRESSION fortranSourceCode( "\\.[fF][:digit:]{0,2}$" );
    REGULAR_EXPRESSION pythonSourceCode( "\\.py$" );
    // REGULAR_EXPRESSION cSourceCode( "\\.c|h$" ); c/c++ is default
    // REGULAR_EXPRESSION cppSourceCode( "\\.cpp|hpp$" );

    enum { Cpp, Fortran, Python } language = Cpp;

    if ( regExpMatches( fortranSourceCode, source.fileName() ) )
    {
        language = Fortran;
    }
    else if ( regExpMatches( pythonSourceCode, source.fileName() ) )
    {
        language = Python;
    }

    if ( language == Cpp )
    {
        new CPPSyntaxHighlighter( textEdit->document() );
    }
    else if ( language == Fortran )
    {
        new FortranSyntaxHighlighter( textEdit->document() );
    }
    else if ( language == Python )
    {
        new PythonSyntaxHighlighter( textEdit->document() );
    }
}
// end of sourceCode()

// Selection of another font in source code dialog.
void
EditorPlugin::onChangeFont()
{
    QFont defaultFont = editorWidget ? textEdit->font() : QTextEdit().font();
    fontSourceCode = QFontDialog::getFont( 0, defaultFont );
    if ( editorWidget )
    {
        textEdit->setFont( fontSourceCode );
    }
}

// this slot is called by the editor for source code;
// it stores the eventually modified source code
//
void
EditorPlugin::onSaveFile()
{
    // "fileName" stores the name for the last source code opening/storing
    QFile file( source.fileName() );

    // if the file cannot be opened for writing
    // display a warning message and return without storing
    if ( !file.open( QFile::WriteOnly | QFile::Text ) )
    {
        QString message = tr( "Cannot write file %1:\n%2." ).arg( source.fileName() ).arg( file.errorString() );
        service->setMessage( message, Error );
        return;
    }

    {
        // write source code into the opened file
        QTextStream out( &file );
        QApplication::setOverrideCursor( Qt::WaitCursor );
        out << textEdit->toPlainText();
        QApplication::restoreOverrideCursor();
        out.flush();
    }
}


// this slot is called by the editor opened for source code;
// it stores the eventually modified source code under a new name
//
void
EditorPlugin::onSaveFileAs()
{
    // get the new file name by user input
    QString tmpName = QFileDialog::getSaveFileName( service->getParentWidget() );
    // tmpName can be empty, if the user canceled the input;
    // in this case just return without storing
    if ( tmpName.isEmpty() )
    {
        return;
    }
    // and call the slot for storing source code
    onSaveFile();
    // update the window title of the dialog widget containing the text editor
    editorWidget->setWindowTitle( tmpName ); // todo set window title if editor widget is detached
}

void
EditorPlugin::onToggleReadOnly( bool checked )
{
    if ( editorWidget )
    {
        textEdit->setReadOnly( checked );
    }
    updateActions();
}

void
EditorPlugin::search( const QString& text )
{
    searchText = text;
    searchBackward();
    searchForward();
}

void
EditorPlugin::searchForward()
{
    textEdit->find( searchText );
}

void
EditorPlugin::searchBackward()
{
    textEdit->find( searchText, QTextDocument::FindBackward );
}

// -------------------------------------------------
// implementation of settings handler interface
// -------------------------------------------------
void
EditorPlugin::loadExperimentSettings( QSettings& settings )
{
    origPath = settings.value( "origPath", "" ).toString();
    userPath = settings.value( "userPath", "" ).toString();
}

void
EditorPlugin::saveExperimentSettings( QSettings& settings )
{
    settings.setValue( "origPath", origPath );
    settings.setValue( "userPath", userPath );
}

void
EditorPlugin::loadGlobalSettings( QSettings& settings )
{
    // --- define default values for external editors
    externalEditors.clear();
    QStringList kate;
    kate << "" << "kate --line %LINE% %SOURCE%";
    externalEditors.insert( "kate", kate );
    QStringList emacs;
    emacs << "emacsclient -c -n --alternate-editor=" << "emacsclient -n +%LINE% %SOURCE%";
    externalEditors.insert( "emacs", emacs );
    QStringList gedit;
    gedit << "" << "gedit +%LINE% %SOURCE%";
    externalEditors.insert( "gedit", gedit );
    // todo emacs does not open new window, once it is closed, tries to open in terminal
    // --- end default values for external editors

    int size = settings.beginReadArray( "ExternalEditors" );
    for ( int i = 0; i < size; ++i )
    {
        settings.setArrayIndex( i );
        QStringList list   = settings.value( "editor" ).toString().split( "," );
        QString     editor = list.takeFirst();
        externalEditors.insert( editor, list );
    }
    settings.endArray();
    externalEditor = settings.value( "DefaultEditor", "" ).toString();

    sourcePathes.clear();
    size = settings.beginReadArray( "PathReplacement" );
    for ( int i = 0; i < size; ++i )
    {
        settings.setArrayIndex( i );
        QStringList path;
        path << settings.value( "origPath" ).toString() << settings.value( "userPath" ).toString();
        sourcePathes.append( path );
    }
    settings.endArray();
}

void
EditorPlugin::saveGlobalSettings( QSettings& settings )
{
    settings.beginWriteArray( "ExternalEditors" );
    int idx = 0;
    foreach( QString editor, externalEditors.keys() )
    {
        QStringList list = externalEditors.value( editor );
        list.prepend( editor );
        QString value = list.join( "," );
        settings.setArrayIndex( idx++ );
        settings.setValue( "editor", value );
    }
    settings.endArray();
    settings.setValue( "DefaultEditor", externalEditor );

    settings.beginWriteArray( "PathReplacement" );
    idx = 0;
    foreach( QStringList path, sourcePathes )
    {
        settings.setArrayIndex( idx++ );
        settings.setValue( "origPath", path.takeFirst() );
        settings.setValue( "userPath", path.takeFirst() );
        if ( idx > 10 )
        {
            break;             // only save latest 10 entries
        }
    }
    settings.endArray();
}

// -------------------------------------------------
// implementation of tab interface
// -------------------------------------------------

void
EditorPlugin::detachEvent( QMainWindow* window, bool isDetached )
{
    if ( isDetached )
    {
        QMenuBar* bar     = new QMenuBar( editorWidget );
        QMenu*    file    = bar->addMenu( tr( "&File" ) );
        QMenu*    display = bar->addMenu( tr( "&Display" ) );
        display->addAction( font );
        file->addAction( textEditSaveAction );
        file->addAction( textEditSaveAsAction );
        file->addAction( findAction );
        file->addAction( externalAction );
        file->addAction( readOnlyAction );

        window->layout()->setMenuBar( bar );
    }
}

QWidget*
EditorPlugin::widget()
{
    return mainWidget;
}

QString
EditorPlugin::label() const
{
    return tr( "Source" );
}

QIcon
EditorPlugin::icon() const
{
    return QIcon( ":/images/source.png" );
}

void
EditorPlugin::setActive( bool active )
{
    if ( active )
    {
        service->connect( service, SIGNAL( treeItemIsSelected( cubepluginapi::TreeItem* ) ),
                          this, SLOT( treeItemSelected( cubepluginapi::TreeItem* ) ) );
        TreeItem* item = service->getSelection( CALL );
        treeItemSelected( item );
    }
    else
    {
        service->disconnect( service, SIGNAL( treeItemIsSelected( cubepluginapi::TreeItem* ) ),
                             this, SLOT( treeItemSelected( cubepluginapi::TreeItem* ) ) );
    }
}

void
EditorPlugin::treeItemSelected( cubepluginapi::TreeItem* item )
{
    if ( item->getDisplayType() != CALL )
    {
        return;
    }

    selectedItem = item;

    showSourceCode();
    updateActions();
}

void
EditorPlugin::startSearch()
{
    QTextCursor   cursor( textEdit->textCursor() );
    const QString selected = cursor.selectedText();
    if ( selected.size() > 0 )
    {
        findEdit->setText( selected );
    }
    searchWidget->setVisible( true );
}

void
EditorPlugin::resetUserPath()
{
    removePathReplacement( origPath );
    userPath = "";
    resetUserPathAction->setEnabled( false );
    showSourceCode();
}

void
EditorPlugin::setSourceType( SourceType newSourceType )
{
    sourceType = newSourceType;
}

// ============================================================================================================

SourceInfo::SourceInfo()
{
}

SourceInfo::SourceInfo( const QString& orig, const QString& file, int start, int end ) :
    originalLocation_( orig ), fileName_( file ), startLine_( start ), endLine_( end )
{
}

void
SourceInfo::invalidate()
{
    originalLocation_ = "";
    fileName_         = "";
    startLine_        = -1;
    endLine_          = -1;
}

bool
SourceInfo::isEmpty() const
{
    return fileName_.isEmpty();
}

const QString&
SourceInfo::originalLocation() const
{
    return originalLocation_;
}

const QString&
SourceInfo::fileName() const
{
    return fileName_;
}

int
SourceInfo::startLine() const
{
    return startLine_;
}

int
SourceInfo::endLine() const
{
    return endLine_;
}

void
SourceInfo::setOriginalLocation( const QString& newOriginalLocation )
{
    originalLocation_ = newOriginalLocation;
}
