/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2023-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <QComboBox>
#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QTableWidget>
#include <QTableWidgetItem>
#include <QTextEdit>
#include <QtPlugin>
#include <QStyleFactory>
#include <string>
#include <iostream>
#include "cubelib-version.h"

#include "CubeIoProxy.h"
#include "CubeNetworkProxy.h"

#include "POPAdvisorPlugin.h"
#include "CubePOPAdvisorWidget.h"
#include "Globals.h"
#include "CubeTypes.h"
#include "PluginServices.h"
#include "CallTree.h"
#include "DefaultCallTree.h"
#include "FlatTree.h"
#include "CubeIdIndexMap.h"

#include "json.hpp"
#include "POPServerRequest.h"
#include "POPCalculation.h"

/**
 * If the plugin doesn't load, start cube with -verbose to get detailed information.
 * If the error message "Plugin verification data mismatch" is printed, check if the
 * same compiler and Qt version have been used.
 */

using namespace cubepluginapi;
using namespace popadvisor_client;
// using json = nlohmann::json;

cubepluginapi::PluginServices* advisor_services;

POPAdvisorPlugin::POPAdvisorPlugin()
{
    // The constructor should be empty, use cubeOpened to initialize. If Qt widgets or
    // signals/slots are used in constructor, they have to be deleted in destructor,
    // otherwise cube may crash if the plugin is unloaded.
}

bool
POPAdvisorPlugin::cubeOpened( PluginServices* service )
{
    this->service      = service;
    advisor_services   = service;
    pop_advisor_widget = new popadvisor_client::CubePOPAdvisorWidget( this );

    isNetworkProxy = dynamic_cast< cube::CubeNetworkProxy* >( service->getCube() ) != nullptr;


    service->addTab( cubegui::SYSTEM, this, cubegui::OTHER_PLUGIN_TAB );
    future = service->createFuture( this );
    if ( !isNetworkProxy ) // alternative: #ifndef __EMSCRIPTEN__
    {
        setActive( true ); // start calculation even before plugin gets active
    }

    connect( service, &PluginServices::treeItemIsSelected,
             this,    &POPAdvisorPlugin::treeItemIsSelected );

    return true;           // initialisation is ok => plugin should be shown
}

void
POPAdvisorPlugin::cubeClosed()
{
    analysisList.clear();
    service->disconnect( future, SIGNAL( calculationFinished() ) );
    delete pop_advisor_widget;
}

/** set a version number, the plugin with the highest version number will be loaded */
void
POPAdvisorPlugin::version( int& major, int& minor, int& bugfix ) const
{
    major  = 1;
    minor  = 0;
    bugfix = 0;
}

/** unique plugin name */
QString
POPAdvisorPlugin::name() const
{
    return "POPAdvisorPlugin";
}

QString
POPAdvisorPlugin::getHelpText() const
{
    return "Plugin to calculate various POP efficiencies";
}

/** widget that will be placed into the tab */
QWidget*
POPAdvisorPlugin::widget()
{
    return pop_advisor_widget;
}

/** tab label */
QString
POPAdvisorPlugin::label() const
{
    return "POP Advisor";
}

/** activate tab */
void
POPAdvisorPlugin::setActive( bool active )
{
    if ( active )
    {
        if ( analysisList.empty() )
        {
            Task* task = new Task([ this ] { testAvailableAnalyses();
                                  } );
            connect( future, &Future::calculationFinished, this, &POPAdvisorPlugin::initialisationFinished );
            future->addCalculation( task );
            future->startCalculation();
        }
    }
}

/** slot, which is called if a tree item is selected */
void
POPAdvisorPlugin::communicateWithServerPlugin()
{
    popserver_request::POPServerRequest pop_request;
    pop_request.pop_analysis =  pop_advisor_widget->currentAnalysis();
    pop_request.operation    = popserver_request::CALCULATE;

    /** Returns selected items in the given active call tree */
    cube::list_of_cnodes lcnodes;
    QList<TreeItem*>     selected_items = service->getSelections( service->getActiveTree( CALL )->getType() );
    if ( selected_items.size() == 0 ) // no selection?  do nothing
    {
        return;
    }
    QStringList          to_calculate_cnodes;
    cube::list_of_cnodes _cnodes;
    if ( service->getActiveTree( CALL )->getTreeType() == DEFAULTCALLTREE || service->getActiveTree( CALL )->getTreeType() == TASKTREE ) // calltree is activbe with the selection
    {
        cubegui::DefaultCallTree* _tree = dynamic_cast<cubegui::DefaultCallTree*>( selected_items[ 0 ]->getTree() );                     // reference to 0 is ok as size got cheecked before
        _cnodes = _tree->getNodes( selected_items );
    }
    if ( service->getActiveTree( CALL )->getTreeType() == FLATTREE )                                   // flattree is active
    {
        cubegui::FlatTree* _tree = dynamic_cast<cubegui::FlatTree*>( selected_items[ 0 ]->getTree() ); // reference to 0 is ok as size got cheecked before
        _cnodes = _tree->getNodes( selected_items );
    }
    lcnodes.insert( lcnodes.end(), _cnodes.begin(), _cnodes.end() );

    std::for_each( lcnodes.begin(), lcnodes.end(), [ &pop_request, &to_calculate_cnodes ]( const cube::cnode_pair& c_pair ){
        pop_request.cnodes.push_back(  c_pair.first->get_id() );
        pop_request.state.push_back(  c_pair.second );
        to_calculate_cnodes.append( QString::fromStdString( c_pair.first->get_callee()->get_name() ) );
    } );



    json              j3 = pop_request;
    std::stringstream sstr;
    sstr << j3;
    // serialize it to MessagePack
    std::string json_request = sstr.str();

    const std::vector<unsigned char> messageToServer( json_request.begin(), json_request.end() );

    future->addCalculation( new ParallelCalculation( name().toStdString(), service->getCube(),
                                                     messageToServer, &server_answer ) ); // calculation data
    future->startCalculation( true );
    QString cnodes_to_calculate = to_calculate_cnodes.join( ", \n" );
    pop_advisor_widget->calculationStart( cnodes_to_calculate );
}




void
POPAdvisorPlugin::calculationFinished()
{
    std::lock_guard<std::mutex> lockGuard( updateMutex );
    std::string                 sanswer { server_answer.begin(), server_answer.end() };

    if ( sanswer.size() > 0 )
    {
        try
        {
            json j          = json::parse( sanswer );
            auto pop_answer = j.template get<popserver_request::POPServerAnswer>();
            pop_advisor_widget->enableControls( true );
            pop_advisor_widget->calculationResult( pop_answer );
        }
        catch ( json::exception& )
        {
            QString message = QString( "Server respond errorneous " ).append( QString::fromStdString( sanswer ) );
            service->setMessage( message );
            return;
        }
    }
    if ( widget()->isVisible() ) // only show message in status line if plugin is active
    {
        QString message = sanswer.size() > 0 ? "Server reply ok" : "ServerPlugin doesn't answer";
        service->setMessage( message );
    }
}


/**
   fills analysisList and analysisListResults
   these lists are required in calculation_finished() to update the GUI in GUI-thread
 */
void
POPAdvisorPlugin::testAvailableAnalyses()
{
    popserver_request::POPServerRequest pop_request;

    std::vector<std::pair<std::string, popcalculation::POP_ANALYSIS> > _list =
    {
        {
            "MPI",
            popcalculation::PURE_MPI
        },
        {
            "HYBRID",
            popcalculation::HYBRID_MULT
        },
        {
            "BSC",
            popcalculation::HYBRID_BSC
        }
    };
    if ( cubegui::Globals::optionIsSet( cubegui::ExpertMode ) )
    {
        _list.push_back( std::make_pair(  "HYBRID ADD", popcalculation::HYBRID_ADD )  );
    }
    analysisList = _list;  // required in calculation_finished

    POPAdvisorPlugin* plugin = this;

    std::for_each( _list.begin(), _list.end(), [ &pop_request, &plugin, this ]
                       ( const std::pair< std::string, popcalculation::POP_ANALYSIS>& pop_analysis_ )
    {
        pop_request.pop_analysis = pop_analysis_.second;
        pop_request.operation = popserver_request::TEST;

        json j3 = pop_request;
        std::stringstream sstr;
        sstr << j3;
        // serialize it to MessagePack
        std::string json_request = sstr.str();
        const std::vector<unsigned char> messageToServer( json_request.begin(), json_request.end() );

        analysisListResults.emplace_back();
        ParallelCalculation _p( name().toStdString(), service->getCube(),
                                messageToServer, &analysisListResults.back() );
        _p.calculate();
    } );
}


void
POPAdvisorPlugin::initialisationFinished()
{
    disconnect( future, &Future::calculationFinished,
                this, &POPAdvisorPlugin::initialisationFinished );
    for ( size_t i = 0; i < analysisList.size(); ++i )
    {
        auto& listElem = analysisList[ i ];
        auto& answer   = analysisListResults[ i ];

        auto        popAnalysis = listElem.second;
        std::string label       = listElem.first;

        std::string sanswer { answer.begin(), answer.end() };

        if ( sanswer.size() > 0 )
        {
            json j          = json::parse( sanswer );
            auto pop_answer = j.template get<popserver_request::POPServerAnswer>();
            if ( pop_answer.analysis_active == popserver_request::OK )
            {
                pop_advisor_widget->enableAnalysis( popAnalysis, label );
            }
            service->setMessage( QString::fromStdString( pop_answer.analysis_message ) );
        }
    }
    connect( future, SIGNAL( calculationFinished() ), this, SLOT( calculationFinished() ) );

    communicateWithServerPlugin();
}


void
POPAdvisorPlugin::treeItemIsSelected( cubepluginapi::TreeItem* item )
{
    // mark results as outdated if a different callpath is selected
    if ( item->getDisplayType() == CALL )
    {
        pop_advisor_widget->enableControls( false );
    }
}




/* one task, that can be calculated in parallel */
void
ParallelCalculation::calculate()
{
    cube::CubeIoProxy*      _io_proxy      = dynamic_cast< cube::CubeIoProxy* >( proxy_ );
    cube::CubeNetworkProxy* _network_proxy = dynamic_cast< cube::CubeNetworkProxy* >( proxy_ );
    if ( _network_proxy )
    {
        *result_ = _network_proxy->sendToPlugin( plugins_name_, request_ );
    }
    else
    if ( _io_proxy )
    {
        *result_ = popcalculation::POPCalculation::calculate( request_, _io_proxy );
    }
    else
    {
        throw cube::RuntimeError( "internal error. Unknown type of CubeProxy." );
    }
}
