/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2024-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <algorithm>
#include <QClipboard>
#include <QGroupBox>
#include <QPushButton>
#include <QComboBox>
#include <QLabel>
#include <QVBoxLayout>
#include <QGridLayout>
#include <QScrollArea>
#include <QWidget>
#include "Globals.h"
#include "CubeTypes.h"
#include "PluginServices.h"
#include "CallTree.h"
#include "DefaultCallTree.h"
#include "FlatTree.h"
#include "CubeIdIndexMap.h"

#include "POPCalculation.h"
#include "POPAdvisorPlugin.h"
#include "CubePOPAdvisorWidget.h"
#include "json.hpp"



using namespace popadvisor_client;
using json = nlohmann::json;

extern cubepluginapi::PluginServices* advisor_services;


CubePOPAdvisorWidget::CubePOPAdvisorWidget( POPAdvisorPlugin* caller,  QWidget* parent ) : QWidget( parent ), slotprocessor( caller )
{
    QVBoxLayout* main_layout = new QVBoxLayout;

    QGridLayout* header_layout = new QGridLayout;
    setLayout( main_layout );

    QGroupBox*   groupBox_title = new QGroupBox( tr( "Calculated for " ) );
    QHBoxLayout* title_layout   = new QHBoxLayout;
    // QHBoxLayout * callpath_layout = new QHBoxLayout();
    callpaths = new QLabel();
    show_more = new QToolButton();

    show_more->hide();
    title_layout->addWidget( callpaths );
    title_layout->addWidget( show_more );

    groupBox_title->setLayout( title_layout );

    header_layout->addWidget( groupBox_title, 1, 1 );
    header_layout->setColumnStretch( 0, 0 );
    header_layout->setColumnStretch( 1, 1 );
    header_layout->setRowStretch( 1, 1 );

    analyses = new QComboBox();
#if QT_VERSION < QT_VERSION_CHECK( 6, 0, 0 )
    connect( analyses, SIGNAL( currentIndexChanged( int ) ), this, SLOT( enableControls( int ) ) );
#else
    connect( analyses, &QComboBox::currentIndexChanged, this, [ this ]() {
        enableControls( false );
    } );
#endif

    calculate_button = new QPushButton( tr( "calculate" ) );
    calculate_button->setToolTip( tr( "Calculate POP Efficiencies for selected call path" ) );
    connect( calculate_button, &QPushButton::clicked, slotprocessor, &popadvisor_client::POPAdvisorPlugin::communicateWithServerPlugin );
    copy_button = new QPushButton( tr( "copy" ) );
    copy_button->setToolTip( tr( "Copy all results to clipboard" ) );
    connect( copy_button, &QPushButton::clicked, this, &popadvisor_client::CubePOPAdvisorWidget::copyToClipboard );

    QVBoxLayout* controls_layout = new QVBoxLayout();
    header_layout->addLayout( controls_layout, 1, 2 );
    controls_layout->addWidget( analyses );
    controls_layout->addWidget( calculate_button );
    controls_layout->addWidget( copy_button );
    main_layout->addLayout( header_layout );

    QScrollArea* scroll_area = new QScrollArea();
    main_layout->addWidget( scroll_area );


    QGridLayout* eff_layout    = new QGridLayout();
    QWidget*     scroll_widget =  new QWidget();

    scroll_area->setWidget( scroll_widget );
    scroll_widget->setLayout( eff_layout );



    QGroupBox*   groupBox_pop = new QGroupBox( tr( "POP Efficiences" ) );
    QVBoxLayout* vbox_pop     = new QVBoxLayout;
    groupBox_pop->setLayout( vbox_pop );
    QGroupBox*   groupBox_pop_values = new QGroupBox( tr( "Values" ) );
    QVBoxLayout* vbox_pop_values     = new QVBoxLayout;
    groupBox_pop_values->setLayout( vbox_pop_values );

    for ( int i = 0; i < 15; i++ )
    {
        QLabel* _pop_title = new QLabel();
        QLabel* _pop_value = new QLabel();
        vbox_pop->addWidget( _pop_title );
        vbox_pop_values->addWidget( _pop_value );
        pop_labels.push_back( _pop_title );
        pop_values.push_back( _pop_value );
        _pop_title->hide();
        _pop_value->hide();
    }


    eff_layout->addWidget( groupBox_pop, 1, 1 );
    eff_layout->addWidget( groupBox_pop_values, 1, 2 );


    QGroupBox*   groupBox_gpu = new QGroupBox( tr( "GPU Efficiences" ) );
    QVBoxLayout* vbox_gpu     = new QVBoxLayout;
    groupBox_gpu->setLayout( vbox_gpu );
    QGroupBox*   groupBox_gpu_values = new QGroupBox( tr( "Values" ) );
    QVBoxLayout* vbox_gpu_values     = new QVBoxLayout;
    groupBox_gpu_values->setLayout( vbox_gpu_values );

    for ( int i = 0; i < 3; i++ )
    {
        QLabel* _gpu_title = new QLabel();
        QLabel* _gpu_value = new QLabel();
        vbox_gpu->addWidget( _gpu_title );
        vbox_gpu_values->addWidget( _gpu_value );
        gpu_labels.push_back( _gpu_title );
        gpu_values.push_back( _gpu_value );
        _gpu_title->hide();
        _gpu_value->hide();
    }

    eff_layout->addWidget( groupBox_gpu, 2, 1 );
    eff_layout->addWidget( groupBox_gpu_values, 2, 2 );


    QGroupBox*   groupBox_io = new QGroupBox( tr( "IO Efficiences" ) );
    QVBoxLayout* vbox_io     = new QVBoxLayout;
    groupBox_io->setLayout( vbox_io );
    QGroupBox*   groupBox_io_values = new QGroupBox( tr( "Values" ) );
    QVBoxLayout* vbox_io_values     = new QVBoxLayout;
    groupBox_io_values->setLayout( vbox_io_values );

    for ( int i = 0; i < 3; i++ )
    {
        QLabel* _io_title = new QLabel();
        QLabel* _io_value = new QLabel();
        vbox_io->addWidget( _io_title );
        vbox_io_values->addWidget( _io_value );
        io_labels.push_back( _io_title );
        io_values.push_back( _io_value );
        _io_title->hide();
        _io_value->hide();
    }

    eff_layout->addWidget( groupBox_io, 3, 1 );
    eff_layout->addWidget( groupBox_io_values, 3, 2 );


    QGroupBox*   groupBox_add = new QGroupBox( tr( "Additional Efficiences" ) );
    QVBoxLayout* vbox_add     = new QVBoxLayout;
    groupBox_add->setLayout( vbox_add );
    QGroupBox*   groupBox_add_values = new QGroupBox( tr( "Values" ) );
    QVBoxLayout* vbox_add_values     = new QVBoxLayout;
    groupBox_add_values->setLayout( vbox_add_values );

    for ( int i = 0; i < 5; i++ )
    {
        QLabel* _add_title = new QLabel();
        QLabel* _add_value = new QLabel();
        vbox_add->addWidget( _add_title );
        vbox_add_values->addWidget( _add_value );
        additional_labels.push_back( _add_title );
        additional_values.push_back( _add_value );
        _add_title->hide();
        _add_value->hide();
    }

    eff_layout->addWidget( groupBox_add, 4, 1 );
    eff_layout->addWidget( groupBox_add_values, 4, 2 );

    QGroupBox*   groupBox_control = new QGroupBox( tr( "Control " ) );
    QVBoxLayout* vbox_control     = new QVBoxLayout;
    groupBox_control->setLayout( vbox_control );
    QGroupBox*   groupBox_control_values = new QGroupBox( tr( "Values" ) );
    QVBoxLayout* vbox_control_values     = new QVBoxLayout;
    groupBox_control_values->setLayout( vbox_control_values );

    for ( int i = 0; i < 3; i++ )
    {
        QLabel* _control_title = new QLabel();
        QLabel* _control_value = new QLabel();
        vbox_control->addWidget( _control_title );
        vbox_control_values->addWidget( _control_value );
        control_labels.push_back( _control_title );
        control_values.push_back( _control_value );
        _control_title->hide();
        _control_value->hide();
    }

    eff_layout->addWidget( groupBox_control, 5, 1 );
    eff_layout->addWidget( groupBox_control_values, 5, 2 );

    eff_layout->setRowStretch( 6, 1 );

    eff_layout->setColumnStretch( 1, 0 );
    eff_layout->setColumnStretch( 2, 1 );
    eff_layout->setSizeConstraint( QLayout::SetMinimumSize  );
}

void
CubePOPAdvisorWidget::enableAnalysis( const popcalculation::POP_ANALYSIS p_analysis, const std::string& p_name )
{
    analyses->addItem( QString::fromStdString( p_name ), p_analysis );
}




static
QString
shortName( QString name, QToolButton& _button )
{
    const int   max_name_size = 55;
    QStringList name_parts    = name.split( "\n" );
    if ( name_parts.size() > 1 )
    {
        _button.show();
        _button.setToolTip( "Calculate for cnodes: \n" + name );
        QMenu* _menu = new QMenu();
        _menu->setToolTipsVisible( true );
        foreach( QString name_part, name_parts )
        {
            QAction* name_in_menu = new QAction( ( name_part.length() > max_name_size ) ? name_part.left( max_name_size ) + QString( "..." ) : name_part );
            name_in_menu->setToolTip( name_part );
            _menu->addAction( name_in_menu );
        }
        _button.setMenu( _menu );
        _button.setPopupMode( QToolButton::InstantPopup );
        QString leading_part_of_name = name.section( "\n", 0, 1 );
        return ( ( leading_part_of_name.length() > max_name_size ) ? leading_part_of_name.left( max_name_size ) : leading_part_of_name ) + QString( "..." );
    }
    return ( name.length() > max_name_size ) ? name.left( max_name_size ) + QString( "..." ) : name;
}


void
CubePOPAdvisorWidget::setCallPathToCalculate( QString& callpaths_ )
{
    show_more->hide();
    callpaths->setText( shortName( callpaths_, *show_more ) );
    callpaths->show();
}

void
CubePOPAdvisorWidget::recalculate()
{
}



void
CubePOPAdvisorWidget::calculationStart( QString& cnodes_to_calculate )
{
    std::for_each( pop_values.begin(), pop_values.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    std::for_each( gpu_values.begin(), gpu_values.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    std::for_each( io_values.begin(), io_values.end(),  [ ](  QLabel* label ){
        label->hide();
    } );
    std::for_each( additional_values.begin(), additional_values.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    std::for_each( control_values.begin(), control_values.end(),  [ ]( QLabel* label ){
        label->hide();
    } );

    std::for_each( pop_labels.begin(), pop_labels.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    std::for_each( gpu_labels.begin(), gpu_labels.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    std::for_each( io_labels.begin(), io_labels.end(),  [ ](  QLabel* label ){
        label->hide();
    } );
    std::for_each( additional_labels.begin(), additional_labels.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    std::for_each( control_labels.begin(), control_labels.end(),  [ ]( QLabel* label ){
        label->hide();
    } );
    setCallPathToCalculate( cnodes_to_calculate );
}


void
CubePOPAdvisorWidget::calculationResult( popserver_request::POPServerAnswer& pop_answer )
{
    // find names of the corrected cnodes
    QStringList _final_cnodes;
    size_t      pop_size =  pop_answer.pop_metric_names.size();
    for ( size_t i = 0; i < pop_answer.cnodes.size(); ++i )
    {
        std::vector<cube::Cnode*> _cnodes   = advisor_services->getCube()->getCnodes();
        uint64_t                  _cnode_id = pop_answer.cnodes[ i ];
        auto                      iter      = std::find_if( _cnodes.begin(), _cnodes.end(), [ &_cnode_id ]( const cube::Cnode* _c ){
            return _c->get_id() == _cnode_id;
        } );
        if ( iter == _cnodes.end() )
        {
            continue;
        }
        _final_cnodes << QString::fromStdString( ( *iter )->get_callee()->get_name() ).append( "(id=" ).append( QString::number( ( *iter )->get_id() ) ).append( " )[ " ).append( QString::fromStdString( pop_answer.state[ i ] == 0 ? "I" : "E" ) ).append( " ]" );
    }


    QString _s = _final_cnodes.join( "\n\t\t" );
    advisor_services->debug( "POP Advisor" ) << "\n\tCalculate POP Efficiencies for : " << _s << Qt::endl;
    callpaths->setToolTip( "Calculate for cnodes: \n\t" + _s );

    QString _copy_s = _final_cnodes.join( "\n" );
    text_to_copy = "POP Analysis: \t";
    text_to_copy.append( analyses->currentText() ).append( "\n" );
    text_to_copy.append( "Calculate for : \n" ).append( _copy_s ).append( "\n\n\n" );

    for ( size_t i = 0; i < pop_size; ++i )
    {
        size_t ii = pop_size - i - 1;
        double _v = pop_answer.pop_values[ i ];
        pop_values[ ii ]->setText( ( _v != -1 ) ? QString::number( _v ) : tr( "not available" ) );
        pop_labels[ ii ]->setText( QString::fromStdString( pop_answer.pop_metric_names[ i ] ) );
        pop_labels[ ii ]->show();
        pop_values[ ii ]->show();
        pop_labels[ ii ]->setEnabled( _v != -1 );
        pop_values[ ii ]->setEnabled( _v != -1 );
        if ( _v != -1 )
        {
            pop_labels[ ii ]->setToolTip( QString::fromStdString( pop_answer.pop_metrics_helps[ i ]  ) );
            pop_values[ ii ]->setToolTip( QString::fromStdString( pop_answer.pop_metrics_helps[ i ]  ) );
        }
        else
        {
            pop_labels[ ii ]->setToolTip(  tr( "POP Efficiency cannot be calculated" ) );
            pop_values[ ii ]->setToolTip( tr( "POP Efficiency cannot be calculated" )  );
        }
    }
    size_t gpu_size =  pop_answer.gpu_metric_names.size();
    for ( size_t i = 0; i < gpu_size; ++i )
    {
        size_t ii = gpu_size - i  - 1;
        double _v = pop_answer.gpu_values[ i ];
        gpu_values[ ii ]->setText( ( _v != -1 ) ? QString::number( _v ) : tr( "not available" ) );
        gpu_labels[ ii ]->setText( QString::fromStdString( pop_answer.gpu_metric_names[ i ] ) );

        gpu_labels[ ii ]->show();
        gpu_values[ ii ]->show();
        gpu_labels[ ii ]->setEnabled( _v != -1 );
        gpu_values[ ii ]->setEnabled( _v != -1 );

        if ( _v != -1 )
        {
            gpu_labels[ ii ]->setToolTip( QString::fromStdString( pop_answer.gpu_metrics_helps[ i ]  ) );
            gpu_values[ ii ]->setToolTip( QString::fromStdString( pop_answer.gpu_metrics_helps[ i ]  ) );
        }
        else
        {
            gpu_labels[ ii ]->setToolTip(  tr( "GPU Efficiency cannot be calculated" ) );
            gpu_values[ ii ]->setToolTip( tr( "GPU Efficiency cannot be calculated" )  );
        }
    }
    size_t io_size =  pop_answer.io_metric_names.size();
    for ( size_t i = 0; i < io_size; ++i )
    {
        size_t ii = io_size - i - 1;
        double _v = pop_answer.io_values[ i ];
        io_values[ ii ]->setText( ( _v != -1 ) ? QString::number( _v ) : tr( "not available" ) );
        io_labels[ ii ]->setText( QString::fromStdString( pop_answer.io_metric_names[ i ] ) );

        io_labels[ ii ]->show();
        io_values[ ii ]->show();
        io_labels[ ii ]->setEnabled( _v != -1 );
        io_values[ ii ]->setEnabled( _v != -1 );

        if ( _v != -1 )
        {
            io_labels[ ii ]->setToolTip( QString::fromStdString( pop_answer.io_metrics_helps[ i ]  ) );
            io_values[ ii ]->setToolTip( QString::fromStdString( pop_answer.io_metrics_helps[ i ]  ) );
        }
        else
        {
            io_labels[ ii ]->setToolTip(  tr( "IO Efficiency cannot be calculated" ) );
            io_values[ ii ]->setToolTip( tr( "IO Efficiency cannot be calculated" )  );
        }
    }
    size_t additional_size =  pop_answer.additional_metric_names.size();
    for ( size_t i = 0; i < additional_size; ++i )
    {
        size_t ii = additional_size - i - 1;
        double _v = pop_answer.additional_values[ i ];
        additional_values[ ii ]->setText( ( _v != -1 ) ? QString::number( _v ) : tr( "not available" ) );
        additional_labels[ ii ]->setText( QString::fromStdString( pop_answer.additional_metric_names[ i ] ) );

        additional_labels[ ii ]->show();
        additional_values[ ii ]->show();
        additional_labels[ ii ]->setEnabled( _v != -1 );
        additional_values[ ii ]->setEnabled( _v != -1 );

        if ( _v != -1 )
        {
            additional_labels[ ii ]->setToolTip( QString::fromStdString( pop_answer.additional_metrics_helps[ i ]  ) );
            additional_values[ ii ]->setToolTip( QString::fromStdString( pop_answer.additional_metrics_helps[ i ]  ) );
        }
        else
        {
            additional_labels[ ii ]->setToolTip(  tr( "Additional efficiencies cannot be calculated" ) );
            additional_values[ ii ]->setToolTip( tr( "Additional efficiencies cannot be calculated" )  );
        }
    }

    size_t control_size =  pop_answer.control_metric_names.size();
    double min_control  = pop_answer.control_values[ 0 ];
    double avg_control  = pop_answer.control_values[ 1 ];
    double max_control  = pop_answer.control_values[ 2 ];

    double min_dev = ( avg_control - min_control ) / avg_control * 100;
    double max_dev = ( max_control - avg_control ) / avg_control * 100;


    control_values[ 0 ]->setText( ( min_control != -1 ) ? QString::number( min_control ) + ", " +  QString::number( min_dev ) + "%" : tr( "not available" ) );
    control_values[ 1 ]->setText( ( avg_control != -1 ) ? QString::number( avg_control ) : tr( "not available" ) );
    control_values[ 2 ]->setText( ( max_control != -1 ) ? QString::number( max_control ) + ", " +  QString::number( max_dev ) + "%" : tr( "not available" ) );
    control_labels[ 0 ]->setText( QString::fromStdString( pop_answer.control_metric_names[ 0 ] ) );
    control_labels[ 1 ]->setText( QString::fromStdString( pop_answer.control_metric_names[ 1 ] ) );
    control_labels[ 2 ]->setText( QString::fromStdString( pop_answer.control_metric_names[ 2 ] ) );

    control_labels[ 0 ]->show();
    control_values[ 0 ]->show();
    control_labels[ 0 ]->setEnabled( min_control != -1 );
    control_values[ 0 ]->setEnabled( min_control != -1 );
    control_labels[ 0 ]->setToolTip( QString::fromStdString( pop_answer.control_metrics_helps[ 0 ]  ) );
    control_values[ 0 ]->setToolTip( QString::fromStdString( pop_answer.control_metrics_helps[ 0 ]  ) );

    control_labels[ 1 ]->show();
    control_values[ 1 ]->show();
    control_labels[ 1 ]->setEnabled( avg_control != -1 );
    control_values[ 1 ]->setEnabled( avg_control != -1 );
    control_labels[ 1 ]->setToolTip( QString::fromStdString( pop_answer.control_metrics_helps[ 0 ]  ) );
    control_values[ 1 ]->setToolTip( QString::fromStdString( pop_answer.control_metrics_helps[ 0 ]  ) );

    control_labels[ 2 ]->show();
    control_values[ 2 ]->show();
    control_labels[ 2 ]->setEnabled( max_control != -1 );
    control_values[ 2 ]->setEnabled( max_control != -1 );
    control_labels[ 2 ]->setToolTip( QString::fromStdString( pop_answer.control_metrics_helps[ 0 ]  ) );
    control_values[ 2 ]->setToolTip( QString::fromStdString( pop_answer.control_metrics_helps[ 0 ]  ) );


    for ( size_t i = 0; i < pop_size; ++i )
    {
        text_to_copy.append( pop_labels[ i ]->text() ).append( "\t" ).append( pop_values[ i ]->text() ).append( "\n" );
    }
    text_to_copy.append( "\n" );
    for ( size_t i = 0; i < gpu_size; ++i )
    {
        text_to_copy.append( gpu_labels[ i ]->text() ).append( "\t" ).append( gpu_values[ i ]->text() ).append( "\n" );
    }
    text_to_copy.append( "\n" );
    for ( size_t i = 0; i < io_size; ++i )
    {
        text_to_copy.append( io_labels[ i ]->text() ).append( "\t" ).append( io_values[ i ]->text() ).append( "\n" );
    }
    text_to_copy.append( "\n" );
    for ( size_t i = 0; i < additional_size; ++i )
    {
        text_to_copy.append( additional_labels[ i ]->text() ).append( "\t" ).append( additional_values[ i ]->text() ).append( "\n" );
    }
    text_to_copy.append( "\n" );
    for ( size_t i = 0; i < control_size; ++i )
    {
        text_to_copy.append( control_labels[ i ]->text() ).append( "\t" ).append( control_values[ i ]->text() ).append( "\n" );
    }
}


void
CubePOPAdvisorWidget::enableControls( bool enable )
{
    if ( enable )
    {
        std::for_each( pop_labels.begin(), pop_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );
        std::for_each( pop_values.begin(), pop_values.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );

        std::for_each( gpu_labels.begin(), gpu_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );
        std::for_each( gpu_values.begin(), gpu_values.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );

        std::for_each( io_labels.begin(), io_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );
        std::for_each( io_values.begin(), io_values.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );

        std::for_each( additional_labels.begin(), additional_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );
        std::for_each( additional_values.begin(), additional_values.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );

        std::for_each( control_labels.begin(), control_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );
        std::for_each( control_values.begin(), control_values.end(), [ ]( QLabel* l ){
            l->setEnabled( true );
        } );
    }
    else
    {
        std::for_each( pop_labels.begin(), pop_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );
        std::for_each( pop_values.begin(), pop_values.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );

        std::for_each( gpu_labels.begin(), gpu_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );
        std::for_each( gpu_values.begin(), gpu_values.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );

        std::for_each( io_labels.begin(), io_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );
        std::for_each( io_values.begin(), io_values.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );

        std::for_each( additional_labels.begin(), additional_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );
        std::for_each( additional_values.begin(), additional_values.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );

        std::for_each( control_labels.begin(), control_labels.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );
        std::for_each( control_values.begin(), control_values.end(), [ ]( QLabel* l ){
            l->setEnabled( false );
        } );
    }
}

void
CubePOPAdvisorWidget::copyToClipboard()
{
    QClipboard* clipboard = QGuiApplication::clipboard();
    clipboard->setText( text_to_copy );
}
