/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2022-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef MEASUREMENTWINDOW_H
#define MEASUREMENTWINDOW_H

#include <QtWidgets>
#include <string>

#include "ContextFreePlugin.h"
#include "console.h"
#include "Cube.h"
#include "ContextFreeServices.h"
#include "CubeIoProxy.h"

namespace measurementwindow
{
// forward declaration of tab classes
class ExecutionTab;
class InstrumentationTab;
class SettingTab;

/**
 * @file measurementwindow.h
 * @brief Contains the MeasurementWindow, SettingTab, InstrumentationTab and ExecutionTab classes.
 */
class MeasurementWindow : public QDialog
{
    Q_OBJECT

public:
    MeasurementWindow( cubepluginapi::ContextFreeServices* service );
    cubepluginapi::ContextFreeServices*
    getService() const;

    SettingTab*         settingTab;
    InstrumentationTab* instrumentationTab;
    ExecutionTab*       executionTab;
    /**
     * @brief The QTabWidget that is used to display the SettingTab, InstrumentationTab and ExecutionTab.
     */
    QTabWidget* tabWidget;
    /**
     * @brief A QSettings object used to to store and retrieve settings.
     */
    QSettings settings;
    bool
    loadSettings( QString prefix = "measurement" );
    QString
    getStatus( int jobid ) const;
    void
    deleteSettings( QString prefix = "measurement/" );

private:
    console::Console*                   console;
    QDialogButtonBox*                   buttonBox;
    cubepluginapi::ContextFreeServices* service;
};

/**
 * @brief The MeasurementTab class provides an interface for creating a tab in the measurement plugin.
 */
class MeasurementTab : public QWidget
{
    Q_OBJECT

public:
    explicit
    MeasurementTab( cubepluginapi::ContextFreeServices* service,
                    console::Console*                   console,
                    QWidget*                            parent = 0 );
    virtual ~MeasurementTab()
    {
    };

    cubepluginapi::ContextFreeServices*
    getService() const;


protected:
    cubepluginapi::ContextFreeServices* service;
    console::Console*                   console;
    MeasurementWindow*                  measurementWindow;
    /**
     * @brief A list containing all config variables that are available via "scorep-info config-vars"
     */
    std::list<std::string> configVars;

    static void
    uncheck( QButtonGroup* buttonGroup );
    static void
    uncheck( QRadioButton* radioButton );
    void
    unsetConfigVars();
    void
    setConfigVars();
    bool
    event( QEvent* event ) override;
};

/**
 * @brief The SettingTab class implements the MeasurementTab interface. It provides steps to select a Score-P version.
 */
class SettingTab : public MeasurementTab
{
    Q_OBJECT

public:
    explicit
    SettingTab( cubepluginapi::ContextFreeServices* service,
                console::Console*                   console,
                QWidget*                            parent = 0 );
    void
    expandPath( QString path,
                bool    saveSettings = true );
    bool
    isValidPath( QString path );
    void
    proceed();

private:
    /**
     * @brief Indicates whether the selected Score-P version was found by manual search, in the user's PATH or via a module.
     */
    enum  { BrowsePath, InPathVariable, Module };
    /**
     * @brief Indicates if the MPI and compiler used to build a Score-P installation matches the selected MPI and compiler.
     */
    enum { Matching, NotMatching, PossiblyMatching };
    QPushButton*  proceedButton;
    QPushButton*  helpButton;
    QPushButton*  browseScorepButton;
    QPushButton*  findScorePButton;
    QRadioButton* inPathButton;
    QLabel*       errorMsg;
    QLabel*       browseDirLabel;
    QLabel*       scorePVersionsLabel;
    QLabel*       foundScorePsLabel;
    QLabel*       inPathUsableConf;
    QLabel*       inPathUsableInfo;
    QLabel*       inPathPapiLabel;
    QLabel*       inPathPapiInfo;
    QLabel*       inPathUnwindedLabel;
    QLabel*       inPathUnwindedInfo;
    QIcon         infoIcon;
    QHBoxLayout*  inPathButtonLayout;
    QHBoxLayout*  proceedButtonLayout;
    QVBoxLayout*  moduleLayout;
    QVBoxLayout*  mainLayout;
    QWidget*      mainWidget;
    QWidget*      loadButtonWidget;
    QWidget*      jobWidget;
    QWidget*      proceedButtonWidget;
    QWidget*      versionWidget;
    QWidget*      environmentWidget;
    QString       browseDirPath;
    QString       formerConfigurationPath;
    QString
    getRealPath( QString path );
    QString
    cutAfterLine( QString text,
                  int     line );
    std::string
    getMpi( QString path );
    std::string
    getCompiler( QString path );

    QButtonGroup* inPathButtonGroup;
    QButtonGroup* moduleButtonGroup;
    QButtonGroup* jobidGroup;
    QButtonGroup* removeJobidGroup;
    QComboBox*    compilerComboBox;
    QComboBox*    mpiComboBox;
    bool          moduleAvail = false;
    bool
    findScorep();
    bool
    papiSupport( QString path );
    bool
    unwindingSupport( QString path );
    void
    removeFromPath( QString removePath );
    int
    isMatching( QString path );

    int                         selectPath;
    int                         minimumWidth               = 0; // holds minimum required width to show content of scorePVersionsLabel
    int                         minimumHeight              = 0; // holds minimum required height to show content of scorePVersionsLabel
    int                         scorePVersionsLayoutMargin = 7;
    QMap<int, QList<QWidget*> > jobMap;

private slots:
    void
    onProceedButtonClicked();
    void
    onBrowseScorePButtonClicked();
    void
    onHelpButtonClicked();
    void
    onFindScorePButtonClicked();
    void
    inPathSelected();
    void
    moduleSelected();
    void
    checkPath();
    void
    onNewMeasurementButtonClicked();
    void
    onLoadMeasurementButtonClicked();
    void
    onLoadJobButtonClicked( QAbstractButton* );
    void
    onRemoveJobButtonClicked( QAbstractButton* );
};

/**
 * @brief The InstrumentationTab class implements the MeasurementTab interface. It provides steps to instrument a selected
 * application.
 */
class InstrumentationTab : public MeasurementTab
{
    Q_OBJECT

public:
    explicit
    InstrumentationTab( cubepluginapi::ContextFreeServices* service,
                        console::Console*                   console,
                        QWidget*                            parent = 0 );
    QString       executableName;
    QString       executableDir;
    QLabel*       fileLabel;
    QWidget*      selectInstrumentationWidget;
    QWidget*      analysisWidget;
    QWidget*      buildWidget;
    QWidget*      selectBuildWidget;
    QWidget*      makefileWidget;
    QWidget*      selectMakefileWidget;
    QPushButton*  selectExecutableButton;
    QPushButton*  analysisButton;
    QRadioButton* useMakefile;
    QLabel*       selectInstrumentationLabel;
    QLabel*       analysisLabel;
    QLineEdit*    buildCommandEdit;
    QRadioButton* formerInstrumentationButton;
    QRadioButton* newInstrumentationButton;
    bool
    isExecutable( QString file,
                  bool    saveSettings = true );
    bool
    isInstrumented( QString path );

private:
    QVBoxLayout*  layout;
    QVBoxLayout*  mainLayout;
    QVBoxLayout*  makefileLayout;
    QPushButton*  buildButton;
    QPushButton*  detectedMakefileButton;
    QPushButton*  browseMakefileButton;
    QPushButton*  saveChangesButton;
    QPushButton*  discardChangesButton;
    QString       detectedMakefile;
    QString       openedFile;
    QLabel*       selectExecutableErrorMessage;
    QLabel*       selectMakeLabel;
    QLabel*       buildLabel;
    QRadioButton* useCMake;
    QGroupBox*    selectInstrumentationBox;
    QGroupBox*    selectMakeBox;
    QTextEdit*    makefileEdit;
    QWidget*      mainWidget;
    QWidget*      selectExecutableWidget;
    QWidget*      stepsWidget;
    bool          useFormerInstrumentation;
    void
    openMakefile( QString filename );

public slots:
    void
    onAnalysisButtonClicked( bool loadSettings = false );
    void
    selectedAdjustMakefile();

private slots:
    void
    onSelectExecutableButtonClicked();
    void
    onBuildButtonClicked();
    void
    onBrowseMakefileButtonClicked();
    void
    onDetectedMakefileButtonClicked();
    void
    onSaveChangesButtonClicked();
    void
    onDiscardChangesClicked();
    void
    selectedFormerInstrumentation();
    void
    selectedNewInstrumentation();
};

/**
 * @brief The ExecutionTab class implements the MeasurementTab interface. It provides steps to perform a measurement for
 * an instrumented application.
 */
class ExecutionTab : public MeasurementTab
{
    Q_OBJECT

public:
    explicit
    ExecutionTab( cubepluginapi::ContextFreeServices* service,
                  console::Console*                   console,
                  QWidget*                            parent = 0 );
    bool
    profileExists();

    QRadioButton* finetunedRunButton;
    QSpinBox*     numProcs;
    QSpinBox*     numThreads;
    void
    addProfileButtons();

    QRadioButton* initialRunButton;
    QRadioButton* browseFilterButton;
    QRadioButton* createFilterButton;
    QPushButton*  inspectFilterFileButton;
    QLabel*       filterInfoLabel;
    QLabel*       filterPathLabel;
    QLabel*       runLabel;
    QString       filterFile;
    QWidget*      executeWidget;
    QButtonGroup* profileGroup;
    QSpinBox*     bufferPercent;
    QSpinBox*     timePerVisit;
    QComboBox*    type;
    QWidget*      informationWidget;
    QWidget*      selectRunWidget;
    QWidget*      customRunWidget;
    QWidget*      finetunedRunWidget;
    QWidget*      showProfileWidget;
    QWidget*      generateFilterWidget;
    QWidget*      configureFilterWidget;
    QWidget*      filterOptionsWidget;
    QWidget*      scorepionWidget;
    QWidget*      mainWidget;
    QWidget*      editFilterWidget;
    QWidget*      editJobScriptWidget;
    QWidget*      prepareJobWidget;
    QWidget*      inspectFilterWidget;
    QString       profileSettingsPath;
    QString       prefix = "measurement";
    QString       slurmScriptName;

private:
    QRadioButton* customRunButton;
    QRadioButton* detailedRunButton;
    QVBoxLayout*  mainLayout;
    QVBoxLayout*  layout;
    QVBoxLayout*  profileBoxLayout;
    QLineEdit*    dirNameEdit;
    QTextEdit*    filterFileEdit;
    QTextEdit*    jobScriptEdit;
    QTextEdit*    inspectFilterEdit;
    QCheckBox*    enableProfiling;
    QCheckBox*    enableTracing;
    QCheckBox*    enableUnwinding;
    QCheckBox*    beVerbose;
    QLineEdit*    execCommandEdit;
    QPushButton*  runApplicationButton;
    QPushButton*  prepareJobScriptButton;
    QGroupBox*    profileBox;
    QLabel*       dirNameErrorMessage;
    QLabel*       shellScriptErrorLabel;
    QLabel*       runInfoLabel;
    QLabel*       editJobScriptLabel;
    QString       profiles;
    QString       generateFilterProfile;
    QString
    convertDate();
    void
    setRunCommand();
    QString
    whitespaces( int n );

    QString jobid;
    QTimer* timer;

public slots:
    void
    enableProfilingClicked( int  i,
                            bool saveSettings = true );
    void
    selectedInitialRun( bool saveSettings = true );
    void
    selectedFinetunedRun();
    void
    onCreateFilterButtonClicked();
    void
    onCreateManuallyButtonClicked();
    void
    onGenerateFilterButtonClicked();
    void
    selectedProfile();
    void
    onConfigureScoreOptionsButtonClicked();
    void
    onRunApplicationButtonClicked( bool saveSettings = true );
    void
    onSaveJobScriptChangesButtonClicked();
    void
    onTakeMeasurementButtonClicked();

private slots:
    void
    selectedNumberOfProcesses( int number );
    void
    selectedNumberOfThreads( int number );
    void
    selectedDetailedRun();
    void
    selectedCustomRun();
    void
    enableTracingClicked( int i );
    void
    enableUnwindingClicked( int i );
    void
    beVerboseClicked( int i );
    void
    onProfilingInfoButtonClicked();
    void
    onTracingInfoButtonClicked();
    void
    onUnwindingInfoButtonClicked();
    void
    onVerboseInfoButtonClicked();
    void
    onShowProfileButtonClicked();
    void
    onBrowseFilterButtonClicked();
    void
    onGenerateButtonClicked();
    void
    setDefaultFilterValues();
    void
    onUseScorepionButtonClicked();
    void
    onSaveChangesButtonClicked();
    void
    onDiscardChangesButtonClicked();
    void
    onCreateShellScriptButtonClicked();
    void
    onPrepareJobScriptButtonClicked();
    void
    onDiscardJobScriptButtonClicked();
    void
    onInspectFilterButtonClicked();
    void
    onDiscardFilterChangesButtonClicked();
    void
    onSaveFilterChangesButtonClicked();
    void
    onOpenOwnJobScriptButtonClicked();
    void
    onOpenGeneratedJobScriptButtonClicked();
    void
    updateStatus();
};
}

#endif // MEASUREMENTWINDOW_H
