/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2022-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <QtWidgets>
#include <QTimer>
#include <unistd.h>
#include <iostream>
#include <regex>
#include <unistd.h>
#include <QtGlobal>
#include "measurementwindow.h"
#include "console.h"
#include "PluginServices.h"
#include "Globals.h"

using namespace cubepluginapi;
using namespace measurementwindow;
using namespace std;
using namespace console;
using namespace cube;

#ifdef HAS_QREGULAR_EXPRESSION
#include <QRegularExpression>
#define REGULAR_EXPRESSION QRegularExpression
#else
#include <QRegExp>
#define REGULAR_EXPRESSION QRegExp
#endif

/**
 * @param service A pointer to a ContextFreeServices object provided by Cube, which is used to initialize the "service" member
 * variable.
 */
MeasurementWindow::MeasurementWindow( ContextFreeServices* service )
{
    this->service = service;

    console   = new Console( service, this );
    tabWidget = new QTabWidget;

    settingTab         = new SettingTab( service, console, this );
    instrumentationTab = new InstrumentationTab( service, console, this );
    executionTab       = new ExecutionTab( service, console, this );

    tabWidget->addTab( settingTab, tr( "Set-up" ) );
    tabWidget->addTab( instrumentationTab, tr( "Instrumentation" ) );
    tabWidget->addTab( executionTab, tr( "Measurement" ) );

    tabWidget->setTabEnabled( 1, false );
    tabWidget->setTabEnabled( 2, false );

    QHBoxLayout* generalLayout = new QHBoxLayout;
    QVBoxLayout* mainLayout    = new QVBoxLayout;
    mainLayout->addWidget( tabWidget );
    generalLayout->addLayout( mainLayout );
    generalLayout->addWidget( console );
    setLayout( generalLayout );

    setWindowTitle( tr( "Measurement Plugin" ) );
}

/**
 * @brief Loads the saved settings from a former measurement. The prefix specifies which measurement is loaded.
 * @param prefix The prefix of the measurement to be loaded. Either "measurement" to load the recent measurement or a
 * specific jobid.
 * @return True if settings were loaded successful; otherwise, false.
 */
bool
MeasurementWindow::loadSettings( QString prefix )
{
    executionTab->prefix = prefix;
    if ( !settings.value( prefix + "/consoleText" ).toString().isEmpty() )
    {
        console->console->setHtml( settings.value( prefix + "/consoleText" ).toString() );
    }

    if ( settings.value( prefix + "/loadScoreP" ).toString() == "Path" && settingTab->isValidPath( settings.value( prefix + "/path" ).toString() ) )
    {
        settingTab->expandPath( settings.value( prefix + "/path" ).toString(), false );
    }
    else if ( settings.value( prefix + "/loadScoreP" ).toString() == "Module" )
    {
        int returncode;
        console->execCommand( settings.value( prefix + "/moduleCommand" ).toString().toStdString(), returncode, true, true );
        if ( returncode != 0 )
        {
            return false;
        }
    }
    else
    {
        return false;
    }

    settingTab->proceed();

    if ( instrumentationTab->isExecutable( settings.value( prefix + "/executableName" ).toString(), false ) )
    {
        instrumentationTab->executableName = settings.value( prefix + "/executableName" ).toString();
        instrumentationTab->executableDir  = settings.value( prefix + "/executableDir" ).toString();
        instrumentationTab->fileLabel->setText( tr( "Selected file: " ) + instrumentationTab->executableName );
        instrumentationTab->fileLabel->setVisible( true );
        instrumentationTab->selectInstrumentationWidget->setVisible( true );
        instrumentationTab->selectExecutableButton->setText( tr( "Browse another executable file" ) );
        instrumentationTab->formerInstrumentationButton->setEnabled( true );
        instrumentationTab->selectInstrumentationLabel->setText( tr( "Your application is already instrumented. Please select if you want to continue with existing instrumentation or if you want to prepare a new instrumentation." ) );
        instrumentationTab->selectInstrumentationLabel->setVisible( true );

        // check if instrumentation was successful
        if ( !instrumentationTab->isInstrumented( instrumentationTab->executableName ) || settings.value( prefix + "/formerInstrumentation" ).toString().isEmpty() )
        {
            if ( !instrumentationTab->isInstrumented( instrumentationTab->executableName ) )
            {
                instrumentationTab->formerInstrumentationButton->setEnabled( false );
            }
            return true;
        }
        if ( settings.value( prefix + "/formerInstrumentation" ).toString() == "True" )
        {
            instrumentationTab->formerInstrumentationButton->setChecked( true );
            instrumentationTab->analysisButton->setVisible( true );
            instrumentationTab->analysisWidget->setVisible( true );
            instrumentationTab->onAnalysisButtonClicked( true );
        }
        else
        {
            instrumentationTab->newInstrumentationButton->setChecked( true );
            instrumentationTab->selectBuildWidget->setVisible( true );
            instrumentationTab->analysisWidget->setVisible( false );
            if ( settings.value( prefix + "/adjustMakefile" ).toString() != "True" )
            {
                return true;
            }
            instrumentationTab->useMakefile->setChecked( true );
            instrumentationTab->selectedAdjustMakefile();

            instrumentationTab->buildWidget->setVisible( true );
            instrumentationTab->selectMakefileWidget->setVisible( true );
            if ( settings.value( prefix + "/buildCmd" ).toString().isEmpty() )
            {
                return true;
            }
            instrumentationTab->buildCommandEdit->setText( settings.value( prefix + "/buildCmd" ).toString() );
            if ( settings.value( prefix + "/buildSuccess" ).toString() != "True" )
            {
                return true;
            }

            instrumentationTab->analysisLabel->setVisible( true );
            instrumentationTab->analysisLabel->setText( tr( "<font color=\"green\">Your application has been rebuilt. The instrumentation was successful.</font>" ) );
            instrumentationTab->analysisButton->setVisible( true );
            instrumentationTab->analysisWidget->setVisible( true );
            instrumentationTab->onAnalysisButtonClicked( true );
        }
        if ( settings.value( prefix + "/selectedRun" ).toString() != "" )
        {
            executionTab->numProcs->setValue( settings.value( prefix + "/numProcs" ).toInt() );
            executionTab->numThreads->setValue( settings.value( prefix + "/numThreads" ).toInt() );
            setenv( "OMP_NUM_THREADS", const_cast<char*>( std::to_string( executionTab->numThreads->value() ).c_str() ), 1 );

            if ( settings.value( prefix + "/selectedRun" ).toString() == "Initial" )
            {
                executionTab->initialRunButton->setChecked( true );
                executionTab->selectedInitialRun( false );
            }
            else if ( settings.value( prefix + "/selectedRun" ).toString() == "Finetuned" )
            {
                executionTab->finetunedRunButton->setChecked( true );
                executionTab->selectedFinetunedRun();
                if ( settings.value( prefix + "/specifyFilter" ).toString() == "Select" )
                {
                    executionTab->filterFile = settings.value( prefix + "/filterFile" ).toString();
                    if ( !QFile::exists( executionTab->filterFile ) )
                    {
                        return true;
                    }
                    executionTab->browseFilterButton->setChecked( true );
                    executionTab->filterPathLabel->setVisible( true );
                    executionTab->filterPathLabel->setText( tr( "Selected filter file: " ) + executionTab->filterFile );
                    setenv( "SCOREP_FILTERING_FILE", const_cast<char*>( executionTab->filterFile.toStdString().c_str() ), 1 );
                    executionTab->executeWidget->setVisible( true );
                    executionTab->inspectFilterFileButton->setVisible( true );
                    executionTab->inspectFilterFileButton->setText( tr( "Inspect selected filter" ) );
                }
                else if ( settings.value( prefix + "/specifyFilter" ).toString() == "Create" )
                {
                    executionTab->createFilterButton->setChecked( true );
                    executionTab->onCreateFilterButtonClicked();
                    if ( settings.value( prefix + "/createFilter" ).toString() == "Manual" )
                    {
                        if ( settings.value( prefix + "/filterFile" ).toString() != "" )
                        {
                            executionTab->filterFile = settings.value( prefix + "/filterFile" ).toString();
                            if ( !QFile::exists( executionTab->filterFile ) )
                            {
                                return true;
                            }
                            executionTab->executeWidget->setVisible( true );
                            executionTab->filterInfoLabel->setText( tr( "<font color=\"green\">The filter file was created successfully. Please rerun the application to perform the measurement.</font>" ) );
                            executionTab->filterInfoLabel->setVisible( true );
                        }
                        else
                        {
                            return true;
                        }
                    }
                    else if ( settings.value( prefix + "/createFilter" ).toString() == "Automatic" )
                    {
                        executionTab->onGenerateFilterButtonClicked();
                        if ( QFile::exists( settings.value( prefix + "/filterProfile" ).toString() ) )
                        {
                            QList<QAbstractButton*> buttons = executionTab->profileGroup->buttons();
                            foreach( const auto & button, buttons )
                            {
                                QRadioButton* radioButton = static_cast<QRadioButton*>( button );
                                if ( radioButton->text() == settings.value( prefix + "/filterProfile" ).toString() )
                                {
                                    radioButton->setChecked( true );
                                }
                            }
                            executionTab->selectedProfile();
                        }
                        else
                        {
                            return true;
                        }
                        if ( settings.value( prefix + "/generateFilter" ).toString() == "ScorePion" )
                        {
                            // further development
                        }
                        else if ( settings.value( prefix + "/generateFilter" ).toString() == "ScorePScore" )
                        {
                            executionTab->onConfigureScoreOptionsButtonClicked();
                            executionTab->filterFile = settings.value( prefix + "/filterFile" ).toString();
                            if ( executionTab->filterFile.endsWith( "initial_scorep.filter" ) )
                            {
                                executionTab->timePerVisit->setValue( settings.value( prefix + "/timePerVisit" ).toInt() );
                                executionTab->bufferPercent->setValue( settings.value( prefix + "/bufferPercent" ).toInt() );
                                executionTab->type->setCurrentIndex( settings.value( prefix + "/type" ).toInt() );

                                executionTab->filterInfoLabel->setText( tr( "<font color=\"green\">The filter file was generated successfully. Please rerun the application to perform the measurement.</font>" ) );
                                executionTab->filterInfoLabel->setVisible( true );
                                executionTab->executeWidget->setVisible( true );
                                executionTab->inspectFilterFileButton->setVisible( true );
                                executionTab->inspectFilterFileButton->setText( tr( "Inspect generated filter" ) );
                            }
                            else
                            {
                                return true;
                            }
                        }
                        else
                        {
                            return true;
                        }
                    }
                    else
                    {
                        return true;
                    }
                }
                else
                {
                    return true;
                }
            }
        }
        else
        {
            return true;
        }
        if ( settings.value( prefix + "/prepareJob" ).toString() == "True" )
        {
            executionTab->prepareJobWidget->setVisible( true );
        }
        if ( settings.value( prefix + "/runSuccess" ).toString() == "True" )
        {
            executionTab->onRunApplicationButtonClicked( false );
            executionTab->profileSettingsPath = settings.value( prefix + "/profilePath" ).toString();
        }
        else if ( !settings.value( prefix + "/jobid" ).toString().isEmpty() )
        {
            executionTab->slurmScriptName = settings.value( prefix + "/slurmScriptName" ).toString();
            QString status = getStatus( settings.value( prefix + "/jobid" ).toInt() );
            if ( status == "COMPLETED" )
            {
                // check if measurement directory exists
                executionTab->runLabel->setText( tr( "<font color=\"green\">Your job is finished" ) );
                executionTab->runLabel->setVisible( true );
                executionTab->showProfileWidget->setVisible( true );

                // presettings can not be changed after run
                executionTab->selectRunWidget->setEnabled( false );
                executionTab->executeWidget->setEnabled( false );
                executionTab->finetunedRunWidget->setEnabled( false );
                executionTab->generateFilterWidget->setEnabled( false );
                executionTab->filterInfoLabel->setVisible( false );
                executionTab->customRunWidget->setEnabled( false );
                executionTab->configureFilterWidget->setEnabled( false );
                executionTab->filterOptionsWidget->setEnabled( false );
                executionTab->scorepionWidget->setEnabled( false );

                executionTab->profileSettingsPath = settings.value( prefix + "/profilePath" ).toString();
            }
            else if ( status == "FAILED" )
            {
                executionTab->runLabel->setText( tr( "<font color=\"green\">Your job has been submitted.</font>" ) );
                // check if measurement directory exists
                executionTab->runLabel->setText( tr( "<font color=\"red\">Your job failed" ) );
                executionTab->runLabel->setVisible( true );

                // presettings can not be changed after run
                executionTab->selectRunWidget->setEnabled( false );
                executionTab->executeWidget->setEnabled( false );
                executionTab->finetunedRunWidget->setEnabled( false );
                executionTab->generateFilterWidget->setEnabled( false );
                executionTab->filterInfoLabel->setVisible( false );
                executionTab->customRunWidget->setEnabled( false );
                executionTab->configureFilterWidget->setEnabled( false );
                executionTab->filterOptionsWidget->setEnabled( false );
                executionTab->scorepionWidget->setEnabled( false );
            }
            else if ( status == "RUNNING" )
            {
                executionTab->runLabel->setText( tr( "<font color=\"green\">Your job has been submitted.</font>" ) );
                // check if measurement directory exists
                executionTab->runLabel->setText( tr( "<font color=\"green\">Your job is running" ) );
                executionTab->runLabel->setVisible( true );

                // presettings can not be changed after run
                executionTab->selectRunWidget->setEnabled( false );
                executionTab->executeWidget->setEnabled( false );
                executionTab->finetunedRunWidget->setEnabled( false );
                executionTab->generateFilterWidget->setEnabled( false );
                executionTab->filterInfoLabel->setVisible( false );
                executionTab->customRunWidget->setEnabled( false );
                executionTab->configureFilterWidget->setEnabled( false );
                executionTab->filterOptionsWidget->setEnabled( false );
                executionTab->scorepionWidget->setEnabled( false );
            }
        }
        else
        {
            return true;
        }
    }
    else
    {
        return true;
    }
    return true;
}

/**
 * @param prefix The prefix of the settings to be deleted. Either "measurement/" to delete the recent measurement
 * or a specific jobid.
 */
void
MeasurementWindow::deleteSettings( QString prefix )
{
    // delete former settings
    QStringList keys = settings.allKeys();
    foreach( const QString &key, keys )
    {
        if ( key.startsWith( prefix ) )
        {
            settings.remove( key );
        }
    }
}

/**
 * @brief Returns the status of a job with the given id.
 * @param jobid An integer containing a jobid.
 * @return A QString indicating the status of the job. It can be either "FAILED", "RUNNING", "COMPLETED"
 * or "STATUS NOT AVAILABLE".
 */
QString
MeasurementWindow::getStatus( int jobid ) const
{
    int     returncode;
    QString status = QString::fromStdString( console->execCommand( "sacct --brief -j " + to_string( jobid ), returncode, false ) );
    if ( status.contains( "FAILED" ) )
    {
        return "FAILED";
    }
    else if ( status.contains( "RUNNING" ) )
    {
        return "RUNNING";
    }
    else if ( status.contains( "COMPLETED" ) )
    {
        return "COMPLETED";
    }
    else
    {
        return "STATUS NOT AVAILABLE";
    }
}

ContextFreeServices*
MeasurementWindow::getService() const
{
    return this->service;
}

MeasurementTab::MeasurementTab( ContextFreeServices* service, Console* console, QWidget* parent ) : QWidget( parent )
{
    this->service = service;
    this->console = console;

    measurementWindow = static_cast<MeasurementWindow*>( parent );
    configVars        = list<string>();
}

/**
 * @brief Unchecks all radio buttons in the given button group.
 * @param buttonGroup A pointer to the QButtonGroup containing the QRadioButtons to be unchecked.
 */
void
MeasurementTab::uncheck( QButtonGroup* buttonGroup )
{
    buttonGroup->setExclusive( false );
    QList<QAbstractButton*> buttons = buttonGroup->buttons();
    foreach( const auto & button, buttons )
    {
        QRadioButton* radioButton = static_cast<QRadioButton*>( button );
        radioButton->setChecked( false );
    }
    buttonGroup->setExclusive( true );
}

/**
 * @brief Unchecks the given radio button.
 * @param radioButton A pointer to the QRadioButton to be unchecked.
 */
void
MeasurementTab::uncheck( QRadioButton* radioButton )
{
    radioButton->setAutoExclusive( false );
    radioButton->setChecked( false );
    radioButton->setAutoExclusive( true );
}

/**
 * @brief Clear the list which contains all config variables that are available via "scorep-info config-vars"
 */
void
MeasurementTab::unsetConfigVars()
{
    for ( const string& s : configVars )
    {
        unsetenv( s.c_str() );
    }
}

/**
 * @brief Fill the list with the config variables that are available via "scorep-info config-vars"
 */
void
MeasurementTab::setConfigVars()
{
    int         returncode;
    QStringList configinfo = QString::fromStdString( console->execCommand( "scorep-info config-vars", returncode, false ) ).split( REGULAR_EXPRESSION( "\\n" ) );
    for ( QString& o : configinfo )
    {
        if ( o.startsWith( "SCOREP" ) )
        {
            configVars.push_back( o.toStdString() );
        }
    }
}

/**
 * @brief Handles QEvent of type QEvent::ToolTip for SettingTab widget. It shows the tooltip for the widget under the mouse
 * pointer, if any. If there is no widget under the mouse pointer, it hides the tooltip and ignores the event.
 * @param event A pointer to the QEvent being processed.
 * @return Returns true if the event is handled; otherwise, returns false.
 */
bool
MeasurementTab::event( QEvent* event )
{
    if ( event->type() == QEvent::ToolTip )
    {
        QHelpEvent* helpEvent = static_cast<QHelpEvent*>( event );
        QWidget*    widget    = childAt( helpEvent->globalPos() );
        if ( widget != nullptr )
        {
            QToolTip::showText( helpEvent->globalPos(), widget->toolTip() );
        }
        else
        {
            QToolTip::hideText();
            event->ignore();
        }
        return true;
    }
    return QWidget::event( event );
}


ContextFreeServices*
MeasurementTab::getService() const
{
    return this->service;
}

/**
 * @brief Sets the layout.
 */
SettingTab::SettingTab( ContextFreeServices* service, Console* console, QWidget* parent ) : MeasurementTab( service, console, parent )
{
    QVBoxLayout* textLayout               = new QVBoxLayout;
    QVBoxLayout* layout                   = new QVBoxLayout;
    QHBoxLayout* comboBoxLayout           = new QHBoxLayout;
    QVBoxLayout* environmentLayout        = new QVBoxLayout;
    QVBoxLayout* scorePVersionsLayout     = new QVBoxLayout;
    QVBoxLayout* scorePVersionsBaseLayout = new QVBoxLayout;

    mainLayout          = new QVBoxLayout;
    inPathButtonLayout  = new QHBoxLayout;
    proceedButtonLayout = new QHBoxLayout;

    findScorePButton = new QPushButton( tr( "Find Score-P (automatic search)" ) );

    QPixmap infoPixmap( style()->standardIcon( QStyle::SP_MessageBoxInformation ).pixmap( 20, 20 ) );
    infoIcon      = QIcon( infoPixmap );
    proceedButton = new QPushButton( QIcon(), "Proceed" );

    browseScorepButton = new QPushButton( tr( "Browse Score-P (manual search)" ) );
    helpButton         = new QPushButton( tr( "Help" ) );
    inPathButton       = new QRadioButton;

    browseScorepButton->setVisible( false );
    proceedButton->setEnabled( false );
    findScorePButton->setToolTip( tr( "Use module to find other Score-P installations" ) );
    helpButton->setToolTip( ( tr( "Show help text" ) ) );
    proceedButton->setToolTip( tr( "Continue with instrumentation and use selected Score-P version" ) );

    connect( findScorePButton, SIGNAL( clicked() ), this, SLOT( onFindScorePButtonClicked() ) );
    connect( proceedButton, SIGNAL( clicked() ), this, SLOT( onProceedButtonClicked() ) );
    connect( browseScorepButton, SIGNAL( clicked() ), this, SLOT( onBrowseScorePButtonClicked() ) );
    connect( helpButton, SIGNAL( clicked() ), this, SLOT( onHelpButtonClicked() ) );

    compilerComboBox = new QComboBox;
    mpiComboBox      = new QComboBox;
    compilerComboBox->addItem( "gcc" );
    compilerComboBox->addItem( "ibm" );
    compilerComboBox->addItem( "intel" );
    compilerComboBox->addItem( "pgi" );
    compilerComboBox->addItem( "studio" );
    compilerComboBox->addItem( "clang" );
    mpiComboBox->addItem( "mpich2" );
    mpiComboBox->addItem( "impi" );
    mpiComboBox->addItem( "openmpi" );

    // if selected compiler/mpi changed check if matching Score-P version is already in path
    connect( compilerComboBox, SIGNAL( activated( int ) ), this, SLOT( checkPath() ) );
    connect( mpiComboBox, SIGNAL( activated( int ) ), this, SLOT( checkPath() ) );

    QLabel* compilerLabel    = new QLabel( tr( "Compiler:" ) );
    QLabel* mpiLabel         = new QLabel( tr( "MPI:" ) );
    QLabel* environmentLabel = new QLabel( tr( "<b>Please specify the compiler and mpi of your application which you want to measure.</b>" ) );
    environmentLabel->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
    QLabel* titleLabel = new QLabel( tr( "Score-P Measurement Wizard" ) );
    QLabel* preLabel   = new QLabel( tr( "Prerequisites:" ) );
    QLabel* pre1       = new QLabel( tr( "\u27a4  installed Score-P" ) );
    QLabel* pre2       = new QLabel( tr( "\u27a4  PAPI (optional)" ) );
    QLabel* pre3       = new QLabel( tr( "\u27a4  libunwinded (optional)" ) );
    QLabel* pre4       = new QLabel( tr( "<b>WARNING</b>: Score-P must be built with the same compiler and mpi as the application. Please make sure that the plugin has been started in the build environment of the application." ) );

    foundScorePsLabel   = new QLabel( tr( "<b>Found Score-P versions</b>" ) );
    scorePVersionsLabel = new QLabel;
    errorMsg            = new QLabel;
    browseDirLabel      = new QLabel;
    inPathPapiLabel     = new QLabel( "PAPI:" );
    inPathPapiInfo      = new QLabel;
    inPathUnwindedLabel = new QLabel( "libunwinded:" );
    inPathUnwindedInfo  = new QLabel;
    inPathUsableConf    = new QLabel;
    inPathUsableInfo    = new QLabel;

    inPathUsableInfo->setPixmap( infoPixmap );
    inPathUsableInfo->setVisible( false );
    inPathUsableConf->setVisible( false );
    inPathButton->setVisible( false );
    inPathPapiInfo->setVisible( false );
    inPathPapiLabel->setVisible( false );
    inPathUnwindedInfo->setVisible( false );
    inPathUnwindedLabel->setVisible( false );

    environmentLayout->addWidget( environmentLabel );
    comboBoxLayout->setContentsMargins( 0, 0, 0, 0 );
    comboBoxLayout->addWidget( compilerLabel, 0 );
    comboBoxLayout->addWidget( compilerComboBox, 0 );
    comboBoxLayout->addWidget( mpiLabel, 0 );
    comboBoxLayout->addWidget( mpiComboBox, 0 );
    comboBoxLayout->addStretch( 10 );
    environmentLayout->addLayout( comboBoxLayout );
    environmentLabel->setWordWrap( true );

    environmentWidget = new QWidget;
    environmentWidget->setLayout( environmentLayout );
    environmentWidget->setVisible( false );

    pre1->setToolTip( tr( "Score-P is necessary to run this plugin" ) );
    pre2->setToolTip( tr( "If PAPI counters are available one can access hardware performance counters" ) );
    pre3->setToolTip( tr( "If libunwinded is available one can use sampling" ) );
    pre1->setMaximumHeight( 22 );
    pre2->setMaximumHeight( 22 );
    pre3->setMaximumHeight( 22 );
    pre4->setWordWrap( true );

    QSizePolicy keepSpace = errorMsg->sizePolicy();
    keepSpace.setRetainSizeWhenHidden( true );
    errorMsg->setVisible( false );
    errorMsg->setWordWrap( true );
    errorMsg->setSizePolicy( keepSpace );
    browseDirLabel->setVisible( false );

    titleLabel->setStyleSheet( "font-weight: bold; font-size: 36px" );
    preLabel->setStyleSheet( "font-size: 17px; text-decoration: underline" );
    pre1->setStyleSheet( "font-size: 17px" );
    pre2->setStyleSheet( "font-size: 17px" );
    pre3->setStyleSheet( "font-size: 17px" );

    mainLayout->addWidget( titleLabel );
    mainLayout->addSpacing( 20 );
    textLayout->addWidget( preLabel );
    textLayout->addWidget( pre1 );
    textLayout->addWidget( pre2 );
    textLayout->addWidget( pre3 );
    textLayout->addSpacing( 20 );
    textLayout->addWidget( pre4 );
    mainLayout->addLayout( textLayout );
    mainLayout->addSpacing( 20 );

    moduleButtonGroup = new QButtonGroup;
    moduleLayout      = new QVBoxLayout;
    inPathButtonGroup = new QButtonGroup;
    moduleLayout->addStretch( 0 );

    inPathButtonLayout->addWidget( inPathButton );
    inPathButtonLayout->addSpacing( 10 );
    inPathButtonLayout->addWidget( inPathPapiLabel );
    inPathButtonLayout->addSpacing( 5 );
    inPathButtonLayout->addWidget( inPathPapiInfo );
    inPathButtonLayout->addSpacing( 10 );
    inPathButtonLayout->addWidget( inPathUnwindedLabel );
    inPathButtonLayout->addSpacing( 5 );
    inPathButtonLayout->addWidget( inPathUnwindedInfo );
    inPathButtonLayout->addSpacing( 20 );
    inPathButtonLayout->addWidget( inPathUsableConf );
    inPathButtonLayout->addSpacing( 5 );
    inPathButtonLayout->addWidget( inPathUsableInfo );
    inPathButtonLayout->addStretch( 0 );

    scorePVersionsLayout->addLayout( inPathButtonLayout );
    scorePVersionsLayout->addLayout( moduleLayout );

    scorePVersionsLabel->setFrameStyle( QFrame::Box | QFrame::Raised );
    scorePVersionsLabel->setAlignment( Qt::AlignCenter );
    scorePVersionsLabel->setLayout( scorePVersionsLayout );
    scorePVersionsLabel->setMargin( 0 );
    scorePVersionsLabel->setVisible( false );
    foundScorePsLabel->setVisible( false );
    comboBoxLayout->setContentsMargins( scorePVersionsLayoutMargin, scorePVersionsLayoutMargin,
                                        scorePVersionsLayoutMargin, scorePVersionsLayoutMargin );
    scorePVersionsLayout->setSpacing( 0 );
    scorePVersionsLayout->addStretch( 0 );

    setBackgroundRole( QPalette::Base );
    scorePVersionsBaseLayout->addWidget( foundScorePsLabel );
    scorePVersionsBaseLayout->addWidget( scorePVersionsLabel );
    scorePVersionsBaseLayout->addWidget( findScorePButton );
    scorePVersionsBaseLayout->addWidget( errorMsg );
    scorePVersionsBaseLayout->addWidget( browseScorepButton );
    scorePVersionsBaseLayout->addWidget( browseDirLabel );
    scorePVersionsBaseLayout->addStretch( 0 );

    // load measurement layout
    QVBoxLayout* loadButtonLayout      = new QVBoxLayout;
    QPushButton* loadMeasurementButton = new QPushButton( tr( "Load recent measurement" ) );
    QPushButton* newMeasurementButton  = new QPushButton( tr( "Start new measurement" ) );
    connect( loadMeasurementButton, SIGNAL( clicked() ), this, SLOT( onLoadMeasurementButtonClicked() ) );
    connect( newMeasurementButton, SIGNAL( clicked() ), this, SLOT( onNewMeasurementButtonClicked() ) );
    loadButtonLayout->addWidget( loadMeasurementButton );
    loadButtonLayout->addWidget( newMeasurementButton );
    loadButtonLayout->addSpacing( 5 );
    loadButtonWidget = new QWidget();
    loadButtonWidget->setLayout( loadButtonLayout );

    // check if recent measurement exists
    bool recentMeasurementExists = false;

    // list all jobs
    QStringList jobids;
    QStringList keys = measurementWindow->settings.allKeys();
    foreach( const QString &key, keys )
    {
        if ( key.contains( "measurement/" ) )
        {
            recentMeasurementExists = true;
        }
        // else if, because otherwise job id of recent measurement is listed twice
        else if ( key.contains( "jobid" ) )
        {
            jobids.append( measurementWindow->settings.value( key ).toString() );
        }
    }

    if ( !recentMeasurementExists )
    {
        loadMeasurementButton->setEnabled( false );
    }

    jobWidget = new QWidget;
    if ( !jobids.empty() )
    {
        QPixmap trashPixmap( style()->standardIcon( QStyle::SP_TrashIcon ).pixmap( 20, 20 ) );
        QIcon   trashIcon = QIcon( trashPixmap );
        jobidGroup       = new QButtonGroup;
        removeJobidGroup = new QButtonGroup;
        QLabel*      jobLabel  = new QLabel( tr( "<b>The following jobs were submitted:</b>" ) );
        QGridLayout* jobLayout = new QGridLayout;
        for ( int i = 0; i < jobids.size(); ++i )
        {
            QLabel*      jobidLabel        = new QLabel( jobids.at( i ) );
            QPushButton* loadJobidButton   = new QPushButton( tr( "Load job" ) );
            QPushButton* removeJobidButton = new QPushButton;

            // check job status
            QLabel* jobStatusLabel = new QLabel( measurementWindow->getStatus( jobids.at( i ).toInt() ) );

            removeJobidButton->setIcon( trashIcon );
            removeJobidButton->setFixedWidth( 25 );
            removeJobidButton->setToolTip( tr( "Remove measurement with job id " ) + jobids.at( i ) );

            jobidGroup->addButton( loadJobidButton, jobids.at( i ).toInt() );
            removeJobidGroup->addButton( removeJobidButton, jobids.at( i ).toInt() );

            jobLayout->addWidget( jobidLabel, i, 0 );
            jobLayout->addWidget( jobStatusLabel, i, 1 );
            jobLayout->addWidget( loadJobidButton, i, 2 );
            jobLayout->addWidget( removeJobidButton, i, 3 );

            QList<QWidget*> l;

            l.append( jobidLabel );
            l.append( jobStatusLabel );
            l.append( loadJobidButton );
            l.append( removeJobidButton );
            jobMap[ jobids.at( i ).toInt() ] = l;
        }
        connect( jobidGroup, SIGNAL( buttonClicked( QAbstractButton* ) ), SLOT( onLoadJobButtonClicked( QAbstractButton* ) ) );
        connect( removeJobidGroup, SIGNAL( buttonClicked( QAbstractButton* ) ), SLOT( onRemoveJobButtonClicked( QAbstractButton* ) ) );

        QVBoxLayout* jobWidgetLayout = new QVBoxLayout;
        jobWidgetLayout->addWidget( jobLabel );
        jobWidgetLayout->addLayout( jobLayout );
        jobWidgetLayout->addStretch( 0 );
        jobWidget->setLayout( jobWidgetLayout );
    }
    else
    {
        loadButtonLayout->addStretch( 0 );
    }

    versionWidget = new QWidget;
    versionWidget->setLayout( scorePVersionsBaseLayout );
    versionWidget->setVisible( false );

    QVBoxLayout* proceedButtonLayout = new QVBoxLayout;
    proceedButtonLayout->addWidget( proceedButton );
    proceedButtonLayout->addWidget( helpButton );
    proceedButtonLayout->addStretch( 0 );
    proceedButtonWidget = new QWidget;
    proceedButtonWidget->setLayout( proceedButtonLayout );
    proceedButtonWidget->setVisible( false );

    mainWidget = new QWidget;
    mainWidget->setLayout( mainLayout );
    layout->addWidget( mainWidget );
    layout->addWidget( environmentWidget );
    layout->addWidget( loadButtonWidget );
    layout->addWidget( jobWidget );
    layout->addWidget( versionWidget );
    layout->addWidget( proceedButtonWidget, 0, Qt::AlignBottom );
    setLayout( layout );
}

/**
 * @brief The settings of the measurement with the job id associated with the specified button are loaded.
 * @param button The button is part of a QButtonGroup and the id of the button is the associated job id.
 */
void
SettingTab::onLoadJobButtonClicked( QAbstractButton* button )
{
    measurementWindow->loadSettings( QString::number( jobidGroup->id( button ) ) );
}

/**
 * @brief Deletes settings for the corresponding job id and removes the option to load this job from the
 * graphical user interface.
 * @param button The button is part of a QButtonGroup and the id of the button is the associated job id.
 */
void
SettingTab::onRemoveJobButtonClicked( QAbstractButton* button )
{
    measurementWindow->deleteSettings( QString::number( removeJobidGroup->id( button ) ) );
    const QList<QWidget*> l = jobMap.value( removeJobidGroup->id( button ) );
    for ( QWidget* w : l )
    {
        w->setVisible( false );
    }
}

/**
 * @brief A new measurement is started instead of loading the settings of a former measurement.
 */
void
SettingTab::onNewMeasurementButtonClicked()
{
    // check if matching Score-P version is already in path
    checkPath();
    versionWidget->setVisible( true );
    proceedButtonWidget->setVisible( true );
    loadButtonWidget->setVisible( false );
    jobWidget->setVisible( false );
    environmentWidget->setVisible( true );
}

/**
 * @brief The settings of the recent measurement are loaded.
 */
void
SettingTab::onLoadMeasurementButtonClicked()
{
    if ( !measurementWindow->loadSettings() )
    {
        QString message = tr( "Recent measurement could not be loaded. Please start a new measurement" );
        QMessageBox::information( this, tr( "Error" ), message );
        onNewMeasurementButtonClicked();
    }
}

/**
 * @brief Adjusts layout depending on the return value of the findScoreP function which is called by
 * this function.
 */
void
SettingTab::onFindScorePButtonClicked()
{
    // uncheck Score-P version in path
    uncheck( inPathButtonGroup );
    proceedButton->setEnabled( false );
    proceedButton->setIcon( QIcon() );
    proceedButton->setToolTip( tr( "Continue with instrumentation and use selected Score-P version" ) );
    errorMsg->setVisible( false );

    // no Score-P version found
    if ( !findScorep() )
    {
        errorMsg->setText( tr( "<font color=\"black\">No Score-P version was autodetected. You can browse your directory to manually find a Score-P version.</font>" ) );
        errorMsg->setVisible( true );
        if ( !moduleAvail )
        {
            errorMsg->setText( tr( "<font color=\"red\">Score-P could not be autodetected. Please make sure you have an installed Score-P version.</font><br>"
                                   "<font color=\"black\">If module is available in your shell you can execute 'module load Score-P[/$VERSION]' and restart the plugin.</font><br>"
                                   "<font color=\"black\">You can browse your directory to manually find a Score-P version.</font>" ) );
        }
    }
    else
    {
        browseScorepButton->setText( tr( "Browse other Score-P version (manual search)" ) );
    }

    findScorePButton->setVisible( false );
    browseScorepButton->setVisible( true );
}

/**
 * @brief The selected Score-P version is saved in the settings and the proceed method called.
 */
void
SettingTab::onProceedButtonClicked()
{
    // use former configuration
    if ( selectPath == InPathVariable )
    {
        expandPath( formerConfigurationPath );
        measurementWindow->deleteSettings();
        measurementWindow->settings.setValue( "measurement/loadScoreP", "Path" );
        measurementWindow->settings.setValue( "measurement/path", formerConfigurationPath );
        proceed();
    }

    if ( selectPath == BrowsePath )
    {
        expandPath( browseDirPath );
        measurementWindow->deleteSettings();
        measurementWindow->settings.setValue( "measurement/loadScoreP", "Path" );
        measurementWindow->settings.setValue( "measurement/path", browseDirPath );
        proceed();
    }

    if ( selectPath == Module )
    {
        int           returncode;
        QRadioButton* checked = static_cast<QRadioButton*>( moduleButtonGroup->checkedButton() );
        QString       version = checked->text().remove( " (module)" );

        // load Score-P
        string cmd = "module load " + version.toStdString();
        console->execCommand( cmd, returncode, true, true );

        if ( returncode != 0 )
        {
            errorMsg->setText( tr( "<font color=\"red\">An error occurred while loading the module.</font>" ) );
        }
        else
        {
            measurementWindow->deleteSettings();
            // save settings
            measurementWindow->settings.setValue( "measurement/loadScoreP", "Module" );
            measurementWindow->settings.setValue( "measurement/moduleCmd", QString::fromStdString( cmd ) );
            proceed();
        }
    }
}

/**
 * @brief A file directory is opened in which the user can specify a folder containing a Score-P installation.
 * The isValidPath method is called to check if it contains a working Score-P installation. Corresponding
 * information is displayed.
 */
void
SettingTab::onBrowseScorePButtonClicked()
{
    // uncheck all other options
    uncheck( inPathButtonGroup );
    uncheck( moduleButtonGroup );

    QString path = QFileDialog::getExistingDirectory( this, tr( "Select bin directory of your Score-P installation" ), QString::fromStdString( getenv( "HOME" ) ), QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks );

    if ( isValidPath( path ) )
    {
        path = getRealPath( path );
        string tmp     = "Score-P version was built with " + getCompiler( path ) + " and " + getMpi( path );
        char*  tooltip = const_cast<char*>( tmp.c_str() );
        inPathUsableInfo->setToolTip( tooltip );
        browseDirLabel->setToolTip( toolTip() );
        if ( isMatching( path ) == Matching )
        {
            browseDirLabel->setText( tr( "<font color=\"black\">selected directory: " ) + path + "</font><br>"
                                     + tr( "<font color=\"green\">Your selected directory contains a matching Score-P installation." ) );
            proceedButton->setEnabled( true );
        }
        else if ( isMatching( path ) == PossiblyMatching )
        {
            browseDirLabel->setText( tr( "<font color=\"black\">selected directory: " ) + path + "</font><br>"
                                     + tr( "<font color=\"orange\">Your selected directory may contain a not matching Score-P installation." ) );
            proceedButton->setEnabled( true );
        }
        else
        {
            browseDirLabel->setText( tr( "<font color=\"black\">selected directory: " ) + path + "</font><br>"
                                     + tr( "<font color=\"red\">Your selected directory contains a not matching Score-P installation." ) );
        }
        selectPath    = BrowsePath;
        browseDirPath = path;
    }
    else
    {
        browseDirLabel->setText( tr( "<font color=\"black\">selected directory: " ) + path + "</font><br>"
                                 + tr( "<font color=\"red\">Your selected directory doesn't contain a working Score-P installation." ) );
        browseDirLabel->setToolTip( QString() );
        proceedButton->setEnabled( false );
    }
    browseDirLabel->setVisible( true );
}

/**
 * @brief A short information what happens in this step and how possible problems can be solved is displayed.
 */
void
SettingTab::onHelpButtonClicked()
{
    QString message = tr( "Trying to find a Score-P installation. Your PATH variable will be updated to include the bin directory of Score-P. "
                          "If Score-P is not found, you can manually extend your PATH variable with the path to Score-P and then restart the plugin. "
                          "This can be done either with <br> (1) export PATH=$SCOREP_ROOT/bin:$PATH <br>or<br> (2) module load Score-P" );
    QMessageBox::information( this, tr( "Help" ), message );
}

/**
 * @brief Handles inPathButton selection, which indicates that the user has selected a Score-P version
 * that was found in PATH. Checks if the selected path is a matching Score-P installation. Updates the
 * state of the proceedButton and sets selectPath to InPathVariable.
 */
void
SettingTab::inPathSelected()
{
    // uncheck other options
    uncheck( moduleButtonGroup );

    QString path;
    path = inPathButton->text().remove( tr( " (already in $PATH)" ) );

    if ( isMatching( path ) == Matching )
    {
        proceedButton->setEnabled( true );
        proceedButton->setIcon( QIcon() );
        proceedButton->setToolTip( tr( "Continue with instrumentation and use selected Score-P version" ) );
        formerConfigurationPath = path;
        selectPath              = InPathVariable;
        proceedButton->setText( tr( "Proceed" ) );
    }
    else if ( isMatching( path ) == PossiblyMatching )
    {
        proceedButton->setIcon( infoIcon );
        if ( getMpi( path ) == "mpich2 (default)" && getCompiler( path ) == "gcc (default)" )
        {
            proceedButton->setToolTip( tr( "Your selected configuration may contain a not matching Score-P installation. \nMpi and compiler were not specified in the configuration and are suggested as default.\nPress proceed to continue at your own risk." ) );
        }
        else if ( getMpi( path ) == "mpich2 (default)" )
        {
            proceedButton->setToolTip( tr( "Your selected configuration may contain a not matching Score-P installation. \nMpi was not specified in the configuration and is suggested as default.\nPress proceed to continue at your own risk." ) );
        }
        else
        {
            proceedButton->setToolTip( tr( "Your selected configuration may contain a not matching Score-P installation. \nCompiler was not specified in the configuration and is suggested as default.\nPress proceed to continue at your own risk." ) );
        }

        proceedButton->setEnabled( true );
        formerConfigurationPath = path;
        selectPath              = InPathVariable;
    }
    else
    {
        proceedButton->setEnabled( false );
        proceedButton->setIcon( QIcon() );
    }
}

/**
 * @brief Updates the state of the proceedButton when moduleButton is selected, and sets selectPath to Module.
 */
void
SettingTab::moduleSelected()
{
    // uncheck other options
    uncheck( inPathButtonGroup );
    proceedButton->setEnabled( true );
    proceedButton->setIcon( QIcon() );
    proceedButton->setToolTip( tr( "Continue with instrumentation and use selected Score-P version" ) );
    selectPath = Module;
}

/**
 * @brief Expands the system PATH environment variable to include the given path.
 * @param path The path to be added to the system PATH environment variable.
 * @param saveSettings Whether to save the output to the command history.
 */
void
SettingTab::expandPath( QString path, bool saveSettings )
{
    string tmp_cmd = string( getenv( "PATH" ) ) + ":" + path.toStdString();
    char*  cmd     = const_cast<char*>( tmp_cmd.c_str() );
    if ( saveSettings )
    {
        console->addCommand( "export PATH=" + path.toStdString() + ":$PATH" );
    }
    setenv( "PATH", cmd, 1 );
}

/**
 * @brief Removes a path from the environment variable PATH.
 * @param removePath The path to be removed from PATH.
 */
void
SettingTab::removeFromPath( QString removePath )
{
    QString path = QString::fromStdString( getenv( "PATH" ) );
    path.remove( ":" + removePath );
    path.remove( removePath );
    string tmp = path.toStdString();
    char*  cmd = const_cast<char*>( tmp.c_str() );
    setenv( "PATH", cmd, 1 );
}

/**
 * @brief Enables the instrumentation tab, changes the current tab to the instrumentation tab, removes the setup tab,
 * and calls setConfigVars method.
 */
void
SettingTab::proceed()
{
    // enable instrumentation tab
    measurementWindow->tabWidget->setTabEnabled( 1, true );

    // change current tab into instrumentation tab
    measurementWindow->tabWidget->setCurrentIndex( 1 );

    // set up tab is no longer needed
    measurementWindow->tabWidget->removeTab( 0 );

    // save available config vars
    setConfigVars();
}

/**
 * @brief Checks if Score-P is available via Path and updates the user interface accordingly.
 */
void
SettingTab::checkPath()
{
    // check if Score-P is available via Path
    int     returncode;
    QString path = QString::fromStdString( console->execCommand( "which scorep-info", returncode, false ) ).remove( REGULAR_EXPRESSION( "\\n" ) );
    path.remove( "/scorep-info" );

    if ( isValidPath( path ) )
    {
        inPathUsableInfo->setVisible( true );
        inPathUsableConf->setVisible( true );
        foundScorePsLabel->setVisible( true );
        inPathButton->setText( path + tr( " (already in $PATH)" ) );
        string tmp     = "Score-P version was built with " + getCompiler( path ) + " and " + getMpi( path );
        char*  tooltip = const_cast<char*>( tmp.c_str() );
        inPathUsableInfo->setToolTip( tooltip );
        inPathUsableConf->setToolTip( tooltip );
        if ( isMatching( path ) == Matching )
        {
            inPathUsableConf->setText( tr( "<font color=\"green\">usable configuration</font>" ) );
            inPathButton->setStyleSheet( "color: black" );
        }
        else if ( isMatching( path ) == PossiblyMatching )
        {
            inPathUsableConf->setText( tr( "<font color=\"orange\">possibly usable configuration</font>" ) );
            inPathButton->setStyleSheet( "color: black" );
        }
        else
        {
            inPathUsableConf->setText( tr( "<font color=\"red\">not usable configuration</font>" ) );
            inPathButton->setStyleSheet( "color: grey" );
            inPathButton->setChecked( "false" );
        }

        inPathPapiLabel->setVisible( true );
        if ( papiSupport( path ) )
        {
            inPathPapiInfo->setText( tr( "<font color=\"green\">yes</font>" ) );
        }
        else
        {
            inPathPapiInfo->setText( tr( "<font color=\"red\">no</font>" ) );
        }
        inPathPapiInfo->setVisible( true );

        inPathUnwindedLabel->setVisible( true );
        if ( unwindingSupport( path ) )
        {
            inPathUnwindedInfo->setText( tr( "<font color=\"green\">yes</font>" ) );
        }
        else
        {
            inPathUnwindedInfo->setText( tr( "<font color=\"red\">no</font>" ) );
        }
        inPathUnwindedInfo->setVisible( true );

        path = getRealPath( path );
        if ( minimumWidth < inPathButton->sizeHint().width() + inPathUsableInfo->sizeHint().width() + 45 + inPathUsableConf->sizeHint().width() + inPathPapiLabel->sizeHint().width() + inPathPapiLabel->sizeHint().width() + inPathUnwindedLabel->sizeHint().width() + inPathUnwindedInfo->sizeHint().width() )
        {
            minimumWidth = inPathButton->sizeHint().width() + inPathUsableConf->sizeHint().width() + 45 + inPathUsableInfo->sizeHint().width() + inPathPapiLabel->sizeHint().width() + inPathPapiLabel->sizeHint().width() + inPathUnwindedLabel->sizeHint().width() + inPathUnwindedInfo->sizeHint().width(); // There was added space of size 45
            scorePVersionsLabel->setMinimumWidth( minimumWidth + 2 * scorePVersionsLayoutMargin );
        }
        if ( !inPathButton->isVisible() )
        {
            inPathButton->setVisible( true );
            scorePVersionsLabel->setVisible( true );
            minimumHeight += inPathButton->sizeHint().height();
            scorePVersionsLabel->setFixedHeight( minimumHeight + 2 * scorePVersionsLayoutMargin );
            scorePVersionsLabel->setMinimumWidth( minimumWidth + 2 * scorePVersionsLayoutMargin );
            inPathButton->setChecked( true );
        }
        string cmd = path.remove( REGULAR_EXPRESSION( "\\n" ) ).toStdString() + "/scorep-info config-summary";

        if ( inPathButton->toolTip() == "" )
        {
            inPathButton->setToolTip( cutAfterLine( QString::fromStdString( console->execCommand( cmd, returncode, true, false, false ) ), 12 ) + tr( "Read more ..." ) );
        }
        inPathButtonGroup->addButton( inPathButton, 0 );
        connect( inPathButton, SIGNAL( clicked() ), this, SLOT( inPathSelected() ) );
        findScorePButton->setText( tr( "Find other Score-P versions (automatic search)" ) );

        if ( inPathButton->isChecked() )
        {
            inPathSelected();
        }
    }
    else
    {
        inPathButton->setChecked( "false" );
        inPathButton->setVisible( false );
        if ( inPathButtonGroup->buttons().contains( inPathButton ) )
        {
            inPathButtonGroup->removeButton( inPathButton );
        }
        inPathUsableInfo->setVisible( false );
        findScorePButton->setText( tr( "Find Score-P (automatic search)" ) );
    }
}

/**
 * @param path The path to check.
 * @return True if the path contains a valid Score-P installation, false otherwise.
 */
bool
SettingTab::isValidPath( QString path )
{
    string tmp_cmd = path.toStdString() + "/scorep-info config-summary 2> /dev/null > /dev/null";
    char*  cmd     = const_cast<char*>( tmp_cmd.c_str() );

    // NOLINTNEXTLINE (suppress code checker warning)
    if ( system( cmd ) == 0 ) // alternative perhaps execve
    {
        return true;
    }
    return false;
}

/**
 * @brief Checks if PAPI support is available for the specified path to a Score-P installation.
 * @param path The Path to check for PAPI support.
 * @return True if PAPI support is available, false otherwise.
 */
bool
SettingTab::papiSupport( QString path )
{
    int     returncode;
    QString configInfo = QString::fromStdString( console->execCommand( path.toStdString() + "/scorep-info config-summary", returncode, false ) );
    if ( configInfo.contains( "PAPI support" ) )
    {
        QString line = cutAfterLine( configInfo.split( "PAPI support" ).at( 1 ), 1 );
        if ( line.contains( "yes" ) )
        {
            return true;
        }
    }
    return false;
}

/**
 * @brief Checks if Unwinding support is available for the specified path to a Score-P installation.
 * @param path The Path to check for Unwinding support.
 * @return True if Unwinding support is available, false otherwise.
 */
bool
SettingTab::unwindingSupport( QString path )
{
    int     returncode;
    QString configInfo = QString::fromStdString( console->execCommand( path.toStdString() + "/scorep-info config-summary", returncode, false ) );
    if ( configInfo.contains( "Unwinding support" ) )
    {
        QString line = cutAfterLine( configInfo.split( "Unwinding support" ).at( 1 ), 1 );
        if ( line.contains( "yes" ) )
        {
            return true;
        }
    }
    return false;
}

/**
 * @brief Searches for Score-P modules and lists all available versions by parsing the output of "module avail" command.
 * The function also sets the tooltip for each module with information about the compiler and MPI version used to build
 * the Score-P module. Layout is adapted depending on the return value.
 * @return True if at least one Score-P module is found, false otherwise.
 */
bool
SettingTab::findScorep()
{
    int  returncode;
    bool ret = false;
    errorMsg->setVisible( false );

    // check if Score-P module is available
    QStringList moduleLines = QString::fromStdString( console->execCommand( "module avail", returncode, true, true ) ).split( REGULAR_EXPRESSION( "\\n" ) );
    if ( returncode == 0 )
    {
        moduleAvail = true;
        QStringList allModules;
        foreach( const auto & m, moduleLines )
        {
            allModules.append( m.split( " " ) );
        }

        QStringList scorepModules;
        for ( auto& m : allModules )
        {
            if ( regex_search( m.toStdString(), regex( "([^ ]*)Score-P([^ ]*)" ) ) )
            {
                // remove default flag to get the name with which the command "module load" can be executed
                m.remove( "(default)" );

                scorepModules.append( m );
            }
        }

        // list modules
        int index = 0;
        for ( const auto& m : scorepModules )
        {
            QRadioButton* button = new QRadioButton( m + " (module)" );
            button->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
            connect( button, SIGNAL( clicked() ), this, SLOT( moduleSelected() ) );
            moduleButtonGroup->addButton( button, index );

            QHBoxLayout* moduleButtonLayout = new QHBoxLayout;
            moduleButtonLayout->addWidget( button );
            QLabel* papiLabel     = new QLabel( "PAPI:" );
            QLabel* papiAvail     = new QLabel;
            QLabel* unwindedLabel = new QLabel( "libunwinded:" );
            QLabel* unwindedAvail = new QLabel;

            console->execCommand( "module load " + m.toStdString(), returncode, false, true );
            QString scorepPath = QString::fromStdString( console->execCommand( "which scorep-info", returncode, false ) ).remove( REGULAR_EXPRESSION( "\\n" ) );
            scorepPath.remove( "/scorep-info" );
            if ( papiSupport( scorepPath ) )
            {
                papiAvail->setText( tr( "<font color=\"green\">yes</font>" ) );
            }
            else
            {
                papiAvail->setText( tr( "<font color=\"red\">no</font>" ) );
            }

            if ( unwindingSupport( scorepPath ) )
            {
                unwindedAvail->setText( tr( "<font color=\"green\">yes</font>" ) );
            }
            else
            {
                unwindedAvail->setText( tr( "<font color=\"red\">no</font>" ) );
            }

            // Add tooltip containing mpi and compiler version
            string tmp     = "Score-P version was built with " + getCompiler( scorepPath ) + " and " + getMpi( scorepPath );
            char*  tooltip = const_cast<char*>( tmp.c_str() );
            button->setToolTip( tooltip );

            moduleButtonLayout->addSpacing( 10 );
            moduleButtonLayout->addWidget( papiLabel );
            moduleButtonLayout->addSpacing( 5 );
            moduleButtonLayout->addWidget( papiAvail );
            moduleButtonLayout->addSpacing( 10 );
            moduleButtonLayout->addWidget( unwindedLabel );
            moduleButtonLayout->addSpacing( 5 );
            moduleButtonLayout->addWidget( unwindedAvail );
            moduleButtonLayout->addStretch( 0 );
            moduleLayout->addLayout( moduleButtonLayout );

            if ( minimumWidth < button->sizeHint().width() )
            {
                minimumWidth = button->sizeHint().width();
            }

            minimumHeight += button->sizeHint().height();
            scorePVersionsLabel->setFixedHeight( minimumHeight + 2 * scorePVersionsLayoutMargin );
            scorePVersionsLabel->setMinimumWidth( minimumWidth + 2 * scorePVersionsLayoutMargin );
        }
        if ( !scorepModules.empty() )
        {
            foundScorePsLabel->setVisible( true );
            scorePVersionsLabel->setVisible( true );
            ret = true;
        }
    }

    findScorePButton->setVisible( false );
    browseScorepButton->setVisible( true );

    return ret;
}

/**
 * @brief Checks if the MPI and compiler used to build the given Score-P installation matches the selected MPI and compiler.
 * @param path The path of the Score-P intallation to check.
 * @return An integer indicating whether the installation matches the selected MPI and compiler or not. Possible values are
 * Matching, PossiblyMatching, and NotMatching.
 */
int
SettingTab::isMatching( QString path )
{
    int returncode;
    if ( isValidPath( path ) )
    {
        QString compiler;
        QString mpi;
        QString configInfo = QString::fromStdString( console->execCommand( path.toStdString() + "/scorep-info config-summary", returncode, false ) );
        if ( configInfo.contains( "-compiler-suite=" ) )
        {
            compiler = cutAfterLine( configInfo.split( "-compiler-suite=" ).at( 1 ), 1 ).split( "'" ).at( 0 );
        }
        else
        {
            // gcc set as default -> check system default
            compiler = "gcc";
        }

        if ( configInfo.contains( "--with-mpi=" ) )
        {
            mpi = cutAfterLine( configInfo.split( "--with-mpi=" ).at( 1 ), 1 ).split( "'" ).at( 0 );
        }
        else
        {
            // mpich2 set as default
            mpi = "mpich2";
        }
        if ( ( getMpi( path ) == "mpich2 (default)" && getCompiler( path ) == "gcc (default)" ) || ( mpiComboBox->currentText() == mpi && getCompiler( path ) == "gcc (default)" ) || ( getMpi( path ) == "mpich2 (default)" && compilerComboBox->currentText() == compiler ) )
        {
            return PossiblyMatching;
        }
        if ( compilerComboBox->currentText() == compiler && mpiComboBox->currentText() == mpi )
        {
            return Matching;
        }
    }
    return NotMatching;
}

/**
 * @brief Cuts text after a given line number. Used to show a preview of the output from the `scorep-info config-summary` command.
 * @param text The text to be processed.
 * @param line The line number to cut after.
 * @return The processed QString with text after the given line number removed
 */
QString
SettingTab::cutAfterLine( QString text, int line )
{
    int     lineat = 0;
    QString ret;
    for ( int i = 0; i < text.length(); i++ )
    {
        if ( text.at( i ) == '\n' )
        {
            lineat++;
            ret += "<br>";
        }
        else
        {
            ret += text.at( i );
        }
        if ( lineat == line )
        {
            return ret;
        }
    }
    return ret;
}

/**
 * @brief Returns the real path of the given path by calling the `realpath` command to avoid problems caused by the use of
 * aliases in further use.
 * @param path The path for which to retrieve the real path.
 * @return QString containing the real path pf the given path.
 */
QString
SettingTab::getRealPath( QString path )
{
    int    returncode;
    string cmd = "realpath " + path.toStdString();
    path = QString::fromStdString( console->execCommand( cmd, returncode, false ) );
    return path;
}

/**
 * @brief Gets the compiler used for building the Score-P installation at the specified path.
 * @param path The path to the Score-P installation.
 * @return The compiler used for building the Score-P installation as a std::string.
 */
string
SettingTab::getCompiler( QString path )
{
    int     returncode;
    QString compiler;
    QString configInfo = QString::fromStdString( console->execCommand( path.toStdString() + "/scorep-info config-summary", returncode, false ) );
    if ( configInfo.contains( "-compiler-suite=" ) )
    {
        compiler = cutAfterLine( configInfo.split( "-compiler-suite=" ).at( 1 ), 1 ).split( "'" ).at( 0 );
    }
    else
    {
        compiler = "gcc (default)";
    }
    return compiler.toStdString();
}

/**
 * @brief Gets the MPI used for building the Score-P installation at the specified path.
 * @param path The path to the Score-P installation.
 * @return The MPI used for building the Score-P installation as a std::string.
 */
string
SettingTab::getMpi( QString path )
{
    int     returncode;
    QString mpi;
    QString configInfo = QString::fromStdString( console->execCommand( path.toStdString() + "/scorep-info config-summary", returncode, false ) );
    if ( configInfo.contains( "--with-mpi=" ) )
    {
        mpi = cutAfterLine( configInfo.split( "--with-mpi=" ).at( 1 ), 1 ).split( "'" ).at( 0 );
    }
    else
    {
        mpi = "mpich2 (default)";
    }
    return mpi.toStdString();
}

/**
 * @brief Sets the layout.
 */
InstrumentationTab::InstrumentationTab( ContextFreeServices* service, Console* console, QWidget* parent ) : MeasurementTab( service, console, parent )
{
    mainLayout     = new QVBoxLayout;
    makefileLayout = new QVBoxLayout;
    layout         = new QVBoxLayout;

    QHBoxLayout* selectExecutableButtonLayout = new QHBoxLayout;
    QVBoxLayout* selectExecutableLayout       = new QVBoxLayout;
    QVBoxLayout* stepsLayout                  = new QVBoxLayout;

    QLabel* stepsLabel = new QLabel( tr( "<b>This wizard will help you to prepare the instrumentation of your program</b> "
                                         "<br>The following steps can be gone through"
                                         "<br>\u27a4 select your executable file<br>\u27a4 decide whether a new instrumentation should be created or "
                                         "the existing one should be used (optional) <br>\u27a4 select how your application was built "
                                         "<br>\u27a4 adjust build settings <br>\u27a4 rebuild your application" ) );
    QWidget* stepsWidget = new QWidget;
    stepsWidget->setLayout( stepsLayout );
    stepsLayout->addWidget( stepsLabel );
    mainLayout->addWidget( stepsWidget );

    mainLayout->addSpacing( 20 );

    selectExecutableButton = new QPushButton( tr( "Browse executable file" ) );
    selectExecutableButton->setToolTip( tr( "It must be ensured that the project can be built and is executable." ) );
    connect( selectExecutableButton, SIGNAL( clicked() ), this, SLOT( onSelectExecutableButtonClicked() ) );

    // dirLabel gets visible when a directory is selected
    fileLabel = new QLabel( tr( "You selected " ) + executableName );
    fileLabel->setVisible( false );
    fileLabel->setWordWrap( true );

    // error message if user doesn't select a executable file
    selectExecutableErrorMessage = new QLabel();
    selectExecutableErrorMessage->setStyleSheet( "color: red" );
    selectExecutableErrorMessage->setWordWrap( true );
    selectExecutableErrorMessage->setVisible( false );

    selectExecutableLayout->addWidget( selectExecutableErrorMessage );
    selectExecutableButtonLayout->addWidget( fileLabel );
    selectExecutableButtonLayout->addWidget( selectExecutableButton );
    selectExecutableLayout->addLayout( selectExecutableButtonLayout );

    selectExecutableWidget = new QWidget;
    selectExecutableWidget->setLayout( selectExecutableLayout );
    mainLayout->addWidget( selectExecutableWidget );
    mainLayout->addSpacing( 10 );

    // create new or use former instrumentation layout
    QVBoxLayout* selectInstrumentationLayout = new QVBoxLayout;
    selectInstrumentationLabel = new QLabel;
    selectInstrumentationLabel->setWordWrap( true );
    selectInstrumentationBox = new QGroupBox;
    QVBoxLayout* selectInstrumentationButtonLayout = new QVBoxLayout;

    formerInstrumentationButton = new QRadioButton( tr( "use former instrumentation" ) );
    newInstrumentationButton    = new QRadioButton( tr( "prepare new instrumentation" ) );

    selectInstrumentationButtonLayout->addWidget( formerInstrumentationButton );
    selectInstrumentationButtonLayout->addWidget( newInstrumentationButton );
    selectInstrumentationBox->setLayout( selectInstrumentationButtonLayout );
    selectInstrumentationBox->setMinimumSize( selectInstrumentationBox->sizeHint() );

    selectInstrumentationWidget = new QWidget;
    selectInstrumentationLayout->addWidget( selectInstrumentationLabel );
    selectInstrumentationLayout->addWidget( selectInstrumentationBox );
    selectInstrumentationWidget->setLayout( selectInstrumentationLayout );
    mainLayout->addWidget( selectInstrumentationWidget );
    selectInstrumentationWidget->setVisible( false );

    // select makefile/ cmake ... layout
    QVBoxLayout* selectBuildLayout = new QVBoxLayout;
    selectMakeLabel = new QLabel( tr( "<br>Please select if your application is build with makefile or CMake.</br> You either have to edit your makefile or use a wrapper script." ) );
    selectMakeLabel->setWordWrap( true );
    selectMakeBox = new QGroupBox;
    QVBoxLayout* selectMakeLayout = new QVBoxLayout;

    useMakefile = new QRadioButton( tr( "adjust Makefile" ) );
    useCMake    = new QRadioButton( tr( "CMake" ) );
    useCMake->setEnabled( false );

    selectMakeLayout->addWidget( useMakefile );
    selectMakeLayout->addWidget( useCMake );
    selectMakeBox->setLayout( selectMakeLayout );
    selectMakeBox->setMinimumSize( selectMakeBox->sizeHint() );

    selectBuildWidget = new QWidget;
    selectBuildLayout->addWidget( selectMakeLabel );
    selectBuildLayout->addWidget( selectMakeBox );
    selectBuildWidget->setLayout( selectBuildLayout );
    selectBuildWidget->setVisible( false );
    mainLayout->addWidget( selectBuildWidget );

    // adjust makefile layout
    QHBoxLayout* adjustMakefileButtonLayout = new QHBoxLayout;
    browseMakefileButton   = new QPushButton( tr( "Browse makefile" ) );
    detectedMakefileButton = new QPushButton( tr( "Open detected makefile" ) );

    adjustMakefileButtonLayout->addWidget( browseMakefileButton );
    adjustMakefileButtonLayout->addWidget( detectedMakefileButton );

    selectMakefileWidget = new QWidget;
    selectMakefileWidget->setLayout( adjustMakefileButtonLayout );
    selectMakefileWidget->setVisible( false );
    mainLayout->addWidget( selectMakefileWidget );
    mainLayout->addSpacing( 10 );

    connect( newInstrumentationButton, SIGNAL( clicked() ), this, SLOT( selectedNewInstrumentation() ) );
    connect( formerInstrumentationButton, SIGNAL( clicked() ), this, SLOT( selectedFormerInstrumentation() ) );
    connect( useMakefile, SIGNAL( clicked() ), this, SLOT( selectedAdjustMakefile() ) );
    connect( browseMakefileButton, SIGNAL( clicked() ), this, SLOT( onBrowseMakefileButtonClicked() ) );
    connect( detectedMakefileButton, SIGNAL( clicked() ), this, SLOT( onDetectedMakefileButtonClicked() ) );

    QHBoxLayout* makefileButtonLayout     = new QHBoxLayout;
    QLabel*      makefileHintsHeaderLabel = new QLabel( tr( "<b>You need to apply the following changes to use the Score-p wrappers:</b>" ) );
    QLabel*      makefileHintsTextLabel   = new QLabel( tr( "Replace the application's compiler and linker with the corresponding wrapper at configuration time so that they will be used at build time. "
                                                            "As compiler and linker commands are usually assigned to build variables like CC, CXX or F77, using the corresponding wrapper is as simple as prefixing the value with \"scorep-\". For example" ) );
    QLabel* makefileHintsExamplesLabel = new QLabel( tr( "CC=mpicc \u2794 CC=scorep-mpicc <br>CC=gcc \u2794 CC=scorep-gcc <br>CXX=g++ \u2794 CXX=scorep-g++" ) );
    QLabel* userGuideLabel             = new QLabel( tr( "For more information please see <a href=\"https://perftools.pages.jsc.fz-juelich.de/cicd/scorep/tags/scorep-7.1/html/instrumentation.html\">Score-P user guide</a>" ) );
    userGuideLabel->setOpenExternalLinks( true );
    makefileHintsTextLabel->setWordWrap( true );

    makefileWidget = new QWidget;
    makefileLayout->addWidget( makefileHintsHeaderLabel );
    makefileLayout->addSpacing( 5 );
    makefileLayout->addWidget( makefileHintsTextLabel );
    makefileLayout->addSpacing( 5 );
    makefileLayout->addWidget( makefileHintsExamplesLabel );
    makefileLayout->addSpacing( 5 );
    makefileLayout->addWidget( userGuideLabel );
    makefileEdit         = new QTextEdit;
    discardChangesButton = new QPushButton( tr( "Discard changes" ) );
    saveChangesButton    = new QPushButton( tr( "Save changes" ) );
    makefileButtonLayout->addWidget( discardChangesButton );
    makefileButtonLayout->addWidget( saveChangesButton );
    makefileLayout->addWidget( makefileEdit );
    makefileLayout->addLayout( makefileButtonLayout );

    makefileWidget->setLayout( makefileLayout );
    makefileWidget->setVisible( false );

    connect( saveChangesButton, SIGNAL( clicked() ), this, SLOT( onSaveChangesButtonClicked() ) );
    connect( discardChangesButton, SIGNAL( clicked() ), this, SLOT( onDiscardChangesClicked() ) );

    // build application layout
    QVBoxLayout* buildLayout = new QVBoxLayout;
    buildLabel = new QLabel( tr( "Your application is now ready to be instrumented. In order to instrument the application, it must be rebuild with the adjusted makefile. To do this, please enter your build command and confirm by pressing build your application." ) );
    buildLabel->setWordWrap( true );
    buildButton = new QPushButton( tr( "Build your application" ) );
    QHBoxLayout* buildCommandLayout = new QHBoxLayout;
    buildCommandEdit = new QLineEdit;
    buildCommandLayout->addWidget( buildCommandEdit );
    buildCommandLayout->addWidget( buildButton );
    buildLayout->addWidget( buildLabel );
    buildLayout->addLayout( buildCommandLayout );
    buildWidget = new QWidget;
    buildWidget->setLayout( buildLayout );
    buildWidget->setVisible( false );
    mainLayout->addWidget( buildWidget );

    QVBoxLayout* analysisLayout = new QVBoxLayout;
    analysisLabel = new QLabel;
    analysisLabel->setVisible( false );
    analysisLabel->setWordWrap( true );
    analysisButton = new QPushButton( tr( "Continue with analysis" ) );
    analysisButton->setVisible( false );
    analysisLayout->addWidget( analysisLabel );
    analysisLayout->addWidget( analysisButton );
    analysisWidget = new QWidget;
    analysisWidget->setLayout( analysisLayout );
    mainLayout->addWidget( analysisWidget );
    mainLayout->addSpacing( 20 );

    connect( analysisButton, SIGNAL( clicked() ), this, SLOT( onAnalysisButtonClicked() ) );
    connect( buildButton, SIGNAL( clicked() ), this, SLOT( onBuildButtonClicked() ) );

    mainLayout->addStretch( 0 );
    mainWidget = new QWidget;
    mainWidget->setLayout( mainLayout );

    layout->addWidget( mainWidget );
    layout->addWidget( makefileWidget );

    setLayout( layout );
}

void
InstrumentationTab::onDiscardChangesClicked()
{
    makefileWidget->setVisible( false );
    mainWidget->setVisible( true );
}

/**
 *  @brief Saves the edited makefile.
 *  @details This slot function is responsible for saving the edited makefile to the opened file, hiding the makefile editing widget,
 *  and showing the main widget and the build step.
 *  @pre A makefile must be edited and opened.
 *  @post The edited makefile is saved to the opened file, the makefile editing widget is hidden, and the build step is shown on the
 *  main widget.
 */
void
InstrumentationTab::onSaveChangesButtonClicked()
{
    QFile file( openedFile );
    file.open( QIODevice::WriteOnly );
    QTextStream stream( &file );
    stream << makefileEdit->toPlainText();

    makefileWidget->setVisible( false );
    mainWidget->setVisible( true );
    buildWidget->setVisible( true );

    measurementWindow->settings.setValue( "measurement/adjustMakefile", "True" );
}

/**
 * @brief Slot function that is called when the user selects the option to adjust the Makefile. Checks if a makefile can
 * be detected in the executable's directory.
 */
void
InstrumentationTab::selectedAdjustMakefile()
{
    // check if makefile can be detected in build directory
    QStringList dirNames = executableName.split( "/" );
    detectedMakefile = executableDir;
    detectedMakefileButton->setVisible( false );

    if ( QFile::exists( detectedMakefile + "/Makefile" ) )
    {
        detectedMakefile += "/Makefile";
        detectedMakefileButton->setVisible( true );
    }
    else if ( QFile::exists( detectedMakefile + "/MAKEFILE" ) )
    {
        detectedMakefile += "/MAKEFILE";
        detectedMakefileButton->setVisible( true );
    }
    else if ( QFile::exists( detectedMakefile + "/makefile" ) )
    {
        detectedMakefile += "/makefile";
        detectedMakefileButton->setVisible( true );
    }

    selectMakefileWidget->setVisible( true );
    detectedMakefileButton->setToolTip( tr( "detected Makefile: " ) + detectedMakefile );
}

/**
 * @brief Slot function that handles the "Browse" button click event for selecting a makefile. It openes a file dialog
 * that allows the user to choose a makefile. If a makefile is selected the function calls the openMakefile method with
 * the selected makefile as argument to open it.
 */
void
InstrumentationTab::onBrowseMakefileButtonClicked()
{
    QFileDialog dialog( this, tr( "Open File" ), detectedMakefile );
    if ( dialog.exec() )
    {
        QString makefile = dialog.selectedFiles().at( 0 );
        openMakefile( makefile );
    }
}

/**
 * @brief Slot function that handles the "Open detected Makefile" button click event. It calls the openMakefile method
 * with the detected makefile as argument to open it.
 */
void
InstrumentationTab::onDetectedMakefileButtonClicked()
{
    openMakefile( detectedMakefile );
}

/**
 * @brief Opens a makefile for editing. If the file cannot be opened, it shows a warning message.
 * @param filename QString containing the path to the makefile to be opened.
 */
void
InstrumentationTab::openMakefile( QString filename )
{
    QFile file( filename );
    if ( !file.open( QIODevice::ReadOnly | QFile::Text ) )
    {
        QMessageBox::warning( this, tr( "Warning" ), tr( "Cannot open the file: " ) + file.errorString() );
        return;
    }
    setWindowTitle( filename );
    QTextStream in( &file );
    QString     text = in.readAll();
    makefileEdit->setText( text );
    file.close();

    mainWidget->setVisible( false );
    makefileWidget->setVisible( true );
    openedFile = filename;
}

/**
 * @brief Checks whether a file is executable by using the `file` command and checking the output. If `file` command is not
 * available it checks for existence and executabliity of the file using shell commands.
 * @param file The path to the file to check for executability.
 * @param saveSettings Whether to save the output to the command history.
 * @return A boolean indicating whether the file is executable or not.
 */
bool
InstrumentationTab::isExecutable( QString file, bool saveSettings )
{
    int     returncode;
    QString filePath = QString::fromStdString( console->execCommand( "which file", returncode, false ) ).remove( REGULAR_EXPRESSION( "\\n" ) );
    if ( returncode == 0 )
    {
        QString output = QString::fromStdString( console->execCommand( filePath.toStdString() + " " + file.toStdString(), returncode, saveSettings, false, saveSettings ) );
        if ( output.contains( tr( "executable" ) ) )
        {
            return true;
        }
        else
        {
            return false;
        }
    }
    else
    {
        console->execCommand( "[ -f " + file.toStdString() + " ] && [ -x " + file.toStdString() + " ]", returncode, false );
        if ( returncode == 0 )
        {
            return true;
        }
        else
        {
            return false;
        }
    }
}

/**
 * @brief Displayes a file dialog to select an executable file. If the selected file is not executable, an error message is
 * displayed and the further instrumentation steps are hidden. If the selected file is already instrumented, the option to
 * use the former instrumentation is enabled, otherwise it is disabled.
 */
void
InstrumentationTab::onSelectExecutableButtonClicked()
{
    QString     formerExecutableName = executableName;
    QFileDialog dialog( this, tr( "Open File" ), QString::fromStdString( getenv( "HOME" ) ) );
    if ( dialog.exec() )
    {
        executableName = dialog.selectedFiles().at( 0 );
        if ( ( executableName != formerExecutableName || formerExecutableName == "" ) && executableName != "" )
        {
            fileLabel->setText( tr( "Selected file: " ) + executableName );
            fileLabel->setVisible( true );
            measurementWindow->settings.setValue( "measurement/executableName", executableName );

            // delete further saved settings
            measurementWindow->settings.setValue( "measurement/formerInstrumentation", "" );
            measurementWindow->settings.setValue( "measurement/adjustMakefile", "" );
            measurementWindow->settings.setValue( "measurement/buildCmd", "" );
            measurementWindow->settings.setValue( "measurement/buildSuccess", "" );
            measurementWindow->settings.setValue( "measurement/executableDir", "" );
            measurementWindow->settings.setValue( "measurement/selectedRun", "" );
            measurementWindow->settings.setValue( "measurement/numProcs", "" );
            measurementWindow->settings.setValue( "measurement/numThreads", "" );
            measurementWindow->settings.setValue( "measurement/filterFile", "" );
            measurementWindow->settings.setValue( "measurement/specifyFilter", "" );
            measurementWindow->settings.setValue( "measurement/createFilter", "" );
            measurementWindow->settings.setValue( "measurement/filterProfile", "" );
            measurementWindow->settings.setValue( "measurement/generateFilter", "" );
            measurementWindow->settings.setValue( "measurement/bufferPercent", "" );
            measurementWindow->settings.setValue( "measurement/timePerVisit", "" );
            measurementWindow->settings.setValue( "measurement/type", "" );
            measurementWindow->settings.setValue( "measurement/runSuccess", "" );
            measurementWindow->settings.setValue( "measurement/submittedJob", "" );
            measurementWindow->settings.setValue( "measurement/jobid", "" );
            measurementWindow->settings.setValue( "measurement/prepareJob", "" );

            // check if selected file is executable
            if ( !isExecutable( executableName ) )
            {
                selectExecutableErrorMessage->setVisible( true );
                selectExecutableErrorMessage->setText( executableName + tr( " is not executable" ) );
                fileLabel->setVisible( false );
                selectInstrumentationWidget->setVisible( false );
            }
            else
            {
                selectExecutableErrorMessage->setVisible( false );
                if ( isInstrumented( executableName ) )
                {
                    formerInstrumentationButton->setEnabled( true );
                    selectInstrumentationLabel->setText( tr( "Your application is already instrumented. Please select if you want to continue with existing instrumentation or if you want to prepare a new instrumentation." ) );
                    selectInstrumentationLabel->setVisible( true );
                }
                else
                {
                    selectInstrumentationLabel->setVisible( false );
                    formerInstrumentationButton->setEnabled( false );
                }
                QStringList dirNames = executableName.split( "/" );
                QString     name     = dirNames.at( dirNames.length() - 1 );
                executableDir = "";
                for ( int i = 0; i < dirNames.length() - 1; i++ )
                {
                    if ( dirNames.at( i ) != "" )
                    {
                        executableDir += "/" + dirNames.at( i );
                    }
                }

                measurementWindow->settings.setValue( "measurement/executableDir", executableDir );
                int     returncode;
                QString makePath = QString::fromStdString( console->execCommand( "which make", returncode, false ) ).remove( REGULAR_EXPRESSION( "\\n" ) );
                buildCommandEdit->setText( "cd " + executableDir + " && " + makePath + " -B " + name );
                measurementWindow->settings.setValue( "measurement/buildCmd", buildCommandEdit->text() );
                fileLabel->setVisible( true );
                selectInstrumentationWidget->setVisible( true );
                selectExecutableButton->setText( tr( "Browse another executable file" ) );
            }
        }
        if ( formerExecutableName != executableName )
        {
            // hide further steps and uncheck selected options
            uncheck( formerInstrumentationButton );
            uncheck( newInstrumentationButton );
            uncheck( useCMake );
            uncheck( useMakefile );
            makefileWidget->setVisible( false );
            buildWidget->setVisible( false );
            selectBuildWidget->setVisible( false );
            selectMakefileWidget->setVisible( false );
            analysisWidget->setVisible( false );
            measurementWindow->tabWidget->setTabEnabled( 1, false );
        }
    }
}

/**
 * @brief Handles the event when the "Build" button is clicked. Executes the build command edited by the user in the corresponding
 * text field, and checks whether the executable file has been instrumented successfully. It shows according help messages.
 */
void
InstrumentationTab::onBuildButtonClicked()
{
    int returncode;

    console->execCommand( buildCommandEdit->text().toStdString(), returncode );
    if ( returncode == 0 && isInstrumented( executableName ) )
    {
        analysisLabel->setVisible( true );
        analysisLabel->setText( tr( "<font color=\"green\">Your application has been rebuilt. The instrumentation was successful.</font>" ) );
        analysisButton->setVisible( true );
        analysisWidget->setVisible( true );
        measurementWindow->settings.setValue( "measurement/buildCmd", buildCommandEdit->text() );
        measurementWindow->settings.setValue( "measurement/buildSuccess", "True" );
    }
    else if ( returncode == 0 && !isInstrumented( executableName ) )
    {
        analysisLabel->setVisible( true );
        analysisLabel->setText( tr( "<font color=\"orange\">Your application may has been rebuilt but the instrumentation was not successful. Please check if you applied the required changes to your makefile.</font>" ) );
        analysisButton->setVisible( false );
        analysisWidget->setVisible( true );
    }
    else
    {
        analysisLabel->setVisible( true );
        analysisButton->setVisible( false );
        analysisLabel->setText( tr( "<font color=\"red\">Your application could not be built</font>" ) );
        analysisWidget->setVisible( true );
    }
}

/**
 * @brief Switches displayed tab from the current tab to the execution tab.
 * @param loadSettings Indicates if the function is called by the loadSettings function to restore a former measurement.
 */
void
InstrumentationTab::onAnalysisButtonClicked( bool loadSettings )
{
    // don't clean up measurement screen if user return to instrumentataion step but applies no changes
    if ( !measurementWindow->tabWidget->isTabEnabled( 1 ) && !loadSettings )
    {
        measurementWindow->executionTab->onTakeMeasurementButtonClicked();
        measurementWindow->executionTab->mainWidget->setVisible( true );
        measurementWindow->executionTab->editFilterWidget->setVisible( false );
        measurementWindow->executionTab->editJobScriptWidget->setVisible( false );
        measurementWindow->executionTab->inspectFilterWidget->setVisible( false );
    }

    // enable execution tab
    measurementWindow->tabWidget->setTabEnabled( 1, true );

    // change current tab into execution tab
    measurementWindow->tabWidget->setCurrentIndex( 1 );

    // refresh available runs
    if ( measurementWindow->executionTab->profileExists() )
    {
        measurementWindow->executionTab->finetunedRunButton->setEnabled( true );
        measurementWindow->executionTab->finetunedRunButton->setToolTip( tr( "finetuned run requires filter file" ) );
        measurementWindow->executionTab->addProfileButtons();
    }
    else
    {
        measurementWindow->executionTab->finetunedRunButton->setEnabled( false );
        measurementWindow->executionTab->finetunedRunButton->setToolTip( tr( "Available when profile exists. Please start with initial run." ) );
    }
}

/**
 * @brief Changes layout. Shows the build widget and hides the analysis widget.
 */
void
InstrumentationTab::selectedNewInstrumentation()
{
    selectBuildWidget->setVisible( true );
    analysisWidget->setVisible( false );
    useFormerInstrumentation = false;
    measurementWindow->settings.setValue( "measurement/formerInstrumentation", "False" );
}

/**
 * @brief Changes layout. Shows the option to continue with the analysis step.
 */
void
InstrumentationTab::selectedFormerInstrumentation()
{
    analysisButton->setVisible( true );
    analysisLabel->setVisible( false );
    analysisWidget->setVisible( true );
    selectBuildWidget->setVisible( false );
    makefileWidget->setVisible( false );
    buildWidget->setVisible( false );
    selectBuildWidget->setVisible( false );
    selectMakefileWidget->setVisible( false );
    useFormerInstrumentation = true;
    uncheck( useCMake );
    uncheck( useMakefile );
    measurementWindow->settings.setValue( "measurement/formerInstrumentation", "True" );
}

/**
 * @brief Checks if the given executable file is instrumented ny parsing the output of the `nm` command.
 * @param path The path to the executable file to check.
 * @return True, if the file is instrumented with Score-P, false otherwise.
 */
bool
InstrumentationTab::isInstrumented( QString path )
{
    int    returncode;
    string s = console->execCommand( "nm " + path.toStdString(), returncode, false );
    if ( returncode != 0 )
    {
        return false;
    }
    QString output = QString::fromStdString( s );
    if ( output.contains( "SCOREP" ) || output.contains( "scorep" ) )
    {
        return true;
    }
    return false;
}

/**
 * @brief Sets the layout.
 */
ExecutionTab::ExecutionTab( ContextFreeServices* service, Console* console, QWidget* parent ) : MeasurementTab( service, console, parent )
{
    unsetConfigVars();

    layout     = new QVBoxLayout;
    mainLayout = new QVBoxLayout;

    QVBoxLayout* informationLayout = new QVBoxLayout;

    QLabel* informationLabel = new QLabel( tr( "<b>This wizard will help you to do the measurement of your program</b><br>"
                                               "You can select one of the following run types<br>"
                                               "\u27a4 initial run is the first run that should be chosen (creates profile) <br>"
                                               "\u27a4 finetuned run is available when there already exists a profile (filter file is required) <br>"
                                               "\u27a4 detailed run is available when a run with filter exists (supports tracing) <br>"
                                               "\u27a4 custom run allowes to configure the environment options manually (recommended for expierienced user)" ) );

    informationLayout->addWidget( informationLabel );
    informationLayout->addSpacing( 20 );
    informationWidget = new QWidget;
    informationWidget->setLayout( informationLayout );
    mainLayout->addWidget( informationWidget );
    mainLayout->addStretch( 0 );

    // select run, number of processes, number of threads layout
    selectRunWidget = new QWidget;
    QGroupBox*   radioBox        = new QGroupBox;
    QGridLayout* radioBoxLayout  = new QGridLayout;
    QVBoxLayout* selectRunLayout = new QVBoxLayout;

    initialRunButton   = new QRadioButton( tr( "initial run" ) );
    detailedRunButton  = new QRadioButton( tr( "detailed run" ) );
    finetunedRunButton = new QRadioButton( tr( "finetuned run" ) );
    customRunButton    = new QRadioButton( tr( "custom run" ) );
    customRunButton->setEnabled( false );
    customRunButton->setToolTip( tr( "Available in a future release" ) );

    initialRunButton->setToolTip( tr( "creates profile" ) );

    // first run: profiling -> create filter -> finetuned run -> detailed run
    if ( true ) //placeholder for counter
    {
        detailedRunButton->setEnabled( false );
        detailedRunButton->setToolTip( tr( "Available when finetuned run was executed. Supports tracing." ) );
    }

    radioBoxLayout->addWidget( initialRunButton, 0, 0 );
    radioBoxLayout->addWidget( detailedRunButton, 0, 1 );
    radioBoxLayout->addWidget( finetunedRunButton, 1, 0 );
    radioBoxLayout->addWidget( customRunButton, 1, 1 );

    radioBox->setLayout( radioBoxLayout );

    connect( initialRunButton, SIGNAL( clicked() ), this, SLOT( selectedInitialRun() ) );
    connect( detailedRunButton, SIGNAL( clicked() ), this, SLOT( selectedDetailedRun() ) );
    connect( finetunedRunButton, SIGNAL( clicked() ), this, SLOT( selectedFinetunedRun() ) );
    connect( customRunButton, SIGNAL( clicked() ), this, SLOT( selectedCustomRun() ) );

    runInfoLabel = new QLabel();
    runInfoLabel->setWordWrap( true );
    runInfoLabel->setVisible( false );

    QHBoxLayout* numLayout   = new QHBoxLayout;
    QLabel*      procsText   = new QLabel( tr( "number of processes:" ) );
    QLabel*      threadsText = new QLabel( tr( "number of threads:" ) );

    numProcs = new QSpinBox;
    numProcs->setRange( 1, 1000 );
    numThreads = new QSpinBox;
    numThreads->setRange( 1, 1000 );

    numLayout->addWidget( procsText );
    numLayout->addWidget( numProcs );
    numLayout->addSpacing( 20 );
    numLayout->addWidget( threadsText );
    numLayout->addWidget( numThreads );

    connect( numProcs, SIGNAL( valueChanged( int ) ), this, SLOT( selectedNumberOfProcesses( int ) ) );
    connect( numThreads, SIGNAL( valueChanged( int ) ), this, SLOT( selectedNumberOfThreads( int ) ) );

    QVBoxLayout* dirLayout = new QVBoxLayout;
    QLabel*      dirText   = new QLabel( tr( "experiment directory name:" ) );
    dirNameEdit         = new QLineEdit;
    dirNameErrorMessage = new QLabel();
    dirNameErrorMessage->setVisible( false );
    dirLayout->addWidget( dirText );
    dirLayout->addWidget( dirNameEdit );
    dirLayout->addWidget( dirNameErrorMessage );

    dirNameEdit->setText( "scorep-" + convertDate() + "_np" + QString::number( numProcs->value() ) );

    selectRunLayout->addWidget( radioBox );
    selectRunLayout->addSpacing( 20 );
    selectRunLayout->addWidget( runInfoLabel );
    selectRunLayout->addSpacing( 20 );
    selectRunLayout->addLayout( numLayout );
    selectRunLayout->addSpacing( 20 );
    selectRunLayout->addLayout( dirLayout );
    selectRunLayout->addStretch( 0 );

    selectRunWidget->setLayout( selectRunLayout );
    mainLayout->addWidget( selectRunWidget );

    // custom run layout
    customRunWidget = new QWidget;
    QVBoxLayout* customRunLayout              = new QVBoxLayout;
    QScrollArea* scrollEnvironmentOptionsArea = new QScrollArea;

    enableProfiling = new QCheckBox( tr( "profiling enabled" ) );
    enableTracing   = new QCheckBox( tr( "tracing disabled" ) );
    enableUnwinding = new QCheckBox( tr( "unwinding disabled" ) );
    beVerbose       = new QCheckBox( tr( "verbose disabled" ) );

    int     returncode;
    QString availableOptions = QString::fromStdString( console->execCommand( "scorep-info config-vars", returncode, false, false, false ) );
    if ( !availableOptions.contains( "SCOREP_ENABLE_TRACING" ) )
    {
        enableTracing->setEnabled( false );
    }
    if ( !availableOptions.contains( "SCOREP_ENABLE_PROFILING" ) )
    {
        enableProfiling->setEnabled( false );
    }
    if ( !availableOptions.contains( "SCOREP_ENABLE_UNWINDING" ) )
    {
        enableUnwinding->setEnabled( false );
    }
    if ( !availableOptions.contains( "SCOREP_VERBOSE" ) )
    {
        beVerbose->setEnabled( false );
    }

    QRadioButton* memoryAccessButton   = new QRadioButton( tr( "efficiency of memory access" ) );
    QRadioButton* dataProcessingButton = new QRadioButton( tr( "efficiency of data processing" ) );
    QVBoxLayout*  efficiencyLayout     = new QVBoxLayout;
    QGroupBox*    efficiencyGroupbox   = new QGroupBox;

    QWidget*     checkboxes             = new QWidget( scrollEnvironmentOptionsArea );
    QGridLayout* checkBoxesWidgetLayout = new QGridLayout( checkboxes );

    QPixmap infoPixmap( style()->standardIcon( QStyle::SP_MessageBoxInformation ).pixmap( 20, 20 ) );
    QIcon   infoIcon = QIcon( infoPixmap );

    QPushButton* unwindingInfoButton = new QPushButton( infoIcon, "" );
    QPushButton* profilingInfoButton = new QPushButton( infoIcon, "" );
    QPushButton* tracingInfoButton   = new QPushButton( infoIcon, "" );
    QPushButton* verboseInfoButton   = new QPushButton( infoIcon, "" );

    checkBoxesWidgetLayout->addWidget( enableProfiling, 0, 0 );
    checkBoxesWidgetLayout->addWidget( profilingInfoButton, 0, 1 );
    checkBoxesWidgetLayout->addWidget( enableTracing, 1, 0 );
    checkBoxesWidgetLayout->addWidget( tracingInfoButton, 1, 1 );
    checkBoxesWidgetLayout->addWidget( enableUnwinding, 2, 0 );
    checkBoxesWidgetLayout->addWidget( unwindingInfoButton, 2, 1 );
    checkBoxesWidgetLayout->addWidget( beVerbose, 3, 0 );
    checkBoxesWidgetLayout->addWidget( verboseInfoButton, 3, 1 );
    enableProfiling->setCheckState( Qt::Checked );

    connect( enableProfiling, SIGNAL( stateChanged( int ) ), this, SLOT( enableProfilingClicked( int ) ) );
    connect( enableTracing, SIGNAL( stateChanged( int ) ), this, SLOT( enableTracingClicked( int ) ) );
    connect( enableUnwinding, SIGNAL( stateChanged( int ) ), this, SLOT( enableUnwindingClicked( int ) ) );
    connect( beVerbose, SIGNAL( stateChanged( int ) ), this, SLOT( beVerboseClicked( int ) ) );

    connect( unwindingInfoButton, SIGNAL( clicked() ), this, SLOT( onUnwindingInfoButtonClicked() ) );
    connect( profilingInfoButton, SIGNAL( clicked() ), this, SLOT( onProfilingInfoButtonClicked() ) );
    connect( tracingInfoButton, SIGNAL( clicked() ), this, SLOT( onTracingInfoButtonClicked() ) );
    connect( verboseInfoButton, SIGNAL( clicked() ), this, SLOT( onVerboseInfoButtonClicked() ) );

    scrollEnvironmentOptionsArea->setWidget( checkboxes );
    efficiencyLayout->addWidget( memoryAccessButton );
    efficiencyLayout->addWidget( dataProcessingButton );
    efficiencyGroupbox->setLayout( efficiencyLayout );
    customRunLayout->addWidget( scrollEnvironmentOptionsArea );
    customRunLayout->addWidget( efficiencyGroupbox );

    customRunWidget->setLayout( customRunLayout );
    customRunWidget->setVisible( false );
    mainLayout->addWidget( customRunWidget );

    // finetuned run layout
    finetunedRunWidget = new QWidget;
    QVBoxLayout* finetunedRunLayout = new QVBoxLayout;
    QLabel*      filterFileLabel    = new QLabel( tr( "Please specify your filter file or create a filter file" ) );
    QVBoxLayout* selectFilterLayout = new QVBoxLayout;
    browseFilterButton = new QRadioButton( tr( "Select filter file" ) );
    createFilterButton = new QRadioButton( tr( "Create filter file" ) );

    selectFilterLayout->addWidget( browseFilterButton );
    selectFilterLayout->addWidget( createFilterButton );
    QGroupBox* selectFilterBox = new QGroupBox;
    selectFilterBox->setLayout( selectFilterLayout );
    finetunedRunLayout->addWidget( filterFileLabel );
    finetunedRunLayout->addWidget( selectFilterBox );
    filterPathLabel = new QLabel;
    filterPathLabel->setVisible( false );
    finetunedRunLayout->addWidget( filterPathLabel );

    finetunedRunWidget->setLayout( finetunedRunLayout );
    mainLayout->addWidget( finetunedRunWidget );
    finetunedRunWidget->setVisible( false );
    mainLayout->addStretch( 0 );

    connect( browseFilterButton, SIGNAL( clicked() ), this, SLOT( onBrowseFilterButtonClicked() ) );
    connect( createFilterButton, SIGNAL( clicked() ), this, SLOT( onCreateFilterButtonClicked() ) );

    QPushButton* createManuallyButton = new QPushButton( tr( "Create filter file manually" ) );
    QPushButton* generateFilterButton = new QPushButton( tr( "Generate initial filter file" ) );

    connect( createManuallyButton, SIGNAL( clicked() ), this, SLOT( onCreateManuallyButtonClicked() ) );
    connect( generateFilterButton, SIGNAL( clicked() ), this, SLOT( onGenerateFilterButtonClicked() ) );

    // create manually or generate
    filterOptionsWidget = new QWidget;
    QHBoxLayout* filterOptionsLayout = new QHBoxLayout;
    filterOptionsLayout->addWidget( createManuallyButton );
    filterOptionsLayout->addWidget( generateFilterButton );
    filterOptionsWidget->setLayout( filterOptionsLayout );
    mainLayout->addWidget( filterOptionsWidget );
    filterOptionsWidget->setVisible( false );
    mainLayout->addStretch( 0 );

    // create manually layout
    QVBoxLayout* editFilterLayout = new QVBoxLayout;
    QLabel*      editFilterLabel  = new QLabel( tr( "<font><b>Please create your filter file</b></font><br>"
                                                    "The filter block for the region names, must be enclosed by SCOREP_REGION_NAMES_BEGIN "
                                                    "and SCOREP_REGION_NAMES_END. In between you can specifiy an arbitrary number of include "
                                                    "and exclude rules which are evaluated in sequential order. At the beginning, all regions "
                                                    "are included. Regions that are excluded after all rules are evaluated, are filtered. "
                                                    "For more information please see <a href=\"https://perftools.pages.jsc.fz-juelich.de/cicd/scorep/tags/scorep-7.1/html/measurement.html#filtering\">Score-P user guide</a>"
                                                    ) );
    editFilterLabel->setWordWrap( true );
    editFilterLabel->setOpenExternalLinks( true );
    editFilterLayout->addWidget( editFilterLabel );
    editFilterLayout->addSpacing( 5 );
    filterFileEdit = new QTextEdit;
    QPushButton* discardChangesButton   = new QPushButton( tr( "Discard changes" ) );
    QPushButton* saveChangesButton      = new QPushButton( tr( "Save changes and apply filter" ) );
    QHBoxLayout* editFilterButtonLayout = new QHBoxLayout;
    editFilterButtonLayout->addWidget( discardChangesButton );
    editFilterButtonLayout->addWidget( saveChangesButton );
    editFilterLayout->addWidget( filterFileEdit );
    filterFileEdit->setText( tr( "# initial filter file, starting point for further adaptation. \n"
                                 "#\n"
                                 "# Considerations:\n"
                                 "#  - check possible wildcard options\n"
                                 "#  - check selected functions for relevancy\n"
                                 "#    (w.r.t. your knowledge of the application)\n"
                                 "#\n"
                                 "# The file contains comments for each region providing additional information\n"
                                 "# regarding the respective region.\n"
                                 "#\n"
                                 "# Please refer to the Score-P user guide for more options on filtering.\n" ) +
                             "SCOREP_REGION_NAMES_BEGIN\n"
                             "   EXCLUDE\n"
                             "SCOREP_REGION_NAMES_END" );
    editFilterLayout->addLayout( editFilterButtonLayout );

    connect( saveChangesButton, SIGNAL( clicked() ), this, SLOT( onSaveChangesButtonClicked() ) );
    connect( discardChangesButton, SIGNAL( clicked() ), this, SLOT( onDiscardChangesButtonClicked() ) );

    editFilterWidget = new QWidget;
    editFilterWidget->setLayout( editFilterLayout );
    editFilterWidget->setVisible( false );

    // create filter layout
    generateFilterWidget = new QWidget;
    QVBoxLayout* generateFilterLayout = new QVBoxLayout;
    QLabel*      generateFilterLabel  = new QLabel( tr( "Please select which measurement should serve as basis for creating the filter" ) );
    generateFilterLayout->addWidget( generateFilterLabel );


    profileBox       = new QGroupBox();
    profileGroup     = new QButtonGroup;
    profileBoxLayout = new QVBoxLayout;
    profileBox->setLayout( profileBoxLayout );


    QScrollArea* profileScrollArea = new QScrollArea;
    profileScrollArea->setWidgetResizable( true );
    profileScrollArea->setWidget( profileBox );
    generateFilterLayout->addWidget( profileScrollArea );

    generateFilterWidget->setLayout( generateFilterLayout );
    generateFilterWidget->setVisible( false );
    mainLayout->addWidget( generateFilterWidget );
    mainLayout->addStretch( 0 );

    // configure scorep-score options or use scorepion plugin
    scorepionWidget = new QWidget;
    QHBoxLayout* scorepionLayout    = new QHBoxLayout;
    QPushButton* useScorePionButton = new QPushButton( tr( "Generate filter with ScorePion plugin" ) );
    // Use of ScorePion plugin not implemented yet
    useScorePionButton->setEnabled( false );
    QPushButton* configureScoreOptionsButton = new QPushButton( tr( "Generate filter here" ) );

    // test if ScorePion is available

    connect( useScorePionButton, SIGNAL( clicked() ), this, SLOT( onUseScorepionButtonClicked() ) );
    connect( configureScoreOptionsButton, SIGNAL( clicked() ), this, SLOT( onConfigureScoreOptionsButtonClicked() ) );

    scorepionLayout->addWidget( useScorePionButton );
    scorepionLayout->addWidget( configureScoreOptionsButton );

    scorepionWidget->setLayout( scorepionLayout );
    mainLayout->addWidget( scorepionWidget );
    scorepionWidget->setVisible( false );
    mainLayout->addStretch( 0 );


    // generate filter layout
    configureFilterWidget = new QWidget;
    QFormLayout* optionsFormLayout = new QFormLayout;
    QVBoxLayout* optionsLayout     = new QVBoxLayout;
    bufferPercent = new QSpinBox;
    bufferPercent->setRange( 0, 100 );
    timePerVisit = new QSpinBox;
    type         = new QComboBox;
    type->addItems( { "usr", "com", "both" } );
    setDefaultFilterValues();

    QLabel* configureInfoLabel = new QLabel( tr( "You can customize the configurations for filtering the regions or use the default values. "
                                                 "A region is included in the filter (i.e., excluded from measurement) if it matches all of the given conditions, with the following keys:" ) );
    configureInfoLabel->setWordWrap( true );
    QLabel* bufferPercentLabel = new QLabel( tr( "bufferpercent (%) " ) );
    bufferPercentLabel->setToolTip( tr( "region is excluded if estimated memory requirements exceed the given threshold in percent of the total estimated trace buffer requirements" ) );
    QLabel* timePerVisitLabel = new QLabel( tr( "timepervisit (ms) " ) );
    timePerVisitLabel->setToolTip( tr( "region is excluded if time per visit value is below the given threshold in microseconds" ) );
    QLabel* typeLabel = new QLabel( tr( "type " ) );
    typeLabel->setToolTip( tr( "region is excluded if region type matches the given value ('usr', 'com', 'both')" ) );

    optionsLayout->addWidget( configureInfoLabel );
    optionsFormLayout->addRow( bufferPercentLabel, bufferPercent );
    optionsFormLayout->addRow( timePerVisitLabel, timePerVisit );
    optionsFormLayout->addRow( typeLabel, type );
    optionsLayout->addLayout( optionsFormLayout );

    QHBoxLayout* configureButtonLayout = new QHBoxLayout;
    QPushButton* setDefaultButton      = new QPushButton( tr( "Set default values" ) );
    QPushButton* generateButton        = new QPushButton( tr( "Generate filter" ) );
    connect( generateButton, SIGNAL( clicked() ), this, SLOT( onGenerateButtonClicked() ) );
    connect( setDefaultButton, SIGNAL( clicked() ), this, SLOT( setDefaultFilterValues() ) );
    configureButtonLayout->addWidget( setDefaultButton );
    configureButtonLayout->addWidget( generateButton );
    optionsLayout->addLayout( configureButtonLayout );

    optionsLayout->addSpacing( 20 );
    configureFilterWidget->setLayout( optionsLayout );
    mainLayout->addWidget( configureFilterWidget );
    configureFilterWidget->setVisible( false );

    mainLayout->addStretch( 0 );

    // inspect filter layout
    inspectFilterWidget = new QWidget;
    QVBoxLayout* inspectFilterLayout       = new QVBoxLayout;
    QHBoxLayout* inspectFilterButtonLayout = new QHBoxLayout;
    QLabel*      inspectFilterLabel        = new QLabel( tr( "<font><b>Please inspect your filter file</b></font><br>"
                                                             "The filter block for the region names, must be enclosed by SCOREP_REGION_NAMES_BEGIN "
                                                             "and SCOREP_REGION_NAMES_END. In between you can specifiy an arbitrary number of include "
                                                             "and exclude rules which are evaluated in sequential order. At the beginning, all regions "
                                                             "are included. Regions that are excluded after all rules are evaluated, are filtered. "
                                                             "For more information please see <a href=\"https://perftools.pages.jsc.fz-juelich.de/cicd/scorep/tags/scorep-7.1/html/measurement.html#filtering\">Score-P user guide</a>"
                                                             ) );
    inspectFilterLabel->setOpenExternalLinks( true );
    inspectFilterLabel->setWordWrap( true );
    inspectFilterLayout->addWidget( inspectFilterLabel );
    inspectFilterEdit = new QTextEdit;
    QPushButton* discardFilterChangesButton = new QPushButton( tr( "Discard changes" ) );
    QPushButton* saveFilterChangesButton    = new QPushButton( tr( "Save changes" ) );

    connect( discardFilterChangesButton, SIGNAL( clicked() ), this, SLOT( onDiscardFilterChangesButtonClicked() ) );
    connect( saveFilterChangesButton, SIGNAL( clicked() ), this, SLOT( onSaveFilterChangesButtonClicked() ) );

    inspectFilterButtonLayout->addWidget( discardFilterChangesButton );
    inspectFilterButtonLayout->addWidget( saveFilterChangesButton );
    inspectFilterLayout->addWidget( inspectFilterEdit );
    inspectFilterLayout->addLayout( inspectFilterButtonLayout );

    inspectFilterWidget->setLayout( inspectFilterLayout );
    inspectFilterWidget->setVisible( false );

    // execution layout
    QVBoxLayout* executeLayout = new QVBoxLayout;

    filterInfoLabel = new QLabel;
    filterInfoLabel->setWordWrap( true );
    executeLayout->addWidget( filterInfoLabel );
    filterInfoLabel->setVisible( false );

    inspectFilterFileButton = new QPushButton( tr( "Inspect generated filter" ) );
    connect( inspectFilterFileButton, SIGNAL( clicked() ), this, SLOT( onInspectFilterButtonClicked() ) );
    executeLayout->addWidget( inspectFilterFileButton );
    inspectFilterFileButton->setVisible( false );

    executeWidget          = new QWidget;
    runApplicationButton   = new QPushButton( tr( "Run local" ) );
    prepareJobScriptButton = new QPushButton( tr( "Prepare job script" ) );
    QHBoxLayout* submitButtonLayout = new QHBoxLayout;
    execCommandEdit = new QLineEdit;
    submitButtonLayout->addWidget( prepareJobScriptButton );
    submitButtonLayout->addWidget( runApplicationButton );
    QLabel* executeLabel = new QLabel( tr( "Please enter your run command" ) );
    runLabel = new QLabel();
    runLabel->setVisible( false );
    executeLayout->addWidget( executeLabel );
    executeLayout->addWidget( execCommandEdit );
    executeLayout->addLayout( submitButtonLayout );
    executeLayout->addWidget( runLabel );
    executeWidget->setLayout( executeLayout );
    executeWidget->setVisible( false );
    mainLayout->addWidget( executeWidget );
    mainLayout->addStretch( 0 );

    connect( runApplicationButton, SIGNAL( clicked() ), this, SLOT( onRunApplicationButtonClicked() ) );
    connect( prepareJobScriptButton, SIGNAL( clicked() ), this, SLOT( onPrepareJobScriptButtonClicked() ) );

    // prepare job layout
    QVBoxLayout* prepareJobLayout             = new QVBoxLayout;
    QHBoxLayout* prepareJobButtonLayout       = new QHBoxLayout;
    QPushButton* openOwnJobScriptButton       = new QPushButton( tr( "Open own job script" ) );
    QPushButton* openGeneratedJobScriptButton = new QPushButton( tr( "Open generated job script" ) );
    QLabel*      jobScriptLabel               = new QLabel( tr( "Please select whether you want to open an existing job script and adapt it, or if you want this plugin to generate one." ) );
    jobScriptLabel->setWordWrap( true );
    prepareJobWidget = new QWidget;

    prepareJobButtonLayout->addWidget( openOwnJobScriptButton );
    prepareJobButtonLayout->addWidget( openGeneratedJobScriptButton );
    prepareJobLayout->addWidget( jobScriptLabel );
    prepareJobLayout->addLayout( prepareJobButtonLayout );

    connect( openOwnJobScriptButton, SIGNAL( clicked() ), this, SLOT( onOpenOwnJobScriptButtonClicked() ) );
    connect( openGeneratedJobScriptButton, SIGNAL( clicked() ), this, SLOT( onOpenGeneratedJobScriptButtonClicked() ) );

    prepareJobWidget->setLayout( prepareJobLayout );
    prepareJobWidget->setVisible( false );
    mainLayout->addWidget( prepareJobWidget );
    mainLayout->addStretch( 0 );

    // show profile layout
    showProfileWidget = new QWidget;
    QVBoxLayout* showProfileLayout = new QVBoxLayout;

    QPushButton* createShellScriptButton = new QPushButton( tr( "Save shell script to reconstruct the measurement" ) );
    QLabel*      shellScriptInfoLabel    = new QLabel( tr( "Do you want to create a shell script from the selected configurations for the performed measurement to be able to reconstruct the measurement by executing this script?" ) );
    shellScriptInfoLabel->setWordWrap( true );
    shellScriptErrorLabel = new QLabel;
    shellScriptErrorLabel->setVisible( false );

    showProfileLayout->addWidget( shellScriptInfoLabel );
    showProfileLayout->addWidget( createShellScriptButton );
    showProfileLayout->addWidget( shellScriptErrorLabel );
    showProfileLayout->addSpacing( 30 );

    QPushButton* showProfileButton     = new QPushButton( tr( "Show profile" ) );
    QPushButton* takeMeasurementButton = new QPushButton( tr( "Take another measurement" ) );
    QLabel*      showProfileLabel      = new QLabel( tr( "You can select if you want to see the created profile or if you want to take another measurement" ) );

    QHBoxLayout* buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget( takeMeasurementButton );
    buttonLayout->addWidget( showProfileButton );
    showProfileLayout->addWidget( showProfileLabel );
    showProfileLayout->addLayout( buttonLayout );
    showProfileWidget->setLayout( showProfileLayout );
    showProfileWidget->setVisible( false );
    mainLayout->addWidget( showProfileWidget );
    mainLayout->addStretch( 0 );

    connect( showProfileButton, SIGNAL( clicked() ), this, SLOT( onShowProfileButtonClicked() ) );
    connect( takeMeasurementButton, SIGNAL( clicked() ), this, SLOT( onTakeMeasurementButtonClicked() ) );
    connect( createShellScriptButton, SIGNAL( clicked() ), this, SLOT( onCreateShellScriptButtonClicked() ) );

    // edit jobscript layout
    editJobScriptWidget = new QWidget;
    QVBoxLayout* editJobScriptLayout = new QVBoxLayout;
    editJobScriptLabel = new QLabel( tr( "<font><b>You can edit the generated job script.</b></font><br>" ) );
    editJobScriptLabel->setTextInteractionFlags( Qt::TextSelectableByMouse );
    editJobScriptLabel->setWordWrap( true );
    editJobScriptLabel->setOpenExternalLinks( true );
    editJobScriptLayout->addWidget( editJobScriptLabel );
    editJobScriptLayout->addSpacing( 5 );

    jobScriptEdit = new QTextEdit;
    jobScriptEdit->setFont( { "Courier" } );
    QPushButton* saveJobScriptChangesButton = new QPushButton( tr( "Save changes and submit job script" ) );
    QPushButton* discardJobScriptButton     = new QPushButton( tr( "Discard changes" ) );
    QHBoxLayout* editJobScriptButtonLayout  = new QHBoxLayout;
    editJobScriptButtonLayout->addWidget( saveJobScriptChangesButton );
    editJobScriptButtonLayout->addWidget( discardJobScriptButton );
    editJobScriptLayout->addWidget( jobScriptEdit );
    editJobScriptLayout->addLayout( editJobScriptButtonLayout );

    connect( discardJobScriptButton, SIGNAL( clicked() ), this, SLOT( onDiscardJobScriptButtonClicked() ) );
    connect( saveJobScriptChangesButton, SIGNAL( clicked() ), this, SLOT( onSaveJobScriptChangesButtonClicked() ) );

    editJobScriptWidget->setLayout( editJobScriptLayout );
    editJobScriptWidget->setVisible( false );

    mainLayout->addStretch( 0 );
    mainWidget = new QWidget;
    mainWidget->setLayout( mainLayout );
    layout->addWidget( mainWidget );
    layout->addWidget( editFilterWidget );
    layout->addWidget( editJobScriptWidget );
    layout->addWidget( inspectFilterWidget );
    setLayout( layout );
};

/**
 * @brief Slot function that handles the "Discard Changes" button click event in the filter inspection step. Responsible for
 * hiding the filter inspection widget and showing the main widget.
 */
void
ExecutionTab::onDiscardFilterChangesButtonClicked()
{
    inspectFilterWidget->setVisible( false );
    mainWidget->setVisible( true );
}

/**
 *  @brief Saves the inspected filter.
 *  @details This slot function is responsible for saving the inspected filer to the opened file, hiding the filter inspection widget,
 *  and showing the main widget.
 *  @pre A filter must be opened.
 *  @post The inspected filter is saved to the opened file, the filter inspection widget is hidden, and the
 *  main widget is shown.
 */
void
ExecutionTab::onSaveFilterChangesButtonClicked()
{
    QFile file( filterFile );
    file.open( QIODevice::WriteOnly );
    QTextStream stream( &file );
    stream << inspectFilterEdit->toPlainText();

    inspectFilterWidget->setVisible( false );
    mainWidget->setVisible( true );
}

/**
 * @brief Opens the selected filter file for inspection. If the file cannot be opened, it shows a warning message.
 */
void
ExecutionTab::onInspectFilterButtonClicked()
{
    inspectFilterWidget->setVisible( true );
    mainWidget->setVisible( false );

    QFile file( filterFile );
    if ( !file.open( QIODevice::ReadOnly | QFile::Text ) )
    {
        QMessageBox::warning( this, tr( "Warning" ), tr( "Cannot open the file: " ) + file.errorString() );
        return;
    }
    setWindowTitle( filterFile );
    QTextStream in( &file );
    QString     text = in.readAll();
    inspectFilterEdit->setText( text );
}

QString
ExecutionTab::whitespaces( int n )
{
    QString ret;
    for ( int i = 0; i < n; i++ )
    {
        ret += " ";
    }
    return ret;
}

/**
 * @brief Slot function that handles the "Discard Changes" button click event in the job script editing step. Responsible for
 * hiding the job script editing widget and showing the main widget.
 */
void
ExecutionTab::onDiscardJobScriptButtonClicked()
{
    editJobScriptWidget->setVisible( false );
    mainWidget->setVisible( true );
}

/**
 *  @brief Saves the edited job script.
 *  @details This slot function is responsible for saving the edited job script, submitting the job, hiding the job script editing widget,
 *  and showing the main widget.
 *  @pre A filter must be opened.
 *  @post The inspected filter is saved to the opened file, the filter inspection widget is hidden, and the
 *  main widget is shown.
 */
void
ExecutionTab::onSaveJobScriptChangesButtonClicked()
{
    QString filter = "(*.slurm)";
    slurmScriptName = QFileDialog::getSaveFileName( this, "Save", measurementWindow->instrumentationTab->executableDir + "/script.slurm", filter, &filter );
    measurementWindow->settings.setValue( "measurement/slurmScriptName", slurmScriptName );
    QFile file( slurmScriptName );
    if ( !file.open( QFile::WriteOnly | QFile::Text ) )
    {
        QMessageBox::warning( this, "Warning", "Cannot save file: " + file.errorString() );
        return;
    }
    QTextStream out( &file );
    QString     text = jobScriptEdit->toPlainText();
    out << text;
    file.close();
    file.setPermissions( QFileDevice::ExeOwner | QFileDevice::ReadOwner | QFileDevice::WriteOwner );

    // check if dirname already exists
    int         returncode;
    QStringList dirNames = text.split( "export SCOREP_EXPERIMENT_DIRECTORY=" );

    QString profilePath = QString( dirNames.at( 1 ).split( REGULAR_EXPRESSION( "\\n" ) ).at( 0 ) ) + "/profile.cubex";
    console->execCommand( "[ -e " + profilePath.toStdString() + " ]", returncode, false );

    if ( returncode != 0 || !QFile::exists( profilePath ) )
    {
        dirNameErrorMessage->setVisible( false );
        // submit job
        QStringList jobidList = QString::fromStdString( console->execCommand( "sbatch " + slurmScriptName.toStdString(), returncode ) ).split( " " );
        jobid = "";
        if ( returncode == 0 )
        {
            for ( QString& tmp : jobidList )
            {
                for ( QChar c : tmp )
                {
                    if ( !c.isNumber() )
                    {
                        break;
                    }
                }
                jobid = tmp.remove( REGULAR_EXPRESSION( "\\n" ) );
            }
            if ( jobid.isEmpty() )
            {
                runLabel->setText( tr( "<font color=\"red\">Error while retrieving the job ID</font>" ) );
            }
            else
            {
                measurementWindow->settings.setValue( "measurement/submittedJob", "True" );
                measurementWindow->settings.setValue( "measurement/jobid", jobid );
                measurementWindow->settings.setValue( "measurement/profilePath", profilePath );
                runLabel->setText( tr( "<font color=\"green\">Your job has been submitted.</font>" ) );
                runLabel->setVisible( true );

                // resave all configurations measurement/var -> <jobid>/var
                QStringList keys = measurementWindow->settings.allKeys();
                for ( const QString& key : keys )
                {
                    if ( key.contains( "measurement/" ) )
                    {
                        QString newKey = key;
                        newKey.replace( "measurement", jobid );
                        measurementWindow->settings.setValue( newKey, measurementWindow->settings.value( key ) );
                    }
                }
            }
        }
        if ( returncode == 0 )
        {
            // presettings can not be changed after run
            selectRunWidget->setEnabled( false );
            executeWidget->setEnabled( false );
            prepareJobWidget->setEnabled( false );
            inspectFilterFileButton->setEnabled( false );
            finetunedRunWidget->setEnabled( false );
            generateFilterWidget->setEnabled( false );
            filterInfoLabel->setVisible( false );
            customRunWidget->setEnabled( false );
            configureFilterWidget->setEnabled( false );
            filterOptionsWidget->setEnabled( false );
            scorepionWidget->setEnabled( false );
            timer = new QTimer( this );
            timer->start( 1000 );
            connect( timer, SIGNAL( timeout() ), this, SLOT( updateStatus() ) );
        }
        else
        {
            runLabel->setText( tr( "<font color=\"red\">Your job could not be submitted</font>" ) );
            runLabel->setVisible( true );
        }
    }
    else
    {
        dirNameErrorMessage->setVisible( true );
        dirNameErrorMessage->setText( tr( "<font color=\"red\">There already exists a file with that name. Please choose another name.</font>" ) );
    }

    mainWidget->setVisible( true );
    editJobScriptWidget->setVisible( false );
}

/**
 * @brief Displays the current status of the submitted job.
 */
void
ExecutionTab::updateStatus()
{
    if ( measurementWindow->getStatus( jobid.toInt() ) == "COMPLETED" )
    {
        runLabel->setText( tr( "<font color=\"green\">Your job is finished" ) );
        showProfileWidget->setVisible( true );
        timer->stop();
    }
    else if ( measurementWindow->getStatus( jobid.toInt() ) == "FAILED" )
    {
        runLabel->setText( tr( "<font color=\"red\">Your job failed" ) );
        timer->stop();
    }
    else if ( measurementWindow->getStatus( jobid.toInt() ) == "RUNNING" )
    {
        runLabel->setText( tr( "<font color=\"green\">Your job is running" ) );
    }
}

/**
 * @brief Shows further options for creating a job script.
 */
void
ExecutionTab::onPrepareJobScriptButtonClicked()
{
    prepareJobWidget->setVisible( true );
    measurementWindow->settings.setValue( "measurement/prepareJob", "True" );
}

/**
 * @brief Opens a file dialog to select a job script file and updates the job script label and text edit accordingly. The file is opened for editing
 * and displayed on the job script editing widget. Lines to be added to the script are displayed to the user.
 */
void
ExecutionTab::onOpenOwnJobScriptButtonClicked()
{
    QFileDialog dialog( this, tr( "Select your job script" ), measurementWindow->instrumentationTab->executableDir, tr( "slurm job (*.slurm)" ) );
    if ( dialog.exec() )
    {
        QString jobFileName = dialog.selectedFiles().at( 0 );
        QFile   file( jobFileName );
        if ( !file.open( QIODevice::ReadOnly | QFile::Text ) )
        {
            QMessageBox::warning( this, tr( "Warning" ), tr( "Cannot open the file: " ) + file.errorString() );
            return;
        }
        editJobScriptLabel->setText( tr( "<font><b>Please add the following lines to your job script:</b></font><br><br>" ) +
                                     "#SBATCH --nodes=" + numProcs->text() + "<br>"
                                     "#SBATCH --ntasks=" + QString::number( numProcs->text().toInt() * numThreads->text().toInt() ) + "<br>"
                                     "#SBATCH --ntasks-per-node=" + numThreads->text() + "<br>" );
        if ( numThreads->text().toInt() > 1 )
        {
            editJobScriptLabel->setText( editJobScriptLabel->text() + "export OMP_NUM_THREADS=" + numThreads->text() + "<br>" );
        }
        if ( initialRunButton->isChecked() )
        {
            editJobScriptLabel->setText( editJobScriptLabel->text() + "export SCOREP_ENABLE_PROFILING=true<br>" );
        }
        else if ( finetunedRunButton->isChecked() )
        {
            // check if filtering file exists
            editJobScriptLabel->setText( editJobScriptLabel->text() + "export SCOREP_FILTERING_FILE=" + filterFile + "<br>" );
        }
        editJobScriptLabel->setText( editJobScriptLabel->text() + "export SCOREP_EXPERIMENT_DIRECTORY=" + measurementWindow->instrumentationTab->executableDir + "/" + dirNameEdit->text() + "<br>" );
        QTextStream in( &file );
        QString     text = in.readAll();
        jobScriptEdit->setText( text );
        editJobScriptWidget->setVisible( true );
        mainWidget->setVisible( false );
    }
}

/**
 * @brief Displays a job script generated by the plugin on the job script editing widget for further editing.
 */
void
ExecutionTab::onOpenGeneratedJobScriptButtonClicked()
{
    editJobScriptLabel->setText( tr( "<font><b>You can edit the generated job script.</b></font><br>" ) );
    // calculate number largest string
    QStringList dirNames   = measurementWindow->instrumentationTab->executableName.split( "/" );
    QString     name       = dirNames.at( dirNames.length() - 1 );
    int         max_length = qMax( QString( "#SBATCH --job-name=" + name.mid( 0, name.indexOf( "." ) ) + tr( "Measurement" ) ).length(), QString( "#SBATCH --ntasks-per-node=" + numThreads->text() ).length() );

    jobScriptEdit->setText( "#!/bin/bash \n"
                            + tr( "# initial job script, starting point for further adaptation." ) + "\n"
                            "\n"
                            "#SBATCH --job-name=" + name.mid( 0, name.indexOf( "." ) ) + tr( "Measurement" ) + whitespaces( max_length - QString( "#SBATCH --job-name=" + name.mid( 0, name.indexOf( "." ) ) + tr( "Measurement" ) ).length() ) + tr( "   # The job name\n" ) +
                            "#SBATCH --nodes=" + numProcs->text() + whitespaces( max_length - QString( "#SBATCH --nodes=" + numProcs->text() ).length() ) + tr( "   # Use " ) + numProcs->text() + tr( " compute nodes\n" ) +
                            "#SBATCH --ntasks=" + QString::number( numProcs->text().toInt() * numThreads->text().toInt() ) + whitespaces( max_length - QString( "#SBATCH --ntasks=" + QString::number( numProcs->text().toInt() * numThreads->text().toInt() ) ).length() ) + "   # Use " + QString::number( numProcs->text().toInt() * numThreads->text().toInt() ) + " MPI ranks\n"
                            "#SBATCH --ntasks-per-node=" + numThreads->text() + whitespaces( max_length - QString( "#SBATCH --ntasks-per-node=" + numThreads->text() ).length() ) + tr( "   # Place " ) + numThreads->text() + tr( " ranks on each node\n" ) +
                            "#SBATCH --hint=nomultithread" + whitespaces( max_length - QString( "#SBATCH --hint=nomultithread" ).length() ) + tr( "   # Do not use SMT\n" ) +
                            "#SBATCH --time=00:15:00" + whitespaces( max_length - QString( "#SBATCH --time=00:15:00" ).length() ) + tr( "   # Wallclock limit 15 min\n" ) );
    if ( numThreads->text().toInt() > 1 )
    {
        jobScriptEdit->append( tr( "# Set #threads explicitly" ) + "\nexport OMP_NUM_THREADS=" + numThreads->text() + "\n" );
    }
    if ( initialRunButton->isChecked() )
    {
        jobScriptEdit->append( "export SCOREP_ENABLE_PROFILING=true\n" );
    }
    else if ( finetunedRunButton->isChecked() )
    {
        // check if filtering file exists
        jobScriptEdit->append( "export SCOREP_FILTERING_FILE=" + filterFile );
    }
    jobScriptEdit->append( "export SCOREP_EXPERIMENT_DIRECTORY=" + measurementWindow->instrumentationTab->executableDir + "/" + dirNameEdit->text() + "\n" );

    jobScriptEdit->append( tr( "# Execute job" ) + "\nsrun " + measurementWindow->instrumentationTab->executableName + "\n" );

    editJobScriptWidget->setVisible( true );
    mainWidget->setVisible( false );
}

/**
 * @brief Saves the current measurement configurations to a shell script that can be used to reproduce the measurement.
 * @pre The measurement was performed successfully.
 */
void
ExecutionTab::onCreateShellScriptButtonClicked()
{
    shellScriptErrorLabel->setText( "" );
    QString scriptName = QFileDialog::getSaveFileName( this, "Save", QString::fromStdString( getenv( "HOME" ) ) );
    QFile   file( scriptName );
    if ( !file.open( QFile::WriteOnly | QFile::Text ) )
    {
        QMessageBox::warning( this, tr( "Warning" ), tr( "Cannot save file: " ) + file.errorString() );
        return;
    }

    QTextStream out( &file );
    // create shell script
    out << "#!/bin/bash" << Qt::endl;
    out << "#";
    // calculate number of *
    int diff = abs( QString( tr( "Function: perform scorep measurement for " ) + measurementWindow->instrumentationTab->executableName ).length() - QString( tr( "Format:   " ) + scriptName ).length() );
    if ( QString( tr( "Function: perform scorep measurement for " ) + measurementWindow->instrumentationTab->executableName ).length() >= QString( tr( "Format:   " ) + scriptName ).length() )
    {
        for ( int i = 0; i < QString( tr( "Function: perform scorep measurement for " ) + measurementWindow->instrumentationTab->executableName ).length() + 4; i++ )
        {
            out << "*";
        }
        out << Qt::endl;
        out << tr( "#* Function: perform scorep measurement for " ) << measurementWindow->instrumentationTab->executableName << " *" << Qt::endl;
        out << tr( "#* Format:   " ) << scriptName;
        for ( int i = 0; i < diff; i++ )
        {
            out << " ";
        }
        out << " *" << Qt::endl;
        out << "#";
        for ( int i = 0; i < QString( tr( "Function: perform scorep measurement for " ) + measurementWindow->instrumentationTab->executableName ).length() + 4; i++ )
        {
            out << "*";
        }
        out << Qt::endl;
    }
    else
    {
        for ( int i = 0; i < QString( tr( "Format:   " ) + scriptName ).length() + 4; i++ )
        {
            out << "*";
        }
        out << Qt::endl;
        out << tr( "#* Function: perform scorep measurement for " ) << measurementWindow->instrumentationTab->executableName;
        for ( int i = 0; i < diff; i++ )
        {
            out << " ";
        }
        out << " *" << Qt::endl;
        out << tr( "#* Format:   " ) << scriptName << " *" << Qt::endl;
        out << "#";
        for ( int i = 0; i < QString( tr( "Format:   " ) + scriptName ).length() + 4; i++ )
        {
            out << "*";
        }
        out << Qt::endl;
    }
    out << "export PATH=" << measurementWindow->settings.value( prefix + "/path" ).toString() << ":$PATH" << Qt::endl;
    // check if executable file exists
    out << "[ -f " << measurementWindow->instrumentationTab->executableName << " ] || { echo \"executable file " << measurementWindow->instrumentationTab->executableName << " doesn't exist. Script is aborted.\"; exit 1; }" << Qt::endl;
    // check if executable is instrumented
    out << "nm " << measurementWindow->instrumentationTab->executableName << " | grep -q 'SCOREP' &> /dev/null" << Qt::endl;
    out << "if [ $? != 0 ] ; then" << Qt::endl;
    out << "    echo " + tr( "\"executable file " ) << measurementWindow->instrumentationTab->executableName << tr( " is not instrumented. Script is aborted.\"" ) << Qt::endl;
    out << "    exit 1" << Qt::endl;
    out << "fi" << Qt::endl;
    out << tr( "#unset all SCOREP variables" ) << Qt::endl;
    out << "unset $(echo \"${!SCOREP*}\")" << Qt::endl;
    if ( initialRunButton->isChecked() )
    {
        out << "export SCOREP_ENABLE_PROFILING=true" << Qt::endl;
    }
    else if ( finetunedRunButton->isChecked() )
    {
        // check if filtering file exists
        out << "export SCOREP_FILTERING_FILE=" << filterFile << Qt::endl;
        out << "[ -f " << filterFile << " ] || { echo " + tr( "\"specified filter file doesn't exist. Script is aborted.\"" ) + "; exit 1; }" << Qt::endl;
    }
    out << "export SCOREP_EXPERIMENT_DIRECTORY=" << dirNameEdit->text() << Qt::endl;
    if ( numThreads->value() > 1 )
    {
        out << "export OMP_NUM_THREADS=" << numThreads->value() << Qt::endl;
    }
    if ( !slurmScriptName.isEmpty() )
    {
        out << "sbatch " + slurmScriptName << Qt::endl;
    }
    else
    {
        out << execCommandEdit->text() << Qt::endl;
    }
    file.close();
    file.setPermissions( QFileDevice::ExeOwner | QFileDevice::ReadOwner | QFileDevice::WriteOwner );
    shellScriptErrorLabel->setText( shellScriptErrorLabel->text() + tr( "\nShell script was created successfully" ) );
    shellScriptErrorLabel->setVisible( true );
}

/**
 * @brief Sets the command line options of scorep-score to their default values.
 */
void
ExecutionTab::setDefaultFilterValues()
{
    bufferPercent->setValue( 1 );
    timePerVisit->setValue( 1 );
    type->setCurrentIndex( 0 );
}

/**
 * @brief Shows the filter edit widget to create a filter file and hides the other widgets.
 */
void
ExecutionTab::onCreateManuallyButtonClicked()
{
    measurementWindow->settings.setValue( "measurement/createFilter", "Manual" );
    generateFilterWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    scorepionWidget->setVisible( false );
    editFilterWidget->setVisible( true );
    mainWidget->setVisible( false );
}

/**
 * @brief Saves the created filter file, exports the `SCOREP_FILTERING_FILE` variable and shows further steps.
 */
void
ExecutionTab::onSaveChangesButtonClicked()
{
    QString filter   = "Filter Files (*.filter)";
    QString fileName = QFileDialog::getSaveFileName( this, "Save", filterFile, filter, &filter );
    QFile   file( fileName );

    if ( !file.open( QFile::WriteOnly | QFile::Text ) )
    {
        QMessageBox::warning( this, "Warning", "Cannot save file: " + file.errorString() );
        return;
    }
    QTextStream out( &file );
    QString     text = filterFileEdit->toPlainText();
    out << text;
    file.close();
    filterFile = fileName;

    console->addCommand( "export SCOREP_FILTERING_FILE=" + filterFile.toStdString() );
    setenv( "SCOREP_FILTERING_FILE", const_cast<char*>( filterFile.toStdString().c_str() ), 1 );
    filterInfoLabel->setText( tr( "<font color=\"green\">The filter file was created successfully. Please rerun the application to perform the measurement.</font>" ) );
    filterInfoLabel->setVisible( true );
    executeWidget->setVisible( true );

    editFilterWidget->setVisible( false );
    mainWidget->setVisible( true );

    measurementWindow->settings.setValue( "measurement/filterFile", filterFile );
}

void
ExecutionTab::onDiscardChangesButtonClicked()
{
    editFilterWidget->setVisible( false );
    mainWidget->setVisible( true );
    filterFileEdit->setText( tr( "# initial filter file, starting point for further adaptation. \n"
                                 "#\n"
                                 "# Considerations:\n"
                                 "#  - check possible wildcard options\n"
                                 "#  - check selected functions for relevancy\n"
                                 "#    (w.r.t. your knowledge of the application)\n"
                                 "#\n"
                                 "# The file contains comments for each region providing additional information\n"
                                 "# regarding the respective region.\n"
                                 "#\n"
                                 "# Please refer to the Score-P user guide for more options on filtering.\n" ) +
                             "SCOREP_REGION_NAMES_BEGIN\n"
                             "   EXCLUDE\n"
                             "SCOREP_REGION_NAMES_END" );
}

/**
 * @brief Displays options to generate a filter file.
 */
void
ExecutionTab::onGenerateFilterButtonClicked()
{
    measurementWindow->settings.setValue( "measurement/createFilter", "Automatic" );

    // uncheck all options
    uncheck( profileGroup );

    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    generateFilterWidget->setVisible( true );
    configureFilterWidget->setVisible( false );
    scorepionWidget->setVisible( false );
}

/**
 * @brief Uses the `scorep-score` command to generate a filter, exports the `SCOREP_FILTERING_FILE` variable and shows further steps.
 */
void
ExecutionTab::onGenerateButtonClicked()
{
    int returncode;
    // create initial filter file
    console->execCommand( "cd " + generateFilterProfile.toStdString() + " && scorep-score -g type=" + type->currentText().toStdString() + ",bufferpercent=" + to_string( bufferPercent->value() ) + ",timepervisit=" + to_string( timePerVisit->value() ) + " profile.cubex", returncode );
    if ( returncode == 0 )
    {
        filterFile = generateFilterProfile + "/initial_scorep.filter";
        console->addCommand( "export SCOREP_FILTERING_FILE=" + filterFile.toStdString() );
        setenv( "SCOREP_FILTERING_FILE", const_cast<char*>( filterFile.toStdString().c_str() ), 1 );
        filterInfoLabel->setText( tr( "<font color=\"green\">The filter file was generated successfully. Please rerun the application to perform the measurement.</font>" ) );
        executeWidget->setVisible( true );
        inspectFilterFileButton->setVisible( true );
        inspectFilterFileButton->setText( tr( "Inspect generated filter" ) );
        measurementWindow->settings.setValue( "measurement/filterFile", filterFile );
        measurementWindow->settings.setValue( "measurement/bufferPercent", bufferPercent->value() );
        measurementWindow->settings.setValue( "measurement/timePerVisit", timePerVisit->value() );
        measurementWindow->settings.setValue( "measurement/type", type->currentIndex() );
    }
    else
    {
        filterInfoLabel->setText( tr( "Something went wrong. The filter file could not be generated." ) );
    }
    filterInfoLabel->setVisible( true );
}

/**
 * @brief Slot function that is called when the user selects a profile for creating a filter file.
 */
void
ExecutionTab::selectedProfile()
{
    QRadioButton* selectedButton = dynamic_cast<QRadioButton*>( profileGroup->checkedButton() );
    measurementWindow->settings.setValue( "measurement/filterProfile", selectedButton->text() );
    generateFilterProfile = selectedButton->text().remove( "/profile.cubex" );

    scorepionWidget->setVisible( true );
}

void
ExecutionTab::onUseScorepionButtonClicked()
{
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    configureFilterWidget->setVisible( false );

    measurementWindow->settings.setValue( "measurement/generateFilter", "ScorePion" );
}

/**
 * @brief Shows widget to configure `scorep-score`'s command line options.
 */
void
ExecutionTab::onConfigureScoreOptionsButtonClicked()
{
    configureFilterWidget->setVisible( true );
    measurementWindow->settings.setValue( "measurement/generateFilter", "ScorePScore" );
}

void
ExecutionTab::onBrowseFilterButtonClicked()
{
    measurementWindow->settings.setValue( "measurement/specifyFilter", "Select" );
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    generateFilterWidget->setVisible( false );
    filterOptionsWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    scorepionWidget->setVisible( false );
    QFileDialog dialog( this, tr( "Select your filter file" ), measurementWindow->instrumentationTab->executableDir, tr( "Filter file (*.filter)" ) );
    if ( dialog.exec() )
    {
        filterFile = dialog.selectedFiles().at( 0 );
        measurementWindow->settings.setValue( "measurement/filterFile", filterFile );
        console->addCommand( "export SCOREP_FILTERING_FILE=" + filterFile.toStdString() );
        setenv( "SCOREP_FILTERING_FILE", const_cast<char*>( filterFile.toStdString().c_str() ), 1 );
        executeWidget->setVisible( true );
        filterPathLabel->setVisible( true );
        filterPathLabel->setText( tr( "Selected filter file: " ) + filterFile );
        inspectFilterFileButton->setVisible( true );
        inspectFilterFileButton->setText( tr( "Inspect selected filter" ) );
    }
}

/**
 * @brief This function deletes existing profile buttons, and then creates a new set of buttons for each profile listed in the @p profiles
 * string. The function connects the clicked() signal of each button to the selectedProfile() slot of the ExecutionTab class.
 * @note This function assumes that the @p profiles string contains one profile per line. In @p profileExists() a value is assigned to the string.
 * @see ExecutionTab::selectedProfile()
 * @see ExecutionTab::profileExists()
 */
void
ExecutionTab::addProfileButtons()
{
    // delete existing buttons
    QList<QAbstractButton*> l = profileGroup->buttons();
    if ( !l.empty() )
    {
        for ( QAbstractButton*& b : l )
        {
            profileGroup->removeButton( b );
            profileBoxLayout->removeWidget( b );
        }
    }

    QStringList profileList = profiles.split( REGULAR_EXPRESSION( "\\n" ) );
    foreach( const auto & p, profileList )
    {
        if ( p.endsWith( "profile.cubex" ) )
        {
            QRadioButton* button = new QRadioButton( p );
            profileGroup->addButton( button );
            profileBoxLayout->addWidget( dynamic_cast<QRadioButton*>( button ) );
            connect( button, SIGNAL( clicked() ), this, SLOT( selectedProfile() ) );
        }
    }
}

/**
 * @brief Displays options for creating a filter file.
 */
void
ExecutionTab::onCreateFilterButtonClicked()
{
    measurementWindow->settings.setValue( "measurement/specifyFilter", "Create" );
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    generateFilterWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    scorepionWidget->setVisible( false );
    filterOptionsWidget->setVisible( true );
}
/**
 * @brief Slot function called when the "Take New Measurement" button is clicked.
 * It calls the @p unsetConfigVars() function to unset any previously set Score-P variables.
 * It unchecks all the previously selected options, hides several widgets related to further steps in the measurement process,
 * and deletes any settings from the previous measurement.
 * It checks if the finetuned run is available and enables/disables the finetunedRunButton accordingly.
 */
void
ExecutionTab::onTakeMeasurementButtonClicked()
{
    unsetConfigVars();
    runInfoLabel->setVisible( false );

    // uncheck selected options
    uncheck( profileGroup );
    uncheck( customRunButton );
    uncheck( initialRunButton );
    uncheck( detailedRunButton );
    uncheck( finetunedRunButton );
    uncheck( profileGroup );

    runLabel->setVisible( false );
    selectRunWidget->setEnabled( true );
    executeWidget->setEnabled( true );
    prepareJobWidget->setEnabled( true );
    inspectFilterFileButton->setEnabled( true );
    finetunedRunWidget->setEnabled( true );
    generateFilterWidget->setEnabled( true );
    customRunWidget->setEnabled( true );
    configureFilterWidget->setEnabled( true );
    filterOptionsWidget->setEnabled( true );
    scorepionWidget->setEnabled( true );


    // check if finetuned run is available
    if ( profileExists() )
    {
        finetunedRunButton->setEnabled( true );
        finetunedRunButton->setToolTip( tr( "finetuned run requires filter file" ) );
        addProfileButtons();
    }
    else
    {
        finetunedRunButton->setEnabled( false );
        finetunedRunButton->setToolTip( tr( "Available when profile exists. Please start with initial run." ) );
    }

    // hide further steps
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    showProfileWidget->setVisible( false );
    finetunedRunWidget->setVisible( false );
    generateFilterWidget->setVisible( false );
    filterOptionsWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    filterPathLabel->setVisible( false );
    filterInfoLabel->setVisible( false );
    scorepionWidget->setVisible( false );
    shellScriptErrorLabel->setVisible( false );

    // delete settings from former measurement
    measurementWindow->settings.remove( "measurement/selectedRun" );
    measurementWindow->settings.remove( "measurement/experimentDirName" );
    measurementWindow->settings.remove( "measurement/filterFile" );
    measurementWindow->settings.remove( "measurement/specifyFilter" );
    measurementWindow->settings.remove( "measurement/createFilter" );
    measurementWindow->settings.remove( "measurement/filterProfile" );
    measurementWindow->settings.remove( "measurement/generateFilter" );
    measurementWindow->settings.remove( "measurement/bufferPercent" );
    measurementWindow->settings.remove( "measurement/timePerVisit" );
    measurementWindow->settings.remove( "measurement/type" );
    measurementWindow->settings.remove( "measurement/runSuccess" );
    measurementWindow->settings.remove( "measurement/prepareJob" );
    measurementWindow->settings.remove( "measurement/jobid" );
    profileSettingsPath = "";
}

/**
 * @brief Opens the created profile in Cube and closes the plugin.
 */
void
ExecutionTab::onShowProfileButtonClicked()
{
    QString profilePath = profileSettingsPath;
    if ( profilePath == "" )
    {
        profilePath = measurementWindow->instrumentationTab->executableDir + "/" + dirNameEdit->text() + "/profile.cubex";
    }
    Url          fileUrl( profilePath.toStdString() );
    CubeProxy*   myCube   = CubeProxy::create( fileUrl.toString() );
    CubeIoProxy* myIoCube = static_cast<CubeIoProxy*>( myCube );
    myIoCube->openReport();
    service->openCube( myCube );
}
/**
 *  @brief This function executes a command specified by the user in the "execCommandEdit" widget and saves the resulting profile
 *  data to a file.
 *  @details The name of the file is determined by the text entered in the "dirNameEdit" widget. If the specified file
 *  already exists, the function displays an error message and does not execute the command. The function also updates the
 *  user interface to reflect the state of the measurement process and show further steps.
 *  @param saveSettings Whether the function is called by the @p MeasurementWindow::loadSettings() method, or not.
 */
void
ExecutionTab::onRunApplicationButtonClicked( bool saveSettings )
{
    prepareJobWidget->setVisible( false );
    measurementWindow->settings.setValue( "measurement/prepareJob", "False" );
    int returncode;
    // check if dirname already exists
    console->execCommand( "[ -e " + dirNameEdit->text().toStdString() + " ]", returncode, false );
    QString profilePath = measurementWindow->instrumentationTab->executableDir + "/" + dirNameEdit->text() + "/profile.cubex";

    if ( returncode != 0 || !QFile::exists( profilePath ) || !saveSettings )
    {
        dirNameErrorMessage->setVisible( false );
        if ( saveSettings )
        {
            setenv( "SCOREP_EXPERIMENT_DIRECTORY", const_cast<char*>( dirNameEdit->text().toStdString().c_str() ), 1 );
            console->addCommand( "export SCOREP_EXPERIMENT_DIRECTORY=" + dirNameEdit->text().toStdString() );
            console->execCommand( execCommandEdit->text().toStdString(), returncode );
            if ( returncode == 0 )
            {
                measurementWindow->settings.setValue( "measurement/runSuccess", "True" );
                measurementWindow->settings.setValue( "measurement/profilePath", profilePath );
            }
        }

        if ( ( returncode == 0 && QFile::exists( profilePath ) ) || !saveSettings )
        {
            runLabel->setText( tr( "<font color=\"green\">Your application has been executed.</font>" ) );
            runLabel->setVisible( true );
            showProfileWidget->setVisible( true );

            // presettings can not be changed after run
            selectRunWidget->setEnabled( false );
            executeWidget->setEnabled( false );
            prepareJobWidget->setEnabled( false );
            inspectFilterFileButton->setEnabled( false );
            finetunedRunWidget->setEnabled( false );
            generateFilterWidget->setEnabled( false );
            filterInfoLabel->setVisible( false );
            customRunWidget->setEnabled( false );
            configureFilterWidget->setEnabled( false );
            filterOptionsWidget->setEnabled( false );
            scorepionWidget->setEnabled( false );
        }
        else
        {
            runLabel->setText( tr( "<font color=\"red\">Your application could not be executed</font>" ) );
            runLabel->setVisible( true );
        }
    }
    else
    {
        dirNameErrorMessage->setVisible( true );
        dirNameErrorMessage->setText( tr( "<font color=\"red\">There already exists a file with that name. Please choose another name.</font>" ) );
    }
}

void
ExecutionTab::onUnwindingInfoButtonClicked()
{
    QString message = "";
    if ( !enableUnwinding->isEnabled() )
    {
        message = tr( "<b> Unwinding is not supported by your selected Score-P version.</b><br><br>" );
    }
    message += tr( "<b>Enables recording calling context information for every event.</b> <br>"
                   "The calling context is the call chain of functions to the current position in the running program. "
                   "This call chain will also be annotated with source code information if possible. "
                   "This is a prerequisite for sampling but also works with instrumented applications. "
                   "Note that when tracing is also enabled, Score-P does not write the usual Enter/Leave records into "
                   "the OTF2 trace, but new records." );
    QMessageBox::information( this, tr( "Unwinding info" ), message );
}

void
ExecutionTab::onProfilingInfoButtonClicked()
{
    QString message = "";
    if ( !enableProfiling->isEnabled() )
    {
        message = tr( "<b> Profiling is not supported by your selected Score-P version.</b> <br><br>" );
    }
    message = tr( "<b>In profiling mode, the events are summarized at runtime seperately for each call-path.</b> <br>"
                  "Additionally, support for phases, dynamic regions and parameter-based profiling has been integrated. "
                  "Also in profiling mode, Score-P supports the automatic detection of MPI wait states. Usually such inefficiencies"
                  " are important bottlenecks and are thoroughly investigated by means of automatic trace analysis and subsequent "
                  "visual analysis using a time-line representation. In the case of Score-P wait time profiling, inefficiencies are "
                  "detected immediately when the respective MPI call is completed and stored as an additional metric in the call-path "
                  "profile.<br>"
                  "Score-P implements a call-tree based profiling system. Every node in the call tree represents a "
                  "recorded region. The edges of the tree represent the caller-callee relationship: The children of a "
                  "node are those regions, that are entered/exited within a region. The path from the root to an arbitrary "
                  ", represents a call-path. Thus, every node in the tree identifies also the call-path from the root to "
                  "itself.\nTogether with a node the statistics for the call-path are stored. <b>By default, the runtime and "
                  "the number of visits are recorded. Profile run has lower space requirements.</b>" );
    QMessageBox::information( this, tr( "Profiling info" ), message );
}

void
ExecutionTab::onTracingInfoButtonClicked()
{
    QString message = "";
    if ( !enableTracing->isEnabled() )
    {
        message = tr( "<b> Tracing is not supported by your selected Score-P version</b>" );
    }
    message = tr( "In tracing mode, the performance events are passed to the tracing back-end of Score-P and are written to files. "
                  "This backend uses the newly developed Open Trace Format 2 (OTF2)." );
    QMessageBox::information( this, tr( "Tracing info" ), message );
}

void
ExecutionTab::onVerboseInfoButtonClicked()
{
}

/**
 * @brief Enables or disables unwinding based on the state of the checkbox and sets the correspoding Score-P variable `SCOREP_ENABLE_UNWINDING`.
 * @param state state of @p ExecutionTab::enableUnwinding checkbox
 */
void
ExecutionTab::enableUnwindingClicked( int state )
{
    if ( state == Qt::Unchecked )
    {
        console->addCommand( "export SCOREP_ENABLE_UNWINDING=false" );
        setenv( "SCOREP_ENABLE_UNWINDING", "false", 1 );
        enableUnwinding->setText( tr( "unwinding disabled" ) );
    }
    else if ( state == Qt::Checked )
    {
        console->addCommand( "export SCOREP_ENABLE_UNWINDING=true" );
        setenv( "SCOREP_ENABLE_UNWINDING", "true", 1 );
        enableUnwinding->setText( tr( "unwinding enabled" ) );
    }
}

/**
 * @brief Enables or disables tracing based on the state of the checkbox and sets the correspoding Score-P variable `SCOREP_ENABLE_TRACING`.
 * @param state state of @p ExecutionTab::enableTracing checkbox
 */
void
ExecutionTab::enableTracingClicked( int state )
{
    if ( state == Qt::Unchecked )
    {
        console->addCommand( "export SCOREP_ENABLE_TRACING=false" );
        setenv( "SCOREP_ENABLE_TRACING", "false", 1 );
        enableTracing->setText( tr( "tracing disabled" ) );
    }
    else if ( state == Qt::Checked )
    {
        console->addCommand( "export SCOREP_ENABLE_TRACING=true" );
        setenv( "SCOREP_ENABLE_PROFILING", "true", 1 );
        enableTracing->setText( tr( "tracing enabled" ) );
    }
}

/**
 * @brief Enables or disables profiling based on the state of the checkbox and sets the correspoding Score-P variable `SCOREP_ENABLE_PROFILING`.
 * @param state state of @p ExecutionTab::enableProfiling checkbox
 */
void
ExecutionTab::enableProfilingClicked( int state, bool saveSettings )
{
    if ( state == Qt::Unchecked )
    {
        if ( saveSettings )
        {
            console->addCommand( "export SCOREP_ENABLE_PROFILING=false" );
        }
        setenv( "SCOREP_ENABLE_PROFILING", "false", 1 );
        enableProfiling->setText( tr( "profiling disabled" ) );
    }
    else if ( state == Qt::Checked )
    {
        if ( saveSettings )
        {
            console->addCommand( "export SCOREP_ENABLE_PROFILING=true" );
        }
        setenv( "SCOREP_ENABLE_PROFILING", "true", 1 );
        enableProfiling->setText( tr( "profiling enabled" ) );
    }
}

/**
 * @brief Enables or disables verbose mode based on the state of the checkbox and sets the correspoding Score-P variable `SCOREP_VERBOSE`.
 * @param state state of @p ExecutionTab::beVerbose checkbox
 */
void
ExecutionTab::beVerboseClicked( int state )
{
    if ( state == Qt::Unchecked )
    {
        console->addCommand( "export SCOREP_VERBOSE=false" );
        setenv( "SCOREP_VERBOSE", "false", 1 );
        beVerbose->setText( tr( "verbose disabled" ) );
    }
    else if ( state == Qt::Checked )
    {
        console->addCommand( "export SCOREP_VERBOSE=true" );
        setenv( "SCOREP_VERBOSE", "true", 1 );
        beVerbose->setText( tr( "verbose enabled" ) );
    }
}


void
ExecutionTab::selectedInitialRun( bool saveSettings )
{
    dirNameEdit->setText( "scorep-" + convertDate() + "_initialrun_np" + QString::number( numProcs->value() ) );
    customRunWidget->setVisible( false );
    finetunedRunWidget->setVisible( false );
    generateFilterWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    showProfileWidget->setVisible( false );
    generateFilterWidget->setVisible( false );
    filterOptionsWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    filterPathLabel->setVisible( false );
    filterInfoLabel->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    scorepionWidget->setVisible( false );

    measurementWindow->settings.setValue( "measurement/selectedRun", "Initial" );

    // show info text
    runInfoLabel->setText( tr( "First possible measurement in which a profile is created. Does not deliver a good result, but forms the basis for improved measurements." ) );
    runInfoLabel->setVisible( true );

    // export enable profiling variable
    enableProfilingClicked( 2, saveSettings );
    setRunCommand();

    executeWidget->setVisible( true );
}

/**
 * @brief Sets the execution command for the measurement run based on the instrumentation tab's executable name and directory.
 * It Checks if mpiexec or mpirun are available.
 */
void
ExecutionTab::setRunCommand()
{
    // edit execution command
    int     returncode;
    QString runPath = QString::fromStdString( console->execCommand( "which mpiexec", returncode, false ) ).remove( REGULAR_EXPRESSION( "\\n" ) );
    if ( returncode != 0 )
    {
        runPath = QString::fromStdString( console->execCommand( "which mpirun", returncode, false ) ).remove( REGULAR_EXPRESSION( "\\n" ) );
        if ( returncode != 0 )
        {
            // error could not autodetect run command
        }
    }

    QStringList executableName = measurementWindow->instrumentationTab->executableName.split( "/" );
    QString     dirName        = measurementWindow->instrumentationTab->executableDir;

    execCommandEdit->setText( "cd " + dirName + " && " + runPath + " -n " + QString::number( numProcs->value() ) + " " + executableName.at( executableName.length() - 1 ) );
}

void
ExecutionTab::selectedDetailedRun()
{
    // show info text
    runInfoLabel->setVisible( true );
    runInfoLabel->setText( "The detailed run is available if an improved measurement already exists. Here the application is tuned by providing e.g. PAPI counters." );

    dirNameEdit->setText( "scorep-" + convertDate() + "_detailedrun_np" + QString::number( numProcs->value() ) );
    customRunWidget->setVisible( false );
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    finetunedRunWidget->setVisible( false );
    generateFilterWidget->setVisible( false );

    showProfileWidget->setVisible( false );
    generateFilterWidget->setVisible( false );
    filterOptionsWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    filterPathLabel->setVisible( false );
    filterInfoLabel->setVisible( false );
    scorepionWidget->setVisible( false );
    measurementWindow->settings.setValue( "measurement/selectedRun", "Detailed" );
}

void
ExecutionTab::selectedFinetunedRun()
{
    // show info text
    runInfoLabel->setVisible( true );
    runInfoLabel->setText( "It is necessary that a initial measurement already exists. The finetuned run improves the measurement, not the application, by using a filter." );

    dirNameEdit->setText( "scorep-" + convertDate() + "_finetunedrun_np" + QString::number( numProcs->value() ) );
    customRunWidget->setVisible( false );
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    showProfileWidget->setVisible( false );
    generateFilterWidget->setVisible( false );
    filterOptionsWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    filterPathLabel->setVisible( false );
    filterInfoLabel->setVisible( false );
    scorepionWidget->setVisible( false );
    uncheck( browseFilterButton );
    uncheck( createFilterButton );

    finetunedRunWidget->setVisible( true );
    measurementWindow->settings.setValue( "measurement/selectedRun", "Finetuned" );
    setRunCommand();
}

void
ExecutionTab::selectedCustomRun()
{
    // show info text
    runInfoLabel->setVisible( true );
    runInfoLabel->setText( "Custom run is only recommended for experienced users. Here all possible options can be configured manually." );

    dirNameEdit->setText( "scorep-" + convertDate() + "_customrun_np" + QString::number( numProcs->value() ) );
    customRunWidget->setVisible( true );
    executeWidget->setVisible( false );
    prepareJobWidget->setVisible( false );
    inspectFilterFileButton->setVisible( false );
    generateFilterWidget->setVisible( false );
    finetunedRunWidget->setVisible( false );
    showProfileWidget->setVisible( false );
    generateFilterWidget->setVisible( false );
    filterOptionsWidget->setVisible( false );
    configureFilterWidget->setVisible( false );
    filterPathLabel->setVisible( false );
    scorepionWidget->setVisible( false );
    filterInfoLabel->setVisible( false );
    measurementWindow->settings.setValue( "measurement/selectedRun", "Custom" );
}

/**
 * @brief Adjusts default experiment directory name depending on the number of processes.
 * @param number Number of processes.
 */
void
ExecutionTab::selectedNumberOfProcesses( int number )
{
    measurementWindow->settings.setValue( "measurement/numProcs", number );
    dirNameEdit->setText( dirNameEdit->text().split( "np" ).at( 0 ) + "np" + QString::number( number ) );
    setRunCommand();
}

/**
 * @brief Exports the `OMP_NUM_THREADS` environment variable according to the number of threads selected.
 * @param number Number of threads.
 */
void
ExecutionTab::selectedNumberOfThreads( int number )
{
    measurementWindow->settings.setValue( "measurement/numThreads", number );
    if ( number == 1 )
    {
        unsetenv( "OMP_NUM_THREADS" );
        console->addCommand( "unset OMP_NUM_THREADS", true );
    }
    else
    {
        setenv( "OMP_NUM_THREADS", const_cast<char*>( std::to_string( number ).c_str() ), 1 );
        console->addCommand( "export OMP_NUM_THREADS=" + std::to_string( number ), true );
    }
}

/**
 * @brief Checks whether a profile exists in the executable directory by using the find command.
 * @return True if profile file exists, false otherwise.
 */
bool
ExecutionTab::profileExists()
{
    int returncode;
    profiles = QString::fromStdString( console->execCommand( "find " + measurementWindow->instrumentationTab->executableDir.toStdString() + " -name profile.cubex", returncode, false ) );
    if ( returncode == 0 && profiles != "" )
    {
        return true;
    }
    return false;
}

/**
 * @brief Converts the current date and time into a formatted string.
 * @return  A QString representing the current date and time in the format "yyyy-MM-dd_hh:mm:sszzz".
 */
QString
ExecutionTab::convertDate()
{
    QString     datetime = QDateTime( QDateTime::currentDateTime() ).toString( "dd:MM:yyyy:hh:mm:sszzz" );
    QStringList date     = datetime.split( ":" );
    return date[ 2 ] + date[ 1 ] + date[ 0 ] + "_" + date[ 3 ] + date[ 4 ] + "_" + date[ 5 ];
}
