/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2022-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include "console.h"
#include <array>
#include <iostream>
#include <fstream>
#include <stdio.h>
#include "measurementwindow.h"

using namespace console;
using namespace measurementwindow;

/**
 * @param service A pointer to the ContextFreeServices object used by the plugin.
 * @param parent An optional parent widget. In this application MeasurementWindow is used as parent.
 */
Console::Console( cubepluginapi::ContextFreeServices* service, QWidget* parent ) : QWidget( parent )
{
    this->service           = service;
    this->measurementWindow = static_cast<MeasurementWindow*>( parent );

    consoleLayout      = new QVBoxLayout;
    consoleTitleLayout = new QHBoxLayout;
    console            = new QTextEdit;
    consoleTitle       = new QLabel( "measurement : cube -- console" );
    consoleButton      = new QPushButton( "\u1433" );

    connect( consoleButton, SIGNAL( clicked() ), this, SLOT( onConsoleButtonClicked() ) );

    console->setMinimumWidth( 350 );

    console->setAlignment( Qt::AlignTop );
    console->setReadOnly( true );
    console->setToolTip( tr( "The console shows the commands executed in the command line" ) );

    bgcolor = console->palette();
    bgcolor.setColor( QPalette::Base, Qt::black );
    console->setPalette( bgcolor );
    QFont font( "TypeWriter" );

    consoleTitle->setFont( font );
    consoleTitle->setStyleSheet( "background-color: #504b52; color: white" );
    consoleTitle->setAlignment( Qt::AlignCenter );
    consoleTitle->setMaximumHeight( 33 );
    consoleTitle->setMinimumHeight( 33 );

    scrollArea = new QScrollArea;
    scrollArea->setBackgroundRole( QPalette::Dark );
    scrollArea->setMinimumWidth( 350 );
    scrollArea->setWidget( console );
    scrollArea->setWidgetResizable( true );

    consoleButton->setMaximumWidth( 33 );
    consoleButton->setMaximumHeight( 33 );
    consoleButton->setToolTip( tr( "Hide console" ) );

    consoleTitleLayout->addWidget( consoleButton );
    consoleTitleLayout->addWidget( consoleTitle );

    consoleLayout->addLayout( consoleTitleLayout );
    consoleLayout->addWidget( scrollArea );
    consoleLayout->setAlignment( Qt::AlignTop );

    setLayout( consoleLayout );
}

/**
 * @brief A slot method that is called when the console button is clicked. Toggles the visibility of the console window.
 */
void
Console::onConsoleButtonClicked()
{
    if ( console->isVisible() )
    {
        consoleTitle->setVisible( false );
        scrollArea->setVisible( false );
        console->setVisible( false );
        consoleButton->setText( "\u1438" );
        consoleButton->setToolTip( tr( "Show console" ) );
    }
    else
    {
        consoleTitle->setVisible( true );
        scrollArea->setVisible( true );
        console->setVisible( true );
        consoleButton->setText( "\u1433" );
        consoleButton->setToolTip( tr( "Hide console" ) );
    }
}

/**
 * @brief Handles QEvent of type QEvent::ToolTip for Console widget.It shows the tooltip for the widget under the mouse
 * pointer, if any. If there is no widget under the mouse pointer, it hides the tooltip and ignores the event.
 * @param event A pointer to the QEvent being processed.
 * @return Returns true if the event is handled; otherwise, returns false.
 */
bool
Console::event( QEvent* event )
{
    if ( event->type() == QEvent::ToolTip )
    {
        QHelpEvent* helpEvent = static_cast<QHelpEvent*>( event );
        QWidget*    widget    = childAt( helpEvent->globalPos() );
        if ( widget != nullptr )
        {
            QToolTip::showText( helpEvent->globalPos(), widget->toolTip() );
        }
        else
        {
            QToolTip::hideText();
            event->ignore();
        }
        return true;
    }
    return QWidget::event( event );
}

/**
 * @brief Executes a command and returns its output as a string. The method executes the given command using popen()
 * function and reads its output into a string buffer. The command's return code is stored in the integer variable passed
 * by reference to the function.
 * @param cmd The command to execute.
 * @param returncode A reference to an integer variable to store the return code of the command.
 * @return A string containing the output of the command.
 * @throws std::runtime_error Thrown if popen() fails to open the pipe.
 */
std::string
Console::execute( const char* cmd, int& returncode )
{
    std::array<char, 128> buffer;
    std::string           result;

    // NOLINTNEXTLINE (suppress code checker warning)
    auto pipe = popen( cmd, "r" ); // perhaps execve is better

    if ( !pipe )
    {
        throw std::runtime_error( "popen() failed!" );
    }

    while ( !feof( pipe ) )
    {
        if ( fgets( buffer.data(), 128, pipe ) != nullptr )
        {
            result += buffer.data();
        }
    }

    returncode = pclose( pipe );
    return result;
}

/**
 * @brief Executes a command and returns its output as a string. The method executes the given command using the execute()
 * method. If showOutput is true, the command is added to the virtual console. If moduleCommand is true, the command's
 * output is redirected to stderr instead of stdout.
 * @param cmd_str The command to execute.
 * @param returncode A reference to an integer variable to store the return code of the command.
 * @param showOutput Whether to display the output in the virtual console.
 * @param moduleCommand Whether the command is a module command.
 * @param saveSettings Whether to save the command in the virtual console.
 * @return A string containing the output of the command.
 */
std::string
Console::execCommand( std::string cmd_str, int& returncode, bool showOutput, bool moduleCommand, bool saveSettings )
{
    // show command on virtual console
    if ( showOutput )
    {
        addCommand( cmd_str, saveSettings );
    }

    // write stderr into file
    std::string stderrFile = static_cast<std::string>( std::getenv( "HOME" ) ) + "/stderr.txt";
    cmd_str += " 2>" + stderrFile;
    char* cmd = const_cast<char*>( cmd_str.c_str() );

    // execute command
    std::string result = execute( cmd, returncode );

    std::ifstream     infile( stderrFile );
    std::stringstream stderr;
    stderr << infile.rdbuf();

    // check returncode
    if ( returncode == EXIT_SUCCESS )
    {
        if ( showOutput )
        {
            // module commands write output to stderr instead of stdout
            if ( moduleCommand )
            {
                addOutput( stderr.str(), saveSettings );
            }
            else
            {
                addOutput( result, saveSettings );
            }
        }
    }
    else
    {
        if ( showOutput )
        {
            addOutput( stderr.str(), saveSettings );
        }
    }

    if ( moduleCommand )
    {
        result = stderr.str();
    }

    std::remove( const_cast<char*>( stderrFile.c_str() ) );
    return result;
}

/**
 * @brief Displays a command on the virtual console.
 * @param cmd The command to display.
 * @param saveSettings Whether to save the command to the command history.
 */
void
Console::addCommand( std::string cmd, bool saveSettings )
{
    console->setTextColor( Qt::yellow );
    console->append( "cube@measurement: " );
    console->setTextColor( Qt::white );
    console->append( QString::fromStdString( cmd ) );
    console->verticalScrollBar()->setValue( console->verticalScrollBar()->maximum() );
    if ( saveSettings )
    {
        measurementWindow->settings.setValue( "measurement/consoleText", console->toHtml() );
    }
}

/**
 * @brief Displays the output on the virtual console.
 * @param output The output to display.
 * @param saveSettings Whether to save the output to the command history.
 */
void
Console::addOutput( std::string output, bool saveSettings )
{
    console->setTextColor( Qt::white );
    console->append( QString::fromStdString( output ) );
    console->verticalScrollBar()->setValue( console->verticalScrollBar()->maximum() );
    if ( saveSettings )
    {
        measurementWindow->settings.setValue( "measurement/consoleText", console->toHtml() );
    }
}
