/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2022                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>

#include "JSCHybridCommunicationEfficiencyTest.h"


using namespace advisor;


JSCCommunicationEfficiencyTest::JSCCommunicationEfficiencyTest( cube::CubeProxy*      cube,
                                                                JSCSerialisationTest* _pop_ser,
                                                                JSCTransferTest*      _pop_transeff ) : PerformanceTest( cube ),
    pop_ser( _pop_ser ),
    pop_transeff( _pop_transeff )
{
    original_scout_cubex = scout_metrics_available( cube );
    setName( tr( "MPI Communication Efficiency" ).toUtf8().data() );
    setWeight( 1 );   // need to be adjusted
    max_comp_time = nullptr;
    max_runtime   = nullptr;

    max_comp_time = cube->getMetric( "mpi_comp" );
    if ( max_comp_time == nullptr )
    {
        adjustForTest( cube );
    }
    max_comp_time = cube->getMetric( "mpi_comp" );
    if ( max_comp_time == nullptr )
    {
        setWeight( 0.2 );
        setValue( 0. );
        return;
    }
    max_runtime = cube->getMetric( "execution" );

    cube::metric_pair metric;
    metric.first  = max_runtime;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmetrics.push_back( metric );

    metric.first  = max_comp_time;
    metric.second = cube::CUBE_CALCULATE_INCLUSIVE;
    lmax_comp_time.push_back( metric );
}

double
JSCCommunicationEfficiencyTest::calculateForScout( const cube::list_of_cnodes& cnodes ) const
{
    if ( pop_ser == nullptr || pop_transeff == nullptr )
    {
        return 0.;
    }
    double pop_ser_value     = pop_ser->analyze( cnodes );
    double pop_transef_value = pop_transeff->analyze( cnodes );
    return pop_ser_value *   pop_transef_value;
}

void
JSCCommunicationEfficiencyTest::applyCnode( const cube::Cnode*, const cube::CalculationFlavour,
                                            const bool )
{
}

void
JSCCommunicationEfficiencyTest::applyCnode( const cube::list_of_cnodes& cnodes,
                                            const bool                  direct_calculation  )
{
    scout_cubex = original_scout_cubex && !direct_calculation;
    setValue(  analyze( cnodes ) );
}

double
JSCCommunicationEfficiencyTest::analyze( const cube::list_of_cnodes& cnodes,
                                         cube::LocationGroup* ) const
{
    if ( !scout_cubex )
    {
        if ( max_runtime == nullptr || max_comp_time == nullptr )
        {
            return 0.;
        }
        cube::value_container inclusive_values1;
        cube::value_container exclusive_values1;
        cube->getSystemTreeValues( lmetrics,
                                   cnodes,
                                   inclusive_values1,
                                   exclusive_values1 );

        cube::value_container inclusive_values2;
        cube::value_container exclusive_values2;
        cube->getSystemTreeValues( lmax_comp_time,
                                   cnodes,
                                   inclusive_values2,
                                   exclusive_values2 );

        const std::vector<cube::LocationGroup*>& _lgs       = cube->getLocationGroups();
        double                                   comm_value = std::numeric_limits<double>::lowest();
        for ( std::vector<cube::LocationGroup*>::const_iterator iter = _lgs.begin(); iter != _lgs.end(); ++iter )
        {
            double max_total_time_value_ =  inclusive_values1[ ( *iter )->get_sys_id() ]->getDouble();
            double max_comp_time_value_  =  inclusive_values2[ ( *iter )->get_sys_id() ]->getDouble();
            double _comm_value           =  max_comp_time_value_ / max_total_time_value_;
            comm_value = std::max( comm_value, _comm_value );
        }
        return comm_value;
    }
    else
    {
        return calculateForScout( cnodes );
    }
}




const std::string&
JSCCommunicationEfficiencyTest::getCommentText() const
{
    return no_comment;
}


// ------ overview tests ---------

bool
JSCCommunicationEfficiencyTest::isActive() const
{
    return ( pop_ser != nullptr && pop_transeff != nullptr &&
             pop_ser->isActive() && pop_transeff->isActive() )
           ||
           (

        ( max_comp_time != nullptr && max_runtime != nullptr ) );
};

bool
JSCCommunicationEfficiencyTest::isIssue() const
{
    return false;
};

void
JSCCommunicationEfficiencyTest::adjustForTest( cube::CubeProxy* cube ) const
{
    add_execution_time( cube );
    add_mpi_comp_time( cube );
}
