/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2015-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <QFrame>
#include <QVBoxLayout>
#include <QGridLayout>
#include <QGroupBox>
#include <QApplication>
#include <QClipboard>
#include <QListIterator>
#include <QHeaderView>
#include <QFontMetrics>

#include "CubeAdvisorProgress.h"
#include "CubeRatingWidget.h"
#include "CubeHelpButton.h"
#include "CubeRegion.h"
#include "PerformanceAnalysis.h"
#include "CubeTestWidget.h"
#include "PluginServices.h"
#include "HelpBrowser.h"
#include "CubeAdvice.h"
#include "CubeAdvisorTableWidgetItem.h"
#include "TreeItem.h"
#include "Future.h"
#include "Globals.h"

using namespace advisor;

extern cubepluginapi::PluginServices* advisor_services;


CubeRatingWidget::CubeRatingWidget( cubepluginapi::TabInterface* _tab,
                                    QString                      _title,
                                    PerformanceAnalysis*         _analysis,
                                    QWidget*                     parent ) : QWidget( parent ), analysis( _analysis ), tab( _tab )
{
    calculating_in_progress = false;
    analysis_possible       = false;
    title                   = _title;
    QVBoxLayout* top_layout = new QVBoxLayout();
    callpath_frame = new QGroupBox( title );
    main_grid      = new QGridLayout();
    callpath_frame->setLayout( main_grid );
    future = advisor_services->createFuture( _tab );

    QList<PerformanceTest*> _tests = analysis->getPerformanceTests();


    foreach( PerformanceTest * test, _tests )
    {
        addPerformanceTest( test );
        analysis_possible |=  ( test->isActive() );
    }

    top_layout->addWidget( callpath_frame );

    setLayout( top_layout );
    updatetimer = new QTimer( this );
    connect( updatetimer, SIGNAL( timeout() ), this, SLOT( calculationProgress() ) );
}

CubeRatingWidget::~CubeRatingWidget()
{
    QListIterator<CubeTestWidget*> i( list_of_tests );
    while ( i.hasNext() )
    {
        delete ( i.next() );
    }
}

void
CubeRatingWidget::copyMetricsValues(  const cube::list_of_cnodes& lcnodes )
{
    for ( cube::list_of_cnodes::const_iterator iter = lcnodes.begin(); iter != lcnodes.end(); ++iter )
    {
        cube::Cnode*                   cnode     = iter->first;
        const std::string&             name      = ( cnode != nullptr ) ? cnode->get_callee()->get_name() : "";
        QClipboard*                    clipboard = QApplication::clipboard();
        QListIterator<CubeTestWidget*> i( list_of_tests );
        i.toFront();
        QString text_to_copy = title + "\t" + QString::fromStdString( name ) + "\n---------------\n" + tr( "POP Metrics" ) + "\t" + tr( "Min\tAvg\tMax" ) + "\n";
        while ( i.hasNext() )
        {
            CubeTestWidget* _t = i.next();
            if ( !_t->isActive() )
            {
                continue;
            }

            if ( _t->getProgressBar() != nullptr )
            {
                if ( _t->getProgressBar()->isSingleValue() )
                {
                    text_to_copy += QString( "%1 \t %2\t %3 \t %4 \n" ).arg( _t->getName()->text().trimmed() ).arg( "" ).arg( _t->getProgressBar()->getAvgValue() ).arg( "" );
                }
                else
                {
                    text_to_copy += QString( "%1 \t %2\t %3 \t %4 \n" ).arg( _t->getName()->text().trimmed() ).arg( _t->getProgressBar()->getMinValue() ).arg( _t->getProgressBar()->getAvgValue() ).arg( _t->getProgressBar()->getMaxValue() );
                }
            }
            else
            {
                text_to_copy += QString( "%1 \t %2\t %3 \t %4 \n" ).arg( _t->getName()->text().trimmed() ).arg( "" ).arg( _t->getValue()->text().trimmed() ).arg( "" );
            }
        }

        clipboard->setText( text_to_copy );
        advisor_services->setMessage( tr( "Copied values of analysis \"" ) + title + tr( "\" into clipboard." ) );
    }
}


void
CubeRatingWidget::addPerformanceTest( PerformanceTest* t )
{
    CubeTestWidget* test_widget = new CubeTestWidget( t );
    int             n           = list_of_tests.size();
    main_grid->addWidget( test_widget->getName(), n, 0 );
    main_grid->addWidget( test_widget->getValue(), n, 1 );
    Bar* bar = test_widget->getProgressBar();
    if ( bar != nullptr )
    {
        bar->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Minimum );
        main_grid->addWidget( bar, n, 2 );
    }
    main_grid->addWidget( test_widget->getValueText(), n, 3 );
    HelpButton* help = new HelpButton( t->getHelpUrl(), t->isActive() );
    main_grid->addWidget( help, n, 4 );
    list_of_tests << test_widget;
}


void
CubeRatingWidget::apply( const cube::list_of_cnodes& cnodes, const bool direct_calculation )
{
    calculating_in_progress = true;
    if ( cnodes.size() != 0 )
    {
        const std::string& name = ( cnodes.size() == 1 ) ? cnodes[ 0 ].first->get_callee()->get_name() :
                                  tr( "...cnodes..." ).toUtf8().data();
        QFontMetrics fm            = this->fontMetrics();
        int          width         = callpath_frame->width() - 2 * fm.boundingRect( "..." ).width();
        QString      correctedName = fm.elidedText( title + QString( ": %1 " ).arg( QString::fromStdString( name ) ), Qt::ElideMiddle, width );
        callpath_frame->setTitle( correctedName  );
    }
    elapsedTimer.start();
    QListIterator<CubeTestWidget*> i( list_of_tests );
    calculations.clear();
    QHash<PerformanceTest*, ParallelCalculation*> tests;
    i.toBack();
    while ( i.hasPrevious() )
    {
        CubeTestWidget*      tt   = i.previous();
        ParallelCalculation* pcal =  new ParallelCalculation( &calculations, tt->getPerformanceTest(), cnodes, direct_calculation );
        tests[ tt->getPerformanceTest() ] = pcal;
        calculations[ pcal ]              = tt;
    }

    QHashIterator<PerformanceTest*, ParallelCalculation*> ii( tests );
    while ( ii.hasNext() )
    {
        ii.next();
        QList<PerformanceTest*> _tests = ii.key()->getPrereqs();
        foreach( PerformanceTest * t, _tests )
        {
            ii.value()->addPrereq( tests[ t ] );
        }
    }
    advisor_services->debug() << "[Advisor] The calculation operation took" << elapsedTimer.elapsed() << "milliseconds";

    for ( ParallelCalculation* tt : tests.values() )
    {
        future->addCalculation( tt );  // calculation data
    }


    connect( future, SIGNAL( calculationFinished() ), this, SLOT( calculationFinished() ) );
    connect( future, SIGNAL( calculationStepFinished() ), this, SLOT( calculationStepFinished() ) );
    elapsedTimer.start();
    elapsedStageTimer.start();
    future->startCalculation();
    updatetimer->start( 1000 );
}


void
CubeRatingWidget::invalidateAnalysis()
{
    if ( !analysis_possible )
    {
        return;
    }
    table->clearContents();
    table->setRowCount( 0 );
}


void
CubeRatingWidget::tableItemClicked( QTableWidgetItem* _item )
{
    if ( !analysis_possible )
    {
        return;
    }
    CubeAdvisorTableWidgetItem* item = dynamic_cast<CubeAdvisorTableWidgetItem*>( _item );
    if ( item == nullptr )
    {
        return;
    }
    cubegui::TreeItem* callpath_item = const_cast<cubegui::TreeItem*>( item->getCallPathItem() );
    callpath_item->setExpanded( false );
    advisor_services->selectItem( callpath_item, false );
}


void
CubeRatingWidget::calculationStepFinished()
{
    calculationProgress();
    advisor_services->debug() << tr( "[Advisor] The calculation stage operation took" ) << elapsedStageTimer.restart() << tr( "milliseconds" );
}


void
CubeRatingWidget::calculationFinished()
{
    calculating_in_progress = false;
    calculationProgress();
    updatetimer->stop();
    advisor_services->setMessage( tr( "Calculation is finished." ) );
    advisor_services->debug() << tr( "[Advisor] The calculation operation took" ) << elapsedTimer.elapsed() << tr( "milliseconds" );
    disconnect( future, SIGNAL( calculationFinished() ), this, SLOT( calculationFinished() ) );
    disconnect( future, SIGNAL( calculationStepFinished() ), this, SLOT( calculationStepFinished() ) );
}

void
CubeRatingWidget::calculationProgress()
{
    if ( !guard.tryLock() )
    {
        return;
    }
    QListIterator<CubeTestWidget*> i( list_of_tests );
    i.toBack();
    while ( i.hasPrevious() )
    {
        CubeTestWidget* tt = i.previous();
        tt->updateCalculation();
    }
    guard.unlock();
}



void
ParallelCalculation::calculate()
{
    QElapsedTimer t;
    qDebug() << QObject::tr( "[Advisor] Start calculation " ) << test->name().c_str();
    t.start();
    guard.lock();
    ( *calculations )[ this ]->setCalculating( true );
    guard.unlock();
    test->apply( cnodes, direct_calculation );
    guard.lock();
    ( *calculations )[ this ]->setCalculating( false );
    guard.unlock();
    foreach( ParallelCalculation * _pr, toNotify )
    {
        _pr->notifiedBy( this );
    }
    qDebug() << QObject::tr( "[Advisor] Finished calculation " ) << test->name().c_str() << QObject::tr( "in " ) << t.elapsed() << QObject::tr( "milliseconds" );
}
