/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"


#include "VersionCheck.h"

#include <iostream>
#include <fstream>
#include <sstream>
#include <limits>

#include <QMessageBox>
#include <QRegularExpression>

#include "Constants.h"
#include "Globals.h"
#include "Environment.h"
#include "DownloadThread.h"

using namespace std;
using namespace cubegui;


VersionCheck::VersionCheck( QWidget* p ) : parent( p )
{
    url              = QUrl( UPDATE_CHECK_URL );
    download         = NULL;
    update_Available = false;
}

void
VersionCheck::checkForUpdates()
{
    Globals::setStatusMessage( tr( "Check for new version..." ) );
    download = new DownloadThread( url );
    connect( download, SIGNAL( downloadFinished( QUrl, QByteArray ) ),
             this, SLOT( check( QUrl, QByteArray ) ) );
    download->loadList();
}

void
VersionCheck::check( const QUrl&, const QByteArray& buffer )
{
    const QString err         = tr( "Check for latest CUBE version failed." );
    const QString updateAvail = tr( "A new major version of CUBE is <a href=\"http://www.scalasca.org/software/cube-4.x/download.html\">available</a>." );
    const QString noUpdate    = tr( "You have the latest version of CUBE." );

    latestRelease = "";
    QString message = err;
    if ( !buffer.isNull() )
    {
        QString pattern( CUBEGUI_VERSION );

        QStringList currentVersionParts = pattern.split( QRegularExpression( "[\\.-]" ) );
        if ( currentVersionParts.size() < 2 )
        {
            Globals::setStatusMessage( tr( "Current version is not a release." ) );
            return;
        }
        QString currentMajor = currentVersionParts.at( 0 ).trimmed().toLower();
        QString currentMidle = currentVersionParts.at( 1 ).trimmed().toLower();
        QString currentMinor = ( currentVersionParts.size() >= 3 ) ? currentVersionParts.at( 2 ).trimmed().toLower() : "0";

        QString content( buffer );

        QStringList list = content.split( "\n" );

        QStringListIterator javaStyleIterator( list );
        while ( javaStyleIterator.hasNext() )
        {
            QString     line = javaStyleIterator.next();
            QStringList pair = line.split( QRegularExpression( "\\s*:\\s*" ) );
            if ( pair.size() < 2 ) // ignore malformed strings
            {
                continue;
            }
            QString field = pair.at( 0 ).trimmed().toLower();
            QString value = pair.at( 1 ).trimmed().toLower();

            if ( field == "version" )
            {
                QStringList versionParts = value.split( QRegularExpression( "\\." ) );
                QString     major        = ( versionParts.size() > 0 ) ? versionParts.at( 0 ).trimmed().toLower() : "";
                QString     midle        = ( versionParts.size() > 1 ) ? versionParts.at( 1 ).trimmed().toLower() : "";
                QString     minor        = ( versionParts.size() > 2 ) ? versionParts.at( 2 ).trimmed().toLower() : "0";

                bool ok     = true;
                int  _major = major.toInt( &ok );
                if ( !ok )
                {
                    return;
                }
                int _midle = midle.toInt( &ok );
                if ( !ok )
                {
                    return;
                }
                int _minor = minor.toInt( &ok );
                if ( !ok )
                {
                    _minor = 0;
                }
                int _currentMajor = currentMajor.toInt( &ok );
                if ( !ok )
                {
                    return;
                }
                int _currentMidle = currentMidle.toInt( &ok );
                if ( !ok )
                {
                    return;
                }
                int _currentMinor = currentMinor.toInt( &ok );
                if ( !ok )
                {
                    _currentMinor = 0;
                }

                if ( ( _major > _currentMajor ) ||
                     (
                         ( _major == _currentMajor ) &&
                         ( _midle > _currentMidle )
                     ) ||
                     (
                         ( _major == _currentMajor ) &&
                         ( _midle == _currentMidle ) &&
                         ( _minor > _currentMinor )
                     )
                     )
                {
                    update_Available = true;
                    if (    ( _major > _currentMajor ) ||
                            (
                                ( _major == _currentMajor ) &&
                                ( _midle > _currentMidle )
                            )
                            )
                    {
                        message       = updateAvail;
                        latestRelease = buffer;
                    }
                }
                else
                {
                    message = noUpdate;
                }
            }
        }
    }
    else
    {
        message =  err;
    }
    download->deleteLater();

    emit updateCheckFinished( message );
    Globals::setStatusMessage( message );

    //QMessageBox::information( parent, tr( "Check for latest version" ), message );
}
