/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBEGUI_REMOTEFILESYSTEMMODEL2_H
#define CUBEGUI_REMOTEFILESYSTEMMODEL2_H

#include <QAbstractItemModel>
#include <QFileIconProvider>
#include <QSortFilterProxyModel>
#include <QDebug>

#include "CubeFileSystemRequest.h"
#include "CubeClientConnection.h"


namespace cubegui
{
enum FSColumns { NAME, SIZE, DATE, COLUMNS };
enum UserRoles { LSIZE = Qt::UserRole, LDATE, IS_DIR };

class Directory;
/**
 *  helper class for remote file system access
 */
class Directory : public QObject
{
    Q_OBJECT
private:
    QString serverUrl;                            // url of the cube server
    cube::ClientConnection::Ptr clientConnection; // client connection to cube server
    std::vector<cube::FileInfo> files;            // directory contents (non-hidden)
    QString                     networkError;     // network error message

public:
    /**
     * @brief Directory
     * @param serverUrl url of the cube server, e.g. cube://localhost:3300
     */
    Directory( const QString& serverUrl );

    virtual
    ~Directory();

    /** updates the file list to the directory contents of the given path */
    void
    update( const QString& path );

    /** gets the directory contents */
    const std::vector<cube::FileInfo>&
    getFiles() const;

    /** if an error occurred, this method returns the error message */
    QString
    getErrorMessage() const;
};

//====================================================================================================
/**
 * @class RemoteFileSystemModel
 * @brief The RemoteFileSystemModel class connects to a cube server to get directory contents of the remote file system
 */
class RemoteFileSystemModel : public QAbstractItemModel
{
    Q_OBJECT
public:
    /**
     * @brief RemoteFileSystemModel
     * @param serverUrl url of the cube server, e.g. cube://localhost:3300
     */
    RemoteFileSystemModel( const QString& serverUrl );
    virtual
    ~RemoteFileSystemModel();

    // QAbstractItemModel interface
    virtual QModelIndex
    index( int                row,
           int                column,
           const QModelIndex& parent ) const;
    virtual QModelIndex
    parent( const QModelIndex& child ) const;
    virtual int
    rowCount( const QModelIndex& parent ) const;
    virtual int
    columnCount( const QModelIndex& parent ) const;
    virtual QVariant
    data( const QModelIndex& index,
          int                role ) const;
    virtual Qt::ItemFlags
    flags( const QModelIndex& index ) const;
    virtual bool
    hasChildren( const QModelIndex& parent ) const;
    virtual QVariant
    headerData( int             section,
                Qt::Orientation orientation,
                int             role ) const;
    QModelIndex
    getModelIndex( const QString& path );

    bool
    containsFile( const QString& absoluteFile ) const; // true, if current directory contains file

    /**
     * @brief itemActivated handles user selection. If the selected item is a directory, the
     * model is filled with the directory contents, otherwise @see fileSelected is emitted.
     * @param index the selected item of the model
     */
    void
    itemActivated( QModelIndex index );

    /**
     * @brief itemActivated emits @see fileClicked with the clicked tree item
     * @param index the selected item of the model
     */
    void
    itemClicked( QModelIndex index );

    /**
     * @brief setDirectory sets the model contents to the contents of the given directory
     * emits directorySelectionFinished() after the directory contents are loaded
     * @param directoryPath
     */
    void
    setDirectory( QString directoryPath );

signals:
    /**
     * @brief fileSelected is emitted, if the item of @see itemActivated is a file
     * @param filename filename of the selected item
     */
    void
    fileSelected( QString filename );

    /**
     * @brief fileSelected is emitted, with the corresponding tree item of @see itemClicked is a file
     * @param filename the filename which corresponds to the clicked QModelIndex
     */
    void
    fileClicked( QString filename );
    void
    directorySelected( QString dirname );

private:
    Directory                   directory;        // remote directory
    std::vector<cube::FileInfo> files;            // directory contents
    QFileIconProvider           iconProvider;
    const QString               spacing = " ";    // spacing for the columns in the table view

    cube::FileInfo*
    getFile( const QModelIndex& idx ) const;

    void
    updateDirectoryContents();
};

/**
 * @brief The FileSortFilterProxyModel class sorts the items of RemnoteFileSystemModel
 */
class FileSortFilterProxyModel : public QSortFilterProxyModel
{
public:
    void
    itemActivated( QModelIndex index )    // calls itemActivated in source model
    {
        RemoteFileSystemModel* model = dynamic_cast<RemoteFileSystemModel*> ( sourceModel() );
        model->itemActivated( this->mapToSource( index ) );
    }
    void
    itemClicked( QModelIndex index )    // calls itemActivated in source model
    {
        RemoteFileSystemModel* model = dynamic_cast<RemoteFileSystemModel*> ( sourceModel() );
        model->itemClicked( this->mapToSource( index ) );
    }

protected:
    bool
    lessThan( const QModelIndex& left,
              const QModelIndex& right ) const;
};
}
#endif // REMOTEFILESYSTEMMODEL_H
