/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBEGUI_REMOTEFILEDIALOG_H
#define CUBEGUI_REMOTEFILEDIALOG_H

#include <QDialog>
#include <QStack>
#include <QTreeView>

namespace cubegui
{
class RemoteFileSystemModel;
class FileSortFilterProxyModel;

class PathWidget : public QWidget
{
    Q_OBJECT
public:
    PathWidget( const QString& path );

    void
    setPath( const QString& path );

    QString
    getPath() const;

signals:
    void
        pathChanged( QString );

private:
    QString         path;
    QString         prevPath;  // path before having selected a parent
    QStack<QString> forwards;
    QStack<QString> backwards;
    bool            buttonsUsed    = false; // action triggered by back or forward button
    bool            parentSelected = false; // action triggered by path button

    QStringList
    getPathList( const QString& pathStr,
                 const QString& currentPath );
};

class UrlWidget : public QWidget
{
    Q_OBJECT
public:
    UrlWidget( const QString& url );

    void
    setUrl( const QString& url );

    QString
    getUrl() const;

signals:
    void
        urlChanged( QString );
private:
    QLineEdit* serverLine;
    QLineEdit* portLine;
};

enum class FileDialogMode { File, Directory };

class RemoteFileDialog : public QDialog
{
public:
    /**
     * @brief getFileUrl opens a connection to a cube server and shows a dialog to select a file
     * @param directory initial directory to show on the file dialog. If empty, the directory where the cube-server is started is choosen.
     * @param title dialog title
     * @param parent parent widget of the file dialog
     * @return url of the file that the user has choosen or empty string, if cancelled
     */
    static QString
    getFileUrl( const QString& directory = "",
                const QString& title = tr( "Open file on server" ),
                QWidget*       parent = nullptr );

    /**
     * @brief getFileUrl opens a connection to a cube server and shows a dialog to select a directory
     * @param directory initial directory to show on the file dialog. If empty, the directory where the cube-server is started is choosen.
     * @param title dialog title
     * @param parent parent widget of the file dialog
     * @return url of the file that the user has choosen or empty string, if cancelled
     */
    static QString
    getDirectoryUrl( const QString& directory = "",
                     const QString& title = tr( "Open directory on server" ),
                     QWidget*       parent = nullptr );

    /**
     * @brief getDirectoryContents recursively searches for all non-hidden files in the given path
     * @param url cube url to the directory to start the search
     * @return list of file names with absolute path
     */
    static QList<QString>
    getDirectoryContents( const QString& url );

private:
    RemoteFileDialog( const QString& server,
                      const QString& directory = ".",
                      const QString& title = "",
                      FileDialogMode mode = FileDialogMode::File,
                      QWidget*       parent = nullptr );
    void
    fileSelected( QString filename );
    void
    markHistory( const QString& pathStr );
    void
    reconnect();
    void
    activateSelection();
    void
    disconnect();
    void
    deleteModel();
    void
    modelPathChanged( const QString& path );

    UrlWidget*                urlWidget;
    PathWidget*               pathWidget;
    QLineEdit*                fileInput;
    QTreeView*                tree;
    RemoteFileSystemModel*    model;
    FileSortFilterProxyModel* proxy;
    QString                   filename;
    QString                   previousPath = "";
    FileDialogMode            mode;
    bool                      okPressed = false;
};
}
#endif // REMOTEFILEDIALOG_H
