/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include <QApplication>
#include <QDebug>
#include <QDialogButtonBox>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QToolButton>
#include <QUrl>
#include <QVBoxLayout>
#include <QThread>
#include <QHeaderView>
#include "RemoteFileDialog.h"
#include "RemoteFileSystemModel.h"
#include "Globals.h"
#include "Compatibility.h"
#include "CubeError.h"
#include "CubeClientConnection.h"
#include "CubeNetworkRequest.h"
#include "CubeSocket.h"

using namespace cubegui;
using namespace cube;

static QString                 defaultServer = "cube://localhost:3300";
static QHash<QString, QString> defaultDir;

QString
RemoteFileDialog::getFileUrl( const QString& directory, const QString& title, QWidget* parent )
{
    try
    {
        if ( parent == nullptr )
        {
            parent = Globals::getMainWindow();
        }
        QString          dir = directory.isEmpty() ? defaultDir[ defaultServer ] : directory;
        RemoteFileDialog dialog( defaultServer, dir, title, FileDialogMode::File, parent );
        int              status = dialog.exec();
        dialog.disconnect();
        if ( ( status == QDialog::Accepted ) && !dialog.filename.trimmed().isEmpty() )
        {
            defaultServer = dialog.urlWidget->getUrl();
            return defaultServer + "/" + dialog.filename;
        }
    }
    catch ( const cube::NetworkError& e )
    {
        QString msg = QString( tr( "Remote file dialog: " ) ) + e.what();
        Globals::setStatusMessage( msg, Error );
    }
    return "";
}

QString
RemoteFileDialog::getDirectoryUrl( const QString& directory, const QString& title, QWidget* parent )
{
    try
    {
        if ( parent == nullptr )
        {
            parent = Globals::getMainWindow();
        }
        QString          dir = directory.isEmpty() ? defaultDir[ defaultServer ] : directory;
        RemoteFileDialog dialog( defaultServer, dir, title, FileDialogMode::Directory, parent );
        int              status = dialog.exec();
        dialog.disconnect();
        if ( status == QDialog::Accepted )
        {
            return dialog.urlWidget->getUrl() + "/" + dialog.pathWidget->getPath();
        }
    }
    catch ( const cube::NetworkError& e )
    {
        QString msg = QString( tr( "Remote file dialog: " ) ) + e.what();
        Globals::setStatusMessage( msg, Error );
    }
    return "";
}

RemoteFileDialog::RemoteFileDialog( const QString& server, const QString& directory, const QString& title, FileDialogMode mode, QWidget* parent ) : QDialog( parent )
{
    this->mode = mode;
    model      = 0;
    okPressed  = false;
    proxy      = new FileSortFilterProxyModel;
    tree       = new QTreeView();
    tree->setSortingEnabled( true );
    tree->sortByColumn( 0, Qt::AscendingOrder );
    tree->setModel( proxy );
    tree->setFocus();

    connect( tree, &QTreeView::activated, proxy, &FileSortFilterProxyModel::itemActivated );
    connect( tree, &QTreeView::clicked, proxy, &FileSortFilterProxyModel::itemClicked );

    this->setWindowTitle( title );
    this->setModal( true );
    resize( 600, 400 );

    // server line (default: cube://localhost:3300)
    urlWidget = new UrlWidget( server );
    connect( urlWidget, &UrlWidget::urlChanged, this, &RemoteFileDialog::reconnect );

    pathWidget = new PathWidget( directory );

    fileInput = new QLineEdit();
    QWidget* fileLine = new QWidget;
    fileLine->setLayout( new QHBoxLayout );
    fileLine->layout()->setContentsMargins( 0, 0, 0, 0 );
    fileLine->layout()->addWidget( new QLabel( tr( "Name:" ) ) );
    fileLine->layout()->addWidget( fileInput );

    QDialogButtonBox* buttonBox = new QDialogButtonBox();
    buttonBox->addButton( QDialogButtonBox::Cancel );
    QPushButton* ok;
    if ( mode == FileDialogMode::File )
    {
        ok = buttonBox->addButton( QDialogButtonBox::Ok ); // enter key in text field => Ok Button
    }
    else
    {
        // enter key has no effect on directory dialog
        ok = buttonBox->addButton( "ok", QDialogButtonBox::YesRole );
        // enter key in text field -> evaluate directory and enter directory
        connect( fileInput, &QLineEdit::returnPressed, this, &RemoteFileDialog::activateSelection );
    }
    ok->setDefault( false );
    ok->setAutoDefault( false );

    connect( buttonBox,  &QDialogButtonBox::rejected, this, [ this ](){
        this->close();
        deleteModel();
    } );
    connect( buttonBox,  &QDialogButtonBox::accepted, this, &RemoteFileDialog::activateSelection );

    QVBoxLayout* layout = new QVBoxLayout();
    this->setContentsMargins( 0, 0, 0, 0 );
#ifndef __EMSCRIPTEN__
    layout->addWidget( urlWidget );
#endif
    layout->addWidget( pathWidget );
    layout->addWidget( tree );
    layout->addWidget( fileLine );
    layout->addWidget( buttonBox );

    reconnect(); // connect to cube server

    this->setLayout( layout );
    fileLine->setFocus();
}

void
RemoteFileDialog::disconnect()
{
    close();
    this->deleteLater();
    this->deleteModel();
    model = nullptr;
    delete proxy;
    proxy = nullptr;
}

/** called in mode FileDialogMode::File, if a file is selected in the model.  */
void
RemoteFileDialog::fileSelected( QString fn )
{
    if ( mode == FileDialogMode::File )
    {
        filename                          = fn;
        defaultDir[ urlWidget->getUrl() ] = pathWidget->getPath();
        accept(); // valid file -> accept and close dialog
    }
}

/** called if path in model has changed */
void
RemoteFileDialog::modelPathChanged( const QString& path )
{
    if ( okPressed )
    {
        if ( mode == FileDialogMode::Directory )
        {
            markHistory( path );
            accept();
        }
        else // check if valid file is given
        {
            if ( model->containsFile( filename ) )
            {
                markHistory( path );
                accept();
            }
            else
            {
                MessageType type = MessageType::Error;
                #ifdef __EMSCRIPTEN__
                // the webassembly version hangs, if a modal dialog is called from another modal dialog -> use warning instead of critical
                type = MessageType::Warning;
                #endif
                Globals::setStatusMessage( QString( tr( "Cannot read file" ) ) + " " + filename, type );
                filename = "";
            }
        }
    }
    pathWidget->setPath( path );
    markHistory( path );
    if ( mode == FileDialogMode::File )
    {
        fileInput->clear();
    }
    else
    {
        fileInput->setText( path );
    }
}

/** If the input in QLineEdit "fileInput" is a valid file or directory, the dialog is accepted.
 *  If the input contains a valid directory, the model changes to this directory.
 */
void
RemoteFileDialog::activateSelection()
{
    okPressed = true;
    QString inputPath = fileInput->text();
    if ( !inputPath.isEmpty() || ( mode == FileDialogMode::Directory ) )
    {
        QString fullPath = inputPath.startsWith( "/" ) ? inputPath : pathWidget->getPath() + "/" + inputPath;
        QString path     = fullPath;

        if ( mode == FileDialogMode::File )  // check if the path of the file is a valid directory
        {
            path = fullPath.left( fullPath.lastIndexOf( "/" ) );
        }
        filename = fullPath;         // set selected file with absolute path
        model->setDirectory( path ); // sets model to new directory (non-blocking)
    }
}

/** marks previous directory */
void
RemoteFileDialog::markHistory( const QString& pathStr )
{
    if ( pathStr != previousPath )
    {
        QModelIndex idx = proxy->mapFromSource( model->getModelIndex( previousPath ) );

        tree->selectionModel()->select( idx, QItemSelectionModel::Select | QItemSelectionModel::Rows );
        tree->scrollTo( idx );
        previousPath = pathStr;
    }
}

/**
 * @brief deleting the model also deletes the CubeClientConnection. This causes messages to be sent to the server which
 * has to be done outside the main thread. Therefore, the model is moved to a new thread and deleted there.
 */
void
RemoteFileDialog::deleteModel()
{
    if ( model != nullptr )
    {
        RemoteFileSystemModel* toDelete = model;
        QThread*               thread   = new QThread();

        toDelete->moveToThread( thread );

        QObject::connect( thread, &QThread::started, [ toDelete ]() {
            // delete the network connection in a new thread
            // no context parameter (3rd parameter) is given to execute deletion in the thread
            delete toDelete;
        } );
        QObject::connect( thread, &QThread::finished, thread, &QThread::deleteLater );
        thread->start();

        model = nullptr;
    }
}

void
RemoteFileDialog::reconnect()
{
    proxy->setSourceModel( NULL ); // show empty tree
    deleteModel();
    try
    {
        model = new RemoteFileSystemModel( urlWidget->getUrl() );
        connect( pathWidget, &PathWidget::pathChanged,
                 model, &RemoteFileSystemModel::setDirectory );

        if ( mode == FileDialogMode::File )
        {
            connect( model, &RemoteFileSystemModel::fileSelected, this, &RemoteFileDialog::fileSelected );

            // special case: on some OS, clicking just marks the file, selection requires double click
            connect( model, &RemoteFileSystemModel::fileClicked, this, [ this ]( const QString& path ){
                QString absolute = "/" + path;
                QString file     = absolute.mid( absolute.lastIndexOf( "/" ) + 1 );
                fileInput->setText( file );
            } );
        }
        connect( model, &RemoteFileSystemModel::directorySelected,
                 this, &RemoteFileDialog::modelPathChanged );

        proxy->setSourceModel( model );
        model->setDirectory( pathWidget->getPath() );

        // ensure proper layout:
        // after columns have been set in the model, let 1st column (file name) take all space
        // and the others resize to contents
        tree->header()->setSectionResizeMode( 0, QHeaderView::Stretch );
        tree->header()->setStretchLastSection( false );
        for ( int i = 1; i < tree->model()->columnCount(); ++i )
        {
            tree->header()->setSectionResizeMode( i, QHeaderView::ResizeToContents );
        }
    }
    catch ( const cube::NetworkError& e )
    {
        Globals::setStatusMessage( QString( tr( "Remote file dialog: " ) ) + e.what(), Error );
    }
}

//==========================================================================================

PathWidget::PathWidget( const QString& dir ) : path( dir )
{
}

/**
 * @brief PathWidget::shortenPath splits the path into a list of directories. For long pathes,
 * the directories in the middle section are united and will be hidden later
 * @param path complete path
 * @param currentPath path to the current directory
 */
QStringList
PathWidget::getPathList( const QString& path, const QString& currentPath )
{
    const int    maxwidth = 400;
    const int    minShow  = 2; // minimum count of directories to show at the beginning and the end
    QStringList  dirs;
    QFontMetrics fm( font() );
    dirs = path.split( "/", SkipEmptyParts );
    bool hideParts = fm.boundingRect( path ).width() > maxwidth;
    if ( hideParts ) // for shortened paths: only show path to current directory, don't show history
    {
        dirs      = currentPath.split( "/", SkipEmptyParts );
        hideParts = fm.boundingRect( currentPath ).width() > maxwidth;
    }
    int width = 0;
    if ( hideParts && dirs.size() > 2 * minShow  )
    {
        for ( int i = 0; i < minShow; i++ )
        {
            width += fm.boundingRect( dirs[ i ] ).width();
            width += fm.boundingRect( dirs[ dirs.length() - i - 1 ] ).width();
        }
        width += minShow * 2 * fm.boundingRect( " > " ).width();
        int tail = dirs.length() - minShow - 1; // first index to show after hidden elements
        while ( tail > minShow && width < maxwidth )
        {
            width += fm.boundingRect( dirs[ tail ] + " > " ).width();
            if ( width < maxwidth )
            {
                tail--;
            }
        }
        QStringList shortened;
        for ( int i = 0; i < minShow; i++ )
        {
            shortened.append( dirs[ i ] );
            width += fm.boundingRect( dirs[ i ] ).width();
        }
        QString middle; // this part will be hidden ("...")
        for ( int i = minShow; i < tail; i++ )
        {
            middle += "/" + dirs[ i ];
        }
        if ( !middle.isEmpty() )
        {
            shortened.append( middle );
        }
        for ( int i = tail; i < dirs.length(); i++ )
        {
            shortened.append( dirs[ i ] );
        }
        dirs = shortened;
    }
    return dirs;
}

void
PathWidget::setPath( const QString& pathStr )
{
    path = pathStr;
    if ( !parentSelected )
    {
        prevPath = pathStr;
    }
    parentSelected = false;

    qDeleteAll( this->findChildren<QWidget*>( QString(), Qt::FindDirectChildrenOnly ) );
    delete ( this->layout() );

    QBoxLayout* layout = new QHBoxLayout();
    setLayout( layout );
    layout->setContentsMargins( 0, 0, 0, 0 );
    layout->setSpacing( 0 );

    QFontMetrics fm( font() );
    QString      currentPath;
    QStringList  dirs = getPathList( prevPath, pathStr );

    for ( const QString& dir : dirs )
    {
        currentPath += "/" + dir;

        if ( dir.startsWith( "/" ) )
        {
            layout->addWidget( new QLabel( " > ... " ), 0, Qt::AlignLeft );
        }
        else
        {
            QPushButton* but = new QPushButton( dir );
            but->setFlat( true );
            if ( currentPath == pathStr )
            {
                but->setStyleSheet( "QPushButton{ font-weight: bold; }" );
            }
            QColor  color = palette().color( QPalette::Normal, QPalette::Highlight );
            QString col   = color.name( QColor::HexRgb );
            // ensure that flat button shows highlighted border if entered
            but->setStyleSheet( but->styleSheet() +
                                "QPushButton:hover { border: 1px solid; border-radius: 2px; border-color:" + col + "} " );

            QSize size = fm.boundingRect( but->text() ).size();
            size += QSize( 10, 10 ); // add 5 pixel margins on each side
            but->setFixedSize( size );
            connect( but, &QPushButton::clicked, this, [ this, currentPath ](){
                this->path = currentPath;
                parentSelected = true;
                emit pathChanged( path );
            } );
            QLabel* sep = new QLabel( ">" );
            sep->setMargin( 0 );
            layout->addWidget( sep, 0, Qt::AlignLeft );
            layout->addWidget( but, 0, Qt::AlignLeft );
        }
    }
    layout->addStretch( 1 );

    // forwards/backwards-buttons
    QToolButton* backBut = new QToolButton();
    backBut->setToolTip( tr( "Go back in history" ) );
    backBut->setIcon( QApplication::style()->standardIcon( QStyle::QStyle::SP_ArrowBack ) );
    backBut->setEnabled( !backwards.isEmpty() );
    layout->addWidget( backBut, 0, Qt::AlignRight );
    if ( !backwards.isEmpty() )
    {
        connect( backBut, &QToolButton::clicked, this, [ this ](){
            buttonsUsed = true;
            forwards.push( backwards.pop() ); // current dir
            this->path = backwards.pop();     // previous dir
            emit pathChanged( path );
        } );
    }

    backwards.push( pathStr );
    if ( !buttonsUsed )
    {
        forwards.clear();
    }
    buttonsUsed = false;

    QToolButton* forBut = new QToolButton();
    forBut->setToolTip( tr( "Go forward in history" ) );
    forBut->setIcon( QApplication::style()->standardIcon( QStyle::QStyle::SP_ArrowForward ) );
    forBut->setEnabled( !forwards.isEmpty() );
    layout->addWidget( forBut, 0, Qt::AlignRight );
    if ( !forwards.isEmpty() )
    {
        connect( forBut, &QToolButton::clicked, this, [ this ](){
            buttonsUsed = true;
            this->path = forwards.pop();
            emit pathChanged( path );
        } );
    }
}

QString
PathWidget::getPath() const
{
    return path;
}

//==========================================================================================

UrlWidget::UrlWidget( const QString& urlStr )
{
    serverLine = new QLineEdit();
    connect( serverLine, &QLineEdit::returnPressed, this, [ this ](){
        emit urlChanged( getUrl() );
    } );
    portLine = new QLineEdit();
    connect( portLine, &QLineEdit::returnPressed, this, [ this ](){
        emit urlChanged( getUrl() );
    } );
    serverLine->setFocusPolicy( Qt::ClickFocus );
    portLine->setFocusPolicy( Qt::ClickFocus );
    setUrl( urlStr );

    QWidget*     serverWidget = new QWidget();
    QHBoxLayout* hlay         = new QHBoxLayout;
    hlay->setContentsMargins( 0, 0, 0, 0 );
    serverWidget->setLayout( hlay );
    hlay->addWidget( new QLabel( tr( "Server" ) + ":" ) );
    hlay->addWidget( serverLine, 1 );
    hlay->addWidget( new QLabel( tr( "Port" ) + ":" ) );
    hlay->addWidget( portLine );
    QPushButton* but = new QPushButton( QApplication::style()->standardIcon( QStyle::SP_BrowserReload ), "" );
    but->setToolTip( tr( "Reconnect to server" ) );
    connect( but, &QPushButton::pressed, this, [ this ](){
        emit urlChanged( getUrl() );
    } );
    hlay->addWidget( but );

    setLayout( hlay );
}


void
UrlWidget::setUrl( const QString& urlStr )
{
    QUrl url( urlStr );
    serverLine->setText( url.host() );
    portLine->setText( QString::number( url.port() ) );
}

QString
UrlWidget::getUrl() const
{
    return "cube://" + serverLine->text().trimmed() + ":" + portLine->text().trimmed();
}

// static function which is used by context free plugins
QList<QString> RemoteFileDialog::getDirectoryContents( const QString& url )
{
    QList<QString> fileNames;

    try
    {
        QUrl    qurl = QUrl( url );
        QString host = "cube://" + qurl.host() + ":" + QString::number( qurl.port() );
        if ( qurl.path().isEmpty() )
        {
            return fileNames;
        }
        cube::ClientConnection::Ptr connection( cube::ClientConnection::create( Socket::create(), host.toStdString() ) );

        std::list<std::string> directories;
        directories.push_back( qurl.path().toStdString() );
        while ( !directories.empty() )
        {
            std::string path = directories.back();
            directories.pop_back();

            NetworkRequest::Ptr request = FileSystemRequest::create( path );
            FileSystemRequest*  fr      = dynamic_cast<FileSystemRequest*>( request.get() );
            request->sendRequest( *connection, NULL );
            request->receiveResponse( *connection, NULL );
            std::vector<FileInfo> allFiles = fr->getFiles(); // all files inclusive hidden

            allFiles.erase( allFiles.begin() );              // ignore first entry (=directory name)

            for ( const FileInfo& file : allFiles )
            {
                QString filename = QFileInfo( file.name().c_str() ).fileName();

                if ( !filename.isEmpty() && ( filename.at( 0 ) != '.' ) ) // no hidden files, no parent dirs
                {
                    if ( file.isDirectory() )
                    {
                        directories.push_back( file.name() );
                    }
                    else
                    {
                        fileNames.append( file.name().c_str() );
                    }
                }
            }
        }
    }
    catch ( const cube::NetworkError& e )
    {
        Globals::setStatusMessage( e.what(), Error );
    }
    return fileNames;
}
