/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 2020-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include "CubeApplication.h"
#include "MainWidget.h"
#include "PresentationCursor.h"

using namespace cubegui;

CubeApplication::CubeApplication( int& argc, char** argv )
    : QApplication( argc, argv )
{
    pcursor = nullptr;
}

void
CubeApplication::setPresentationMode( bool enabled )
{
    presentationIsEnabled = enabled;
    if ( enabled  && !pcursor )
    {
        pcursor = new PresentationCursor();
        pcursor->move( QCursor::pos() );
    }
    pcursor->setVisible( false );

    setEventFilter();
}

void
CubeApplication::emulateRightMouseMode( bool enabled )
{
    emulateMouseIsEnabled = enabled;
    setEventFilter();
}

void
CubeApplication::setEventFilter()
{
    if ( emulateMouseIsEnabled || presentationIsEnabled )
    {
        this->installEventFilter( this );
    }
    else
    {
        this->removeEventFilter( this );
    }
}

void
CubeApplication::setMain( MainWidget* m )
{
    mainWidget = m;
}

bool
CubeApplication::event( QEvent* event )
{
    if ( mainWidget && event->type() == QEvent::FileOpen )
    {
        QFileOpenEvent* openEvent = static_cast<QFileOpenEvent*>( event );
        mainWidget->loadFile( openEvent->file() );
    }
    return QApplication::event( event );
}

void
CubeApplication::emulateRightMouse( QObject* obj, QEvent* event )
{
    if ( event->type() == QEvent::MouseButtonPress )
    {
        QMouseEvent* ev = static_cast<QMouseEvent*>( event );
        if ( ev->button() == Qt::LeftButton )
        {
            if ( !mousePressTimer.isActive() )
            {
                // clicking outside the cube app generates a mouse press event for the previous object -> don't generate emulated left click event
                bool mouseIsInsideCube = ( ev->localPos().x() > 0 ) && ( ev->localPos().y() > 0 );
                if ( !emulationIsActive && mouseIsInsideCube )
                {
                    QMouseEvent e = QMouseEvent( ev->type(), ev->localPos(), Qt::RightButton, Qt::RightButton, ev->modifiers() );
                    mousePressTimer.start( 1000 );
                    mousePressTimer.setSingleShot( true );
                    connect( &mousePressTimer, &QTimer::timeout, this, [ this, obj, &e ]() {
                        emulationIsActive = true;

                        // emulate left mouse button release (on position 0,0 to disable action)
                        QMouseEvent emulated1( QEvent::MouseButtonRelease, QPointF( 0, 0 ), Qt::LeftButton, Qt::LeftButton, e.modifiers() );
                        QApplication::sendEvent( obj, &emulated1 );

                        // emulate right mouse button click
                        QMouseEvent emulated2( e.type(), e.localPos(), Qt::RightButton, Qt::RightButton, e.modifiers() );
                        QApplication::sendEvent( obj, &emulated2 );
                    } );
                }
            }
        }
    }
    else if ( event->type() == QEvent::MouseButtonRelease )
    {
        emulationIsActive = false;
        mousePressTimer.disconnect();
        mousePressTimer.stop();
    }
}

void
CubeApplication::setPresentationCursor( QObject* obj, QEvent* event )
{
    if ( !obj->isWidgetType() )
    {
        return;
    }
    QWidget* widget = qobject_cast<QWidget*> ( obj );

    if ( event->type() == QEvent::MouseMove )
    {
        QMouseEvent* ev = static_cast<QMouseEvent*>( event );
        QPoint       p  =  ev->globalPos();
        p += QPoint( -4, 4 ); // PresentationCursor widget mustn't be placed over cursor, WA_TransparentForMouseEvents doesn't seem to work

        pcursor->move( p );
        pcursor->raise();
    }
    else if ( event->type() == QEvent::MouseButtonPress )
    {
        QMouseEvent* ev = static_cast<QMouseEvent*>( event );
        if ( ev->button() == Qt::LeftButton )
        {
            pcursor->setCursor( CursorType::LEFT );
        }
        else if ( ev->button() == Qt::RightButton )
        {
            pcursor->setCursor( CursorType::RIGHT );
        }
        // MouseButtonRelease is not emitted if context menu is active and mouse is clicked enywhere else. For that reason
        // set a delay after which the default mouse pointer is shown again.
        pcursor->setCursor( CursorType::DEFAULT, 1000 );
    }
    // else if ( event->type() == QEvent::MouseButtonRelease )
    else if ( event->type() == QEvent::Wheel )
    {
        pcursor->setCursor( CursorType::WHEEL );
        pcursor->setCursor( CursorType::DEFAULT, 1000 );
    }
    else if ( widget && widget->isWindow() && event->type() == QEvent::Enter )
    {
        if ( !pcursor->isVisible() )
        {
            pcursor->setVisible( true );
        }
        pcursor->raise();
    }
    else if ( widget && widget->isWindow() && event->type() == QEvent::Leave )
    {
        QWidget* nextWindow = qApp->widgetAt( QCursor::pos() );
        if ( !nextWindow ) // if cursor is not above a window from this app then hide the presentation cursor
        {
            pcursor->setVisible( false );
        }
    }
    else if ( event->type() == QEvent::Show ) // e.g. a popup menu has been shown -> raise cursor
    {
        pcursor->raise();
    }
}

bool
CubeApplication::eventFilter( QObject* obj, QEvent* event )
{
    static QTimer mousePressTimer;

    if ( emulateMouseIsEnabled )
    {
        emulateRightMouse( obj, event );
    }
    if ( presentationIsEnabled )
    {
        setPresentationCursor( obj, event );
    }

    return false; // do not filter
}
