/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2022                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef CUBEGUI_CHART_H
#define CUBEGUI_CHART_H

#include <QWidget>
#include <QPainter>
#include <QString>
#include <QDebug>
#include <QStack>

#include <cmath>
#include <utility>
#include <vector>
#include <string>
#include "StatisticalInformation.h"

namespace cubegui
{
struct AxisLabel
{
    double  value;
    QString label;
};

/**
 * @brief Creates a widget with a coordinate system
 */
class Chart : public QWidget
{
public:
    const double ZERO_EPS = 1e-8;         // values lower than ZERO_EPS are treated as zero

    Chart( QWidget* parent = nullptr );

    /**
     * @brief setYRange
     * @param min lowest value on the scale
     * @param max highest value on the scale
     */
    void
    setYRange( double min,
               double max );

    /** value -> y screen coord */
    int
    getY( double value ) const;

    /** y screen coord -> value */
    double
    getValue( int y ) const;

    /** returns true, if the axis values should be treated as integer */
    bool
    isIntegerType() const;
    void
    setIntegerType( bool isIntegerType );

    /* Enables the selection of an y area with the left mouse button.
       @see getAreaDescription
     */
    void
    enableAreaSelection( bool enable );

    static QDialog*
    showStatisticToolTip( QWidget*       parent,
                          const QPoint&  pos,
                          const QString& html );

    static void
    showStatisticWindow( QWidget*       parent,
                         const QString& title,
                         const QString& html,
                         QDialog*       dialog = nullptr );

protected:
    int    startX;            // x position of the left lower corner of the chart
    int    startY;            // y position of the left lower corner of the chart
    int    chartHeight;       // chart height in pixel, borders excluded
    int    chartWidth;        // chart width in pixel, borders excluded
    int    tickWidth;         // width of the axis ticks
    int    lowerBorderHeight; // space below the chart
    int    upperBorderHeight; // space above the chart
    double yMin;              // minimum y value
    double yMax;              // maximum y value
    double tickSpacing;       // gap between the ticks of the left axis
    QRect  selectedArea;      // area which is selected with the left mouse button

    /** draws subclass specific items */
    virtual void
    drawChart( QPainter& painter ) = 0;

    /** calculates axis values and labels for the left axis.
     *  Sets a scale according to the given value range */
    virtual std::vector<AxisLabel>
    generateLeftAxisValues();

    /** calculates axis values and labels for the right axis. default: no right axis */
    virtual std::vector<AxisLabel>
    generateRightAxisValues();

    /* returns information about the selected area which will be displayed
     */
    virtual QString
    getAreaDescription() const
    {
        return "";
    }

    std::vector<AxisLabel>
    generateStatisticAxis( const StatisticalInformation& stat,
                           const StatisticalInformation& absolute );

    int
    getLowerBorderHeight()
    const;

    int
    getUpperBorderHeight()
    const;

    /** Get the space below the chart.
     *  This function should be overwritten if a legend is painted below the chart */
    virtual int
    calculateLowerBorderHeight( int chartWidth );

    virtual int
    calculateUpperBorderHeight( int chartWidth );

    QString
    numberToQString( double value,
                     double referenceValue,
                     double spacing,
                     int    minDecimalPlaces = 1,
                     int    maxDecimalPlaces = 6 );

    // ====== Qt event handling
    void
    resizeEvent( QResizeEvent* ) override
    {
        calculateGeometry();
    }
    void
    mousePressEvent( QMouseEvent* event ) override;
    void
    mouseMoveEvent( QMouseEvent* event ) override;
    void
    mouseReleaseEvent( QMouseEvent* event ) override;
    virtual void
    paintEvent( QPaintEvent* ) override;

private:
    std::vector<AxisLabel> leftAxisValues;       // values of the left y axis tick marks
    std::vector<AxisLabel> rightAxisValues;      // values and labels of the right y axis ticks marks
    bool                   isInteger;            // true, if the values should be treated as int
    bool                   areaSelectionEnabled; // true, if the user can select an area with left mouse

    // area selection
    double factor;                        // x value -> coordinate
    QPoint mousePressPos;                 // mouse position of the latest mouse press event

    /** calculates width and hight of all graphical components after widget size has been changed */
    void
    calculateGeometry();

    /** draws the x axis using the range of setYRange */
    void
    drawLeftAxis( QPainter& painter );

    /** draws the x axis */
    void
    drawRightAxis( QPainter& painter );
};

// =========================================================================================

/** algorithm of the pseudo-code example from "Graphics Gems" by Andrew S. Glassner */
class NiceScale
{
public:
    /**
     * @param min the minimum value on the axis
     * @param max the maximum value on the axis
     */
    NiceScale( double min,
               double max );

    double
    getMinimum()
    {
        return niceMin;
    }
    double
    getMaximum()
    {
        return niceMax;
    }
    int
    getTickCount()
    {
        return ( int )( lround( niceMax - niceMin ) / tickSpacing );
    }
    double
    getTickSpacing()
    {
        return tickSpacing;
    }

private:
    /**
     * Returns a "nice" number approximately equal to range Rounds
     * the number if round = true Takes the ceiling if round = false.
     * @param range the data range
     * @param round whether to round the result
     * @return a "nice" number to be used for the data range
     */
    double
    niceNum( double range,
             bool   round );

    /**
     * Calculate and update values for tick spacing and nice
     * minimum and maximum data points on the axis.
     */
    void
    calculate();

    /**
     * Sets maximum number of tick marks we're comfortable with
     * @param maxTicks the maximum number of tick marks for the axis
     */
    void
    setMaxTicks( double maxTicks );

private:
    double minValue;
    double maxValue;
    double maxTicks;

    double tickSpacing;
    double range;
    double niceMin;
    double niceMax;
};
}
#endif
