/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2022                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include <fstream>
#include <iostream>
#include <limits>
#include <math.h>
#include <sstream>

#include <QApplication>
#include <QClipboard>
#include <QDebug>
#include <QDialog>
#include <QFontMetrics>
#include <QLabel>
#include <QMouseEvent>
#include <QPen>
#include <QPushButton>
#include <QScrollArea>
#include <QString>
#include <QTextDocumentFragment>
#include <QVBoxLayout>

#include "BoxPlot.h"
#include "Globals.h"
#include "StatisticalInformation.h"

using namespace std;
using namespace cubegui;

Chart::Chart( QWidget* parent ) : QWidget( parent )
{
    yMin                 = 0;
    yMax                 = 0;
    tickSpacing          = 0;
    isInteger            = false;
    areaSelectionEnabled = false;
}

void
Chart::enableAreaSelection( bool enable )
{
    areaSelectionEnabled = enable;
    setMouseTracking( enable );
}

void
Chart::setYRange( double min, double max )
{
    if ( min > max )
    {
        std::swap( min, max );
    }
    yMin = min;
    yMax = max;

    leftAxisValues  = generateLeftAxisValues();
    rightAxisValues = generateRightAxisValues();
    calculateGeometry();

    repaint();
}

int
Chart::getUpperBorderHeight() const
{
    return upperBorderHeight;
}

int
Chart::getLowerBorderHeight() const
{
    return lowerBorderHeight;
}

int
Chart::calculateLowerBorderHeight( int )
{
    return 20;
}

int
Chart::calculateUpperBorderHeight( int )
{
    return 20;
}

void
Chart::paintEvent( QPaintEvent* )
{
    QPainter painter( this );

    drawLeftAxis( painter );
    drawRightAxis( painter );
    drawChart( painter );

    // border rectangle
    painter.drawRect( startX, startY - chartHeight, chartWidth, chartHeight );

    if ( areaSelectionEnabled && selectedArea.isValid() )
    {
        QColor gray = Qt::gray;
        gray.setAlpha( 50 );
        painter.fillRect( selectedArea, QBrush( gray ) );

        // paint area information
        QFontMetrics fm( font() );
        QString      txt  = getAreaDescription();
        QRect        rect = fm.boundingRect( txt );
        rect.moveTo( selectedArea.x() + 10, selectedArea.y() + 10 );
        rect.setWidth( rect.width() + 10 );
        rect.setHeight( rect.height() + 10 );

        painter.fillRect( rect, Qt::white );
        painter.drawRect( rect );
        painter.drawText( rect, Qt::AlignCenter, txt );
    }
}



int
Chart::getY( double value ) const
{
    value = std::min( value, yMax ); //  ensure that values are inside the range ( required for zooming )
    value = std::max( value, yMin );
    int distance = static_cast<int>( ( value - yMin ) * factor );
    return startY - distance;
}

double
Chart::getValue( int y ) const
{
    double distance = startY - y;
    return distance / factor + yMin;
}

bool
Chart::isIntegerType() const
{
    return isInteger;
}

void
Chart::setIntegerType( bool isIntegerType )
{
    isInteger = isIntegerType;
}

void
Chart::calculateGeometry()
{
    // define values
    tickWidth = 5;               // width of the ticks on the scale
    int padding = 3 * tickWidth; // padding between the ticks and the text

    // text geometry
    QString textLeft  = numberToQString( yMax, yMax, tickSpacing );
    QString textRight = "";      // largest value on the right y axis
    if ( rightAxisValues.size() > 0 )
    {
        auto max = std::max_element( rightAxisValues.begin(), rightAxisValues.end(),
                                     [ ]( const AxisLabel& a, const AxisLabel& b ) {
            return a.value < b.value;
        } );
        textRight = max->label;
    }
    QFontMetrics fm( font() );
    int          leftBorderWidth  = fm.boundingRect( textLeft ).width() + padding;
    int          rightBorderWidth = fm.boundingRect( textRight ).width() + padding;

    // calculate values
    startX            = leftBorderWidth;                          // lower left corner
    chartWidth        = width() - leftBorderWidth - rightBorderWidth;
    upperBorderHeight = calculateUpperBorderHeight( chartWidth ); // space on the top of the chart
    lowerBorderHeight = calculateLowerBorderHeight( chartWidth );
    startY            = height() - lowerBorderHeight;             // lower left corner
    chartHeight       = height() - lowerBorderHeight - upperBorderHeight;
    factor            = chartHeight / ( yMax - yMin );
}


std::vector<AxisLabel>
Chart::generateLeftAxisValues()
{
    std::vector<AxisLabel> ticks;
    double                 min = yMin;
    double                 max = yMax;
    int                    tickCount;

    if ( fabs( max - min ) < ZERO_EPS )
    {
        tickCount   = 2;
        max         = min + 1; // create valid box, if min == max
        min         = max - 2.0;
        tickSpacing = 0.5;
    }
    else
    {
        NiceScale scale( min, max );
        min         = scale.getMinimum();
        max         = scale.getMaximum();
        tickCount   = scale.getTickCount();
        tickSpacing = scale.getTickSpacing();
    }

    for ( int i = 0; i <= tickCount; ++i )
    {
        double value = min + i * ( max - min ) / tickCount;
        // set empty label if integer data type and value has fractional part
        bool    showLabel = !isIntegerType() || ( value == floor( value ) );
        QString label     = showLabel ? numberToQString( value, max, tickSpacing ) : "";
        ticks.push_back( AxisLabel { value, label } );
    }
    yMin = min;
    yMax = max;

    return ticks;
}

/**
 * @brief generates axis ticks for statistical data
 */
std::vector<AxisLabel>
Chart::generateStatisticAxis( const StatisticalInformation& stat, const StatisticalInformation& absolute )
{
    std::vector<AxisLabel> ticks;

    vector<double> yAxisValues;
    yAxisValues.push_back( stat.getMaximum() );
    yAxisValues.push_back( stat.getMinimum() );
    yAxisValues.push_back( stat.getMedian() );
    yAxisValues.push_back( stat.getMean() );
    yAxisValues.push_back( stat.getQ1() );
    yAxisValues.push_back( stat.getQ3() );

    std::vector<double> absValues;
    absValues.push_back( absolute.getMaximum() );
    absValues.push_back( absolute.getMinimum() );
    absValues.push_back( absolute.getMedian() );
    absValues.push_back( absolute.getMean() );
    absValues.push_back( absolute.getQ1() );
    absValues.push_back( absolute.getQ3() );

    /** Calculate lowest spacing between two axis values. This value is uses to ensure, that the precision is sufficient to
     *  distinguish the values.
     *  The spacing of nearby values, which would overlap in the chart, will be ignored */
    std::vector<double> values = absValues;
    std::sort( values.begin(), values.end() );
    int    fontHeight = QFontMetrics( Globals::getMainWindow()->font() ).height();
    double range      = absolute.getMaximum() - absolute.getMinimum();
    double minSpacing = range * fontHeight / chartHeight; // spacing required to draw one line

    // Use at least one more decimal place than the left axis. Otherwise left and right axes
    // could e.g. show 0.2, but their ticks differ because right axis value is 0.15 and rounded
    double spacing = tickSpacing / 10;
    for ( unsigned i = 1; i < values.size(); i++ )
    {
        double diff = values[ i ] - values[ i - 1 ];
        if ( diff > minSpacing )
        {
            spacing = std::min( diff, spacing );
        }
    }

    int    minDigits = isIntegerType() ? 0 : Globals::getPrecision( FORMAT_TREES );
    double reference = fabs( absolute.getMaximum() ) < ZERO_EPS ? 1 : absolute.getMaximum();
    for ( unsigned i = 0; i < yAxisValues.size(); i++ )
    {
        QString label;
        // set empty label for values < 1 of integer metrics
        bool showLabel = !isIntegerType() ||  absValues[ i ] >= 1 ||
                         ( absValues[ i ] == floor( absValues[ i ] ) );
        if ( showLabel )
        {
            label = Chart::numberToQString( absValues[ i ], reference, spacing, minDigits );
        }

        ticks.push_back( AxisLabel { yAxisValues[ i ], label } );
    }

    return ticks;
}



/** default: no right axis
 *  override this method to draw the given ticks
 */
std::vector<AxisLabel> Chart::generateRightAxisValues()
{
    return std::vector<AxisLabel>();
}

/**
 * @brief Chart::numberToQString
 * @param value the value to be converted to String
 * @param referenceValue value that determines if scientific or decimal format and the number of decimal places
 * @param spacing the String representation of value should differ from that of value+spacing
 * @return
 */
QString
Chart::numberToQString( double value, double referenceValue, double spacing, int minDecimalPlaces, int maxDecimalPlaces )
{
    int expReference; // exponent of the reference value
    int expSpacing;   // exponent of the spacing value
    int decimal;      // decimal places of all values

    expReference = ( fabs( referenceValue ) < ZERO_EPS ) ? 0 : floor( log10( referenceValue ) );
    expSpacing   = ( fabs( spacing ) < ZERO_EPS ) ? 0 : floor( log10( spacing ) );

    QString text;
    bool    scientific = ( referenceValue >= 1e4 ) || ( referenceValue <= 1.e-3 );
    if ( scientific )
    {
        // decimal = number of decimal places required to show both: the reference value and the value of spacing
        decimal = expReference - expSpacing;
        decimal = std::max( decimal, minDecimalPlaces ); // use at least minimum decimal places
        decimal = std::min( decimal, maxDecimalPlaces ); // don't draw more than max decimal places

        value /= pow( 10.0, expReference );
        text   = QString::number( value, 'f', decimal );
        text  += "e" + QString::number( expReference );
    }
    else
    {
        if ( isInteger )
        {
            decimal = 0;
        }
        else
        {
            decimal = std::max( abs( expSpacing ), minDecimalPlaces );
            decimal = std::min( decimal, maxDecimalPlaces );
        }
        text = QString::number( value, 'f', decimal );
    }

    return text;
}

void
Chart::drawLeftAxis( QPainter& painter )
{
    QFontMetrics fm( font() );
    int          textHeight = fm.height();

    painter.drawLine( startX, startY, startX, startY - chartHeight );

    QPen standardPen = painter.pen();
    QPen dashedPen( Qt::DashLine );
    dashedPen.setColor( Qt::cyan );

    for ( AxisLabel tick : leftAxisValues )
    {
        painter.setPen( standardPen );
        int currYPos = getY( tick.value );
        // tick
        painter.drawLine( startX, currYPos, startX - tickWidth, currYPos );
        // label
        painter.drawText( 0, currYPos - ( textHeight + 1 ) / 2,
                          startX - 2 * tickWidth, textHeight, Qt::AlignRight,
                          tick.label );
        // dashed cyan line
        painter.setPen( dashedPen );
        painter.drawLine( startX + 1, currYPos, startX + chartWidth, currYPos );
    }
    painter.setPen( standardPen );
}

static bool
isOverlapping( double value, double width, std::vector<double>& values )
{
    bool overlap = false;
    for ( double y : values )
    {
        if ( fabs( value - y ) <= width )
        {
            overlap = true;
            break;
        }
    }
    return overlap;
}

void
Chart::drawRightAxis( QPainter& painter )
{
    if ( rightAxisValues.size() == 0 )
    {
        return;
    }

    painter.drawLine( startX + chartWidth, startY, startX + chartWidth, startY - chartHeight );
    QFontMetrics fm( font() );
    int          textHeight = fm.height();

    std::vector<double> oldValues;
    for ( auto tick : rightAxisValues )
    {
        int currPosY = getY( tick.value );

        if ( isOverlapping( currPosY, textHeight, oldValues ) )
        {
            continue; // label overlaps a previous one -> don't paint
        }
        oldValues.push_back( currPosY );

        // tick
        painter.drawLine( startX + chartWidth, currPosY, startX + chartWidth + tickWidth, currPosY );
        // label
        painter.drawText( startX + chartWidth + 2 * tickWidth, currPosY - ( textHeight + 1 ) / 2,
                          width(), textHeight, Qt::AlignLeft, tick.label );
    }
}

void
Chart::mousePressEvent( QMouseEvent* event )
{
    mousePressPos = event->pos();
}

void
Chart::mouseMoveEvent( QMouseEvent* event )
{
    if ( event->buttons() & Qt::LeftButton )
    {
        double y1 = std::min( event->y(), mousePressPos.y() );
        double y2 = std::max( event->y(), mousePressPos.y() );
        selectedArea = QRect( startX + 1, y1, chartWidth - 1, y2 - y1 );
        repaint();
    }
}

void
Chart::mouseReleaseEvent( QMouseEvent* )
{
    selectedArea.setHeight( -1 ); // invalidate
    repaint();
}


// ====================================================================================================


NiceScale::NiceScale( double min, double max )
{
    minValue = min;
    maxValue = max;
    maxTicks = 8; // default tick count, border excluded
    calculate();
}

void
NiceScale::calculate()
{
    range       = niceNum( maxValue - minValue, false );
    tickSpacing = niceNum( range / ( maxTicks - 1 ), true );
    niceMin     = floor( minValue / tickSpacing ) * tickSpacing;
    niceMax     = ceil( maxValue / tickSpacing ) * tickSpacing;
}

double
NiceScale::niceNum( double range, bool round )
{
    double exponent;     /** exponent of range */
    double fraction;     /** fractional part of range */
    double niceFraction; /** nice, rounded fraction */

    exponent = floor( log10( range ) );
    fraction = range / pow( 10.f, exponent );

    if ( round )
    {
        if ( fraction < 1.5 )
        {
            niceFraction = 1;
        }
        else if ( fraction < 3 )
        {
            niceFraction = 2;
        }
        else if ( fraction < 7 )
        {
            niceFraction = 5;
        }
        else
        {
            niceFraction = 10;
        }
    }
    else
    {
        if ( fraction <= 1 )
        {
            niceFraction = 1;
        }
        else if ( fraction <= 2 )
        {
            niceFraction = 2;
        }
        else if ( fraction <= 5 )
        {
            niceFraction = 5;
        }
        else
        {
            niceFraction = 10;
        }
    }

    return niceFraction * pow( 10, exponent );
}

void
NiceScale::setMaxTicks( double maxTicks )
{
    this->maxTicks = maxTicks;
    calculate();
}

// ====================================================================================================

/**
 * shows information in a tooltip
 * @param parent parent window
 * @param pos tooltip position on screen
 * @param html text to display
 */
QDialog*
Chart::showStatisticToolTip( QWidget* parent, const QPoint& pos, const QString& html )
{
    QDialog* dialog = new QDialog( parent );
    dialog->setWindowFlags( Qt::ToolTip ); // create separate window
    dialog->setStyleSheet( "QDialog { border: 1px solid black }" );

    QLabel*      text   = new QLabel( html );
    QVBoxLayout* layout = new QVBoxLayout();
    layout->setContentsMargins( 5, 5, 5, 5 );
    layout->addWidget( text );
    dialog->setLayout( layout );

    dialog->move( pos );
    dialog->show();
    return dialog;
}

/**
 * shows information in a dialog, which is automatically deleted after closing
 * @brief BoxPlot::showInWindow
 * @param parent parent window
 * @param title title of the window
 * @param html text to display
 * @param infoDialog if not null, the given dialog is used to show the information. In this case the
 * dialog isn't deleted after closing.
 */
void
Chart::showStatisticWindow( QWidget* parent, const QString& title, const QString& html, QDialog* infoDialog )
{
    QDialog* dialog = infoDialog;
    if ( !dialog ) // create new dialog and delete after closing
    {
        dialog = new QDialog( parent );
        dialog->setAttribute( Qt::WA_DeleteOnClose );
    }
    else // use given dialog
    {
        for ( QWidget* widget : dialog->findChildren<QWidget*>() )
        {
            widget->deleteLater();
        }
        delete dialog->layout();
    }
    dialog->setWindowTitle( title );

    QVBoxLayout* layout = new QVBoxLayout();
    QLabel*      text   = new QLabel( html );
    text->setTextInteractionFlags( Qt::TextSelectableByMouse );
    QScrollArea* scroll = new QScrollArea;
    scroll->setWidgetResizable( true );
    text->setContentsMargins( 5, 5, 5, 5 );
    scroll->setWidget( text );
    layout->addWidget( scroll );

    QWidget*     buttons = new QWidget( dialog );
    QHBoxLayout* bLayout = new QHBoxLayout();
    buttons->setLayout( bLayout );

    bLayout->addItem( new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Minimum ) );
    QPushButton* clip = new QPushButton( tr( "To Clipboard" ), dialog );
    connect( clip, &QPushButton::pressed, [ html ](){
        QApplication::clipboard()->setText( QTextDocumentFragment::fromHtml( html ).toPlainText() );
    } );
    bLayout->addWidget( clip );
    QPushButton* okButton = new QPushButton( tr( "Close" ), dialog );
    connect( okButton, SIGNAL( pressed() ), dialog, SLOT( accept() ) );
    bLayout->addWidget( okButton );
    layout->addWidget( buttons );

    dialog->setLayout( layout );
    dialog->setModal( false );
    dialog->show();
}
