/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"

#include "TreeStatistics.h"
#include "Tree.h"
#include "Globals.h"
#include "TabManager.h"

using namespace cubegui;

TreeStatistics::TreeStatistics( Tree* tree )
{
    this->tree = tree;

    valid              = true;
    absoluteMode       = true;
    userDefinedMinMax_ = false;
    isInteger_         = false;
    isRelativeMaximum_ = false;

    update();
}

void
TreeStatistics::update()
{
    minAbsValue_   = std::nan( "" );
    maxAbsValue_   = std::nan( "" );
    minValue_      = std::nan( "" );
    maxValue_      = std::nan( "" );
    absValue_      = std::nan( "" );
    mean_          = std::nan( "" );
    varianceSqrt_  = std::nan( "" );
    selectedValue_ = std::nan( "" );
    absValue_      = std::nan( "" );

    absoluteMode = tree->getValueModus() == ABSOLUTE_VALUES || tree->getValueModus() == ABSOLUTE_PEER_COLORS;

    const QList<TreeItem*>& selectionList = tree->getSelectionList();

    if ( tree->isManualCalculation() || selectionList.isEmpty() )
    {
        return; // not yet ready
    }

    // get aggregated selected value
    if ( tree->isValidSelection() && tree->getSelectionSummary() && tree->getSelectionSummary()->getValueObject() )
    {
        selectedValue_ = tree->getSelectionSummary()->calculateValue( tree->getValueModus(), false );
        if ( !std::isnan( selectedValue_ ) && !absoluteMode ) // additionally calculate absolute values
        {
            absValue_ = tree->getSelectionSummary()->calculateValue( ABSOLUTE_VALUES, false );
        }
        isInteger_ = tree->getLastSelection()->isIntegerType();
    }

    Tree* metric = Globals::getTabManager()->getActiveTree( METRIC );
    isRelativeMaximum_ = ( metric->getLastSelection()->isDerivedMetric() ) ? true : false;

    if ( selectionList.size() > 0 )
    {
        TreeItem* selection    =  tree->getLastSelection();
        double    maxTreeValue = tree->getMaxValue( selection );
        if ( tree->hasUserDefinedMinMaxValues() )
        {
            minValue_ = tree->getUserDefinedMinValue();
            maxValue_ = tree->getUserDefinedMaxValue();
        }
        else
        {
            minValue_ = 0.0;
            maxValue_ = absoluteMode ? maxTreeValue : 100.0;
        }

        // absolute values
        minAbsValue_ = 0;
        maxAbsValue_ = maxTreeValue;
    } // end selection exist

    userDefinedMinMax_ = tree->hasUserDefinedMinMaxValues();

    // mean and variance are only displayed in system tree, if no thread or process is selected in
    // percent value modus
    if ( std::isnan( selectedValue_ ) && tree->getType() == SYSTEM )
    {
        double   tmp = 0.;
        unsigned N   = 0;
        foreach( TreeItem * item, tree->getItems() )
        {
            if ( item->getChildren().size() == 0 ) // get sum of all leafs
            {
                tmp    = item->calculateValue( ABSOLUTE_VALUES, false );
                mean_ += tmp;
                N++;
            }
        }
        mean_ = ( N != 0 ) ? mean_ / N : 0.;

        double variance = 0.;
        foreach( TreeItem * item, tree->getItems() )
        {
            if ( item->getChildren().size() == 0 )
            {
                tmp       = item->calculateValue( ABSOLUTE_VALUES, false );
                variance += ( mean_ - tmp ) * ( mean_ - tmp );
                N++;
            }
        }
        varianceSqrt_ = ( N != 0 ) ? sqrt( variance / N ) : 0.;
    }

    valid = true;
}

void
TreeStatistics::invalidate()
{
    valid = false;
}

bool
TreeStatistics::isRelativeMaximum() const
{
    return isRelativeMaximum_;
}

bool
TreeStatistics::isValid() const
{
    return valid;
}

bool
TreeStatistics::isAbsoluteValueModus() const
{
    return absoluteMode;
}

bool
TreeStatistics::hasUserDefinedMinMaxValues() const
{
    return userDefinedMinMax_;
}

bool
TreeStatistics::isInteger() const
{
    return isInteger_;
}  // end TreeStatistics::update()

double
TreeStatistics::mean() const
{
    return mean_;
}

double
TreeStatistics::varianceSqrt() const
{
    return varianceSqrt_;
}

double
TreeStatistics::minValue() const
{
    return minValue_;
}

double
TreeStatistics::maxValue() const
{
    return maxValue_;
}

double
TreeStatistics::selectedValue() const
{
    return selectedValue_;
}

double
TreeStatistics::minAbsValue() const
{
    return minAbsValue_;
}

double
TreeStatistics::maxAbsValue() const
{
    return maxAbsValue_;
}

double
TreeStatistics::absValue() const
{
    return absValue_;
}
