/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2024                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"

#include <assert.h>
#include <math.h>
#include <QDebug>
#include <ctype.h>

#include "CubeRegion.h"
#include "CubeVertex.h"
#include "CubeValues.h"
#include "CubeCnode.h"

#include "PluginManager.h"
#include "Globals.h"
#include "Tree.h"
#include "TreeView.h"
#include "TreeItem.h"
#include "AggregatedTreeItem.h"
#include "MetricTree.h"
#include "TreeItemMarker.h"
#include "ValueView.h"
#include "DefaultValueView.h"
#include "TreeConfig.h"

using namespace cubegui;

TreeItem::TreeItem(   Tree*          tree,
                      const QString& name,
                      TreeItemType   type,
                      cube::Vertex*  cube )
{
    this->name        = name;
    this->label       = name;
    this->type        = type;
    this->callType    = CallItemType::StandardItem;
    this->cubeObject  = cube;
    this->tree        = tree;
    this->displayName = "";
    ownValue_adv      = nullptr;
    totalValue_adv    = nullptr;

    rootItem   = NULL;
    parentItem = 0;
    depth      = 0;

    singleValue = true;
    expanded    = false;
    hidden      = false;
    selected    = false;
    toCalculate = !tree->isManualCalculation();
    invalidate();
}

TreeItem::~TreeItem()
{
    if ( ownValue_adv != NULL )
    {
        delete ownValue_adv;
    }
    if ( totalValue_adv != NULL )
    {
        delete totalValue_adv;
    }
}

int
TreeItem::getDepth() const
{
    return depth;
}

bool
TreeItem::isCalculated() const
{
    return getCalculationStatus() != INVALID;
}

bool
TreeItem::isCalculated( ItemState state ) const
{
    CalculationStatus calc = ( state == ItemState::Exclusive ) ? statusExclusive : statusInclusive;
    return calc != INVALID;
}

void
TreeItem::invalidate( bool inclusive, bool exclusive )
{
    if ( inclusive )
    {
        statusInclusive = INVALID;
        delete totalValue_adv;
        totalValue_adv = nullptr;
        totalValue     = std::nan( "" );
        textCollapsed  = "";
        colorCollapsed = Qt::white;
    }
    if ( exclusive )
    {
        statusExclusive = INVALID;
        delete ownValue_adv;
        ownValue_adv  = nullptr;
        ownValue      = std::nan( "" );
        textExpanded  = "";
        colorExpanded = Qt::white;
    }

    currentValue = nan( "" );
    displayValue = nan( "" );

    // the minimal and maximal peer values for system tree items
    minValue = 0.0;
    maxValue = 0.0;

    if ( displayName.isEmpty() )
    {
        displayName = TreeConfig::getInstance()->createDisplayName( this );
    }
    label = QString( " - " ).append( displayName );
}

void
TreeItem::invalidateLabel()
{
    if ( getCalculationStatus() == READY )
    {
        setCalculationStatus( CALCULATED );
    }
    if ( displayName.isEmpty() )
    {
        displayName = TreeConfig::getInstance()->createDisplayName( this );
    }
    label = QString( " - " ).append( displayName );

    displayValue   = nan( "" );
    currentValue   = nan( "" );
    colorExpanded  = Qt::white;
    colorCollapsed = Qt::white;
}

Tree*
TreeItem::getTree() const
{
    return tree;
}

cube::Vertex*
TreeItem::getCubeObject() const
{
    return cubeObject;
}


TreeItem*
TreeItem::child( int row ) const
{
    return children.value( row );
}

void
TreeItem::addChild( TreeItem* item )
{
    children.append( item );

    item->parentItem = this;
    item->setDepth( this->getDepth() + 1 );
}

bool
TreeItem::isHidden() const
{
    return this->hidden;
}

void
TreeItem::setHidden( bool hide )
{
    tree->setHidden( this, hide ); // set hidden flag but also update map of hidden children
}

bool
TreeItem::isSelected() const
{
    return this->selected;
}


const QColor&
TreeItem::getColor() const
{
    return expanded ? colorExpanded : colorCollapsed;
}


bool
TreeItem::isIntegerType() const
{
    if ( tree->getValueModus() != ABSOLUTE_VALUES )
    {
        return false;
    }
    if ( tree->getType() == METRIC )
    {   // each item has different type
        return ( static_cast<cube::Metric*> ( cubeObject ) )->isIntegerType();
    }
    else
    {   //  all tree items have the same type
        return tree->hasIntegerType();
    }
}

bool
TreeItem::isTopLevelItem() const
{
    return !( getParent() && getParent()->getParent() );
}

bool
TreeItem::isDerivedMetric() const
{
    bool isDerived = false;

    cube::Metric* metric = dynamic_cast<cube::Metric*> ( this->getCubeObject() );
    if ( metric )
    {
        cube::TypeOfMetric metricType = metric->get_type_of_metric();
        isDerived = metricType == cube::CUBE_METRIC_POSTDERIVED || metricType == cube::CUBE_METRIC_PREDERIVED_EXCLUSIVE || metricType == cube::CUBE_METRIC_PREDERIVED_INCLUSIVE;
    }

    return isDerived;
}

const TreeItem*
TreeItem::getTopLevelItem() const
{
    return rootItem;
}


void
TreeItem::getSourceInfo( QString& filename, int& startLine, int& endLine ) const
{
    cube::Region* region = 0;
    startLine = -1;
    endLine   = -1;
    filename.clear();

    if ( !getCubeObject() )
    {
        return;
    }

    if ( type == CALLITEM )
    {
        cube::Cnode* cnode = static_cast<cube::Cnode*>( getCubeObject() );
        region = static_cast<cube::Region*>( cnode->get_callee() );
    }
    else if ( type == REGIONITEM )
    {
        region = static_cast<cube::Region*>( getCubeObject() );
    }

    if ( region )
    {
        std::string str = region->get_mod();
        filename  = QString::fromStdString( region->get_mod() ).trimmed();
        startLine = region->get_begn_ln();
        endLine   = region->get_end_ln();
    }
    if ( ( filename == "MPI" ) || ( filename == "INTERNAL" ) || ( filename ==  "OMP" )  || ( filename == "PTHREAD" ) ||
         ( filename == "CUDA" ) || ( filename ==  "OPENCL" ) )
    {
        filename = "";
    }
}

void
TreeItem::setMarker( const TreeItemMarker* marker, bool isDependencyMarker, bool markParents )
{
    if ( isDependencyMarker )
    {
        if ( !dependencyMarkerList.contains( marker ) )
        {
            dependencyMarkerList.append( marker );
        }
    }
    else
    {
        markerList.append( marker );
    }

    if ( markParents )
    {
        TreeItem* parent = this->getParent();
        while ( parent )
        {
            if ( !parent->parentMarkerList.contains( marker ) )
            {
                parent->parentMarkerList.append( marker );
            }
            parent = parent->getParent();
        }
    }
}

void
TreeItem::removeMarker( const TreeItemMarker* marker )
{
    bool found = markerList.removeOne( marker );
    if ( found )
    {
        TreeItem* parent = this->getParent();
        while ( parent )
        {
            found  = parent->parentMarkerList.removeOne( marker );
            parent = found ? parent->getParent() : 0;
        }
    }

    dependencyMarkerList.removeOne( marker );
}

const QList<const TreeItemMarker*>&
TreeItem::getMarkerList() const
{
    return markerList;
}

double
TreeItem::getAbsoluteValue() const
{
    return expanded ? ownValue : totalValue;
}

double
TreeItem::getOwnValue() const
{
    return ownValue;
}

const cube::Value*
TreeItem::getOwnValueObject() const
{
    const std::lock_guard<std::mutex> lockGuard( cmutex );
    return ownValue_adv;
}

double
TreeItem::getTotalValue() const
{
    return totalValue;
}

const cube::Value*
TreeItem::getTotalValueObject() const
{
    const std::lock_guard<std::mutex> lockGuard( cmutex );
    return totalValue_adv;
}

cube::Value*
TreeItem::getValueObject() const
{
    const std::lock_guard<std::mutex> lockGuard( cmutex );
    return expanded ? this->ownValue_adv : this->totalValue_adv;
}

/* functions for use in the corresponding model TreeModel */

TreeItem*
TreeItem::getParent() const
{
    return parentItem;
}

int
TreeItem::row() const
{
    if ( parentItem )
    {
        return parentItem->children.indexOf( const_cast<TreeItem*>( this ) );
    }

    return 0;
}

TreeItemType
TreeItem::getType() const
{
    return type;
}

// private function for use in trees after new values have been calculated
void
TreeItem::setValues( cube::Value* total, cube::Value* own )
{
    setInclusiveValue( total );
    setExclusiveValue( own );
}

void
TreeItem::setValue( cube::Value* value )
{
    if ( isExpanded() )
    {
        setExclusiveValue( value );
    }
    else
    {
        setInclusiveValue( value );
    }
}

void
TreeItem::setInclusiveValue( cube::Value* value )
{
    const std::lock_guard<std::mutex> lockGuard( cmutex );
    if ( totalValue_adv != value && totalValue_adv != NULL )
    {
        delete totalValue_adv;
    }
    totalValue_adv  = value;
    totalValue      = ( totalValue_adv != NULL && !( totalValue_adv->isZero() ) ) ? Globals::getValueView( totalValue_adv->myDataType() )->getDoubleValue( totalValue_adv ) : 0.;
    statusInclusive = CALCULATED;
    singleValue     = ( totalValue_adv != NULL ) ? totalValue_adv->singleValue() : true;
}

void
TreeItem::setExclusiveValue( cube::Value* value )
{
    const std::lock_guard<std::mutex> lockGuard( cmutex );
    if ( ownValue_adv != value && ownValue_adv != NULL )
    {
        delete ownValue_adv;
    }
    ownValue_adv    = value;
    ownValue        = ( ownValue_adv != NULL && !( ownValue_adv->isZero() ) ) ? Globals::getValueView( ownValue_adv->myDataType() )->getDoubleValue( ownValue_adv ) : 0.;
    statusExclusive = CALCULATED;
    singleValue     = ( ownValue_adv != NULL ) ? ownValue_adv->singleValue() : true;
}

bool
TreeItem::isLeaf() const
{
    return getChildren().count() == 0;
}

bool
TreeItem::isAggregatedLoopItem() const
{
    return callType == CallItemType::LoopItem;
}

bool
TreeItem::isAggregatedRootItem() const
{
    return callType == CallItemType::AggregatedLoopRoot;
}

CallItemType
TreeItem::getCallType() const
{
    return callType;
}


cube::Cnode*
TreeItem::getCnode() const
{
    return dynamic_cast<cube::Cnode* > ( cubeObject );
}

void
TreeItem::setCallType( CallItemType newCallType )
{
    callType = newCallType;
}

QList<TreeItem*>
TreeItem::getLeafs() const
{
    QList<TreeItem*> leafs;

    QList<TreeItem*> children = getChildren();
    while ( !children.isEmpty() )
    {
        TreeItem* item = children.last();
        children.removeLast();
        if ( item->getChildren().size() == 0 )
        {
            leafs.append( item );
        }
        else
        {
            foreach( TreeItem * child, item->getChildren() )
            {
                children.append( child );
            }
        }
    }

    return leafs;
}

// ======== private ===================================================================
//

void
TreeItem::removeChild( TreeItem* item )
{
    assert( item->getParent() == this );

    children.removeOne( item );
    item->parentItem = NULL;
}

/**
 * @brief returns the value of this item adapted to the given value modus
 */
double
TreeItem::calculateValue( ValueModus valueModus, bool exclusive ) const
{
    cube::Value* valueObj = ( exclusive ? this->ownValue_adv : this->totalValue_adv );

    if ( !valueObj )
    {
        return std::nan( "" );
    }

    // if the metric tab is on the right, then the value is undefined
    DisplayType        currentDisplay = getTree()->getType();
    QList<DisplayType> order          = Globals::getTabManager()->getOrder();
    foreach( DisplayType display, order )
    {
        if ( display == METRIC ) // metric is on the left or current is metric
        {
            break;
        }
        else if ( display == currentDisplay ) // metric is on the right of current one
        {
            return std::nan( "" );
        }
    }

    // get the absolute value
    double value = Globals::getValueView( valueObj->myDataType() )->getDoubleValue( valueObj );

    if ( value != 0. && valueObj->isZero() )
    {
        /* For some metrics (e.g. metrics which are building a minimum) no valid exclusive value exists.
         * The zero value for a minimum metrics is the maximum negative value. In this case, the value
         * should not be displayed and be marked as invalid.
         */
        return std::nan( "" );
    }

    double roundThreshold = Globals::getRoundThreshold( FORMAT_TREES );

    if ( tree->getType() == SYSTEM && exclusive && children.size() > 0 )
    {
        return std::nan( " " );
    }

    // compute the value in the current modus
    if ( valueModus != ABSOLUTE_VALUES && valueModus != ABSOLUTE_PEER_COLORS )
    {
        double referenceValue = 0.0;

        if ( valueModus == OWNROOT_VALUES )
        {
            referenceValue = this->rootItem->totalValue;
        }
        else if ( valueModus == PEER_VALUES )
        {
            referenceValue = this->maxValue;
        }
        else if ( type == METRICITEM && valueModus == EXTERNAL_VALUES )
        {
            MetricTree*     metricTree = static_cast<MetricTree*> ( tree );
            const TreeItem* item       = getTopLevelItem();
            cube::Metric*   metric     = static_cast<cube::Metric* > ( item->getCubeObject() );
            QString         metricName = QString::fromStdString( metric->get_uniq_name() );

            referenceValue = metricTree->getExternalReferenceValue( metricName );
        }
        else // only one reference value for the complete tree
        {
            referenceValue = tree->getValueModusReferenceValue();
        }

        if ( value <= roundThreshold && value >= -roundThreshold ) // don't calculate percentage for values near zero
        {
            value = 0.0;
        }

        if ( !std::isnan( value ) )
        {
            if ( valueModus != PEERDIST_VALUES )
            {
                if ( referenceValue == 0.0 )
                {
                    if ( value != 0.0 )
                    {
                        value = std::nan( "" );
                    }
                }
                else
                {
                    value = 100.0 * value / referenceValue;
                }
            }
            else // peer distribution
            {
                double referenceValue1 = this->maxValue;
                double referenceValue2 = this->minValue;
                double diff            = ( referenceValue1 - referenceValue2 );
                if ( diff == 0.0 )
                {
                    if ( value - referenceValue2 != 0 )
                    {
                        value = std::nan( "" );
                    }
                    else
                    {
                        value = 0.0;
                    }
                }
                else
                {
                    if ( referenceValue2 <= roundThreshold && referenceValue2 >= -roundThreshold )
                    {   // don't calculate percentage for values near zero
                        referenceValue2 = 0.0;
                    }
                    value = 100.0 * ( value - referenceValue2 ) / diff;
                }
            }
        }
    }

    // round to 0.0 if very close to it
    if ( value <= roundThreshold && value >= -roundThreshold )
    {
        value = 0.0;
    }

    return value;
}

static const QColor unfinishedColor( 220, 220, 220 );

/*
 * @brief updateItem has to be called if the value of an tree item has been changed.
 * It sets the item label and its color.
 * @param calculationFinished true, if all tree values and maximum/minimum have been calculated
 */
void
TreeItem::updateItem( bool maximumIsCalculated )
{
    if ( !maximumIsCalculated && tree->getValueModus() != ABSOLUTE_VALUES )
    {
        return; // only show progress in absolute value modus
    }

    // cubelib has finished calculation -> update item label and value
    if ( getCalculationStatus() == CALCULATED )
    {
        colorExpanded  = unfinishedColor;
        colorCollapsed = unfinishedColor;

        setCalculationStatus( READY );

        double value;                // value including values of hidden children

        value              = calculateValue( tree->getValueModus(), isExpanded() );
        this->currentValue = value;

        // ---- sets the label of the item which will be displayed in the tree view
        QString text = "";
        if ( std::isnan( value ) )
        {
            text.append( "-" );
        }
        else // value exists
        {
            if ( singleValue )
            {
                text.append( Globals::formatNumber( value, isIntegerType(), FORMAT_TREES, this ) );
            }
            else
            {
                text.append( Globals::getValueView( getValueObject()->myDataType() )->toString( this ) );
            }
        }
        text.append( " " );

        text.append( displayName );

        // add units to the labels of the top level items of the metric tree
        if ( isTopLevelItem() && type == METRICITEM && tree->getValueModus() == ABSOLUTE_VALUES )
        {
            cube::Metric* met = static_cast<cube::Metric*> ( this->cubeObject );
            QString       unit( met->get_uom().c_str() );

            if ( met->get_parent() == NULL )
            {
                if ( unit.size() > 0 )
                {
                    text.append( " (" );
                    text.append( unit ); // add the unit of measurement
                    text.append( ")" );
                }
            }
        }

        // ----- write info about hidden children's value/number
        QList<TreeItem*> children = tree->getHiddenChildren( this );
        if ( !std::isnan( value ) && expanded && !children.isEmpty() )
        {
            Tree* metric          = Globals::getTabManager()->getActiveTree( METRIC );
            bool  isDerivedMetric = metric->getLastSelection()->isDerivedMetric();
            if ( ( tree->getFilter() == Tree::FILTER_DYNAMIC ) && ( !isDerivedMetric ) )
            {
                double hiddenChildrenValue = 0.0;
                for ( TreeItem* child : children )
                {
                    if (  std::isnan( child->getValue() ) )
                    {
                        child->updateItem( true ); // hidden tree items don't get updated automatically
                    }

                    hiddenChildrenValue += child->getValue();
                }
                double eps     = Globals::getRoundThreshold( FORMAT_TREES );
                double percent = value > eps ? 100.0 * hiddenChildrenValue / value : 0;
                text.append( " (" );
                text.append( Globals::formatNumber( percent, false, FORMAT_TREES, this ) );
                text.append( "% hidden)" );
            }
            else // derived metric or statically hidden (children are not calculated)
            {
                text.append( " (" );
                text.append( QString::number( children.size() ) );
                QString hidden = children.size() > 1 ? "hidden children" : "hidden child";
                text.append( " " + hidden + ")" );
            }
        }

        this->label        = text;
        this->displayValue = value;
    } // end if CALCULATED

    if ( maximumIsCalculated )
    {
        int  hiddenChildren = 0;
        bool hiddenValueOk  = true;  // todo

        // set the colors for the item
        QColor color = Globals::getColor( 0, 0, 0 );
        if ( std::isnan( displayValue ) || ( hiddenChildren > 0 && !hiddenValueOk ) )
        {
            color = Globals::getColor( 0, 0, 0 );
        }
        else
        {
            if ( this->getTree()->getValueModus() != ABSOLUTE_PEER_COLORS )
            {
                color = calculateColor( displayValue );
            }
            else // special case: use absolute values but calculate percentage for coloring
            {
                double referenceValue = this->maxValue;
                double colorValue     = 0;
                if ( referenceValue == 0.0 )
                {
                    colorValue =  displayValue == 0.0 ? 0 : std::nan( "" );
                }
                else
                {
                    colorValue = 100.0 * displayValue / referenceValue;
                }
                color = calculateColor( colorValue );
            }
        }

        if ( expanded )
        {
            colorExpanded = color;
        }
        else
        {
            colorCollapsed = color;
        }
    }
}
// end of displayItem()

QColor
TreeItem::calculateColor( double value ) const
{
    QColor color;
    if ( !tree->hasUserDefinedMinMaxValues() )
    {
        double min = 0.0;
        double max = 0.0;
        if ( tree->getValueModus() == ABSOLUTE_VALUES )
        {
            max = tree->getMaxValue( this );
        }
        else
        {
            max = 100.0;
        }
        // in the case, calculated min is greater than max in its ablut value, we swap them here
        if ( fabs( min ) > fabs( max ) )
        {
            double tmp = min;
            min = max;
            max = tmp;
        }

        // coloring is only according to the absolut values of the min and max.
        color = Globals::getColor( fabs( value ), fabs( min ), fabs( max ) );
    }
    else // userDefinedMinMaxValues
    {
        // coloring is only according to the absolut values of the min and max.
        color = Globals::getColor( fabs( value ),
                                   fabs( tree->getUserDefinedMinValue() ),
                                   fabs( tree->getUserDefinedMaxValue() ) );
    }

    return color;
}

void
TreeItem::setExpanded( bool expand )
{
    tree->expandItem( this, expand );
}

void
TreeItem::select( bool addToSelection )
{
    tree->selectItem( this, addToSelection );
}

void
TreeItem::deselect()
{
    tree->deselectItem( this );
}

// ---- methods which are called by the view, if user has made selections

void
TreeItem::setExpandedStatus( bool expanded, bool recursive )
{
    if ( !isLeaf() )
    {
        this->expanded  = expanded;
        statusExclusive = statusExclusive == READY ? CALCULATED : statusExclusive;
        statusInclusive = statusInclusive == READY ? CALCULATED : statusInclusive;
    }
    if ( recursive )
    {
        foreach( TreeItem * item, this->getChildren() )
        {
            if ( !item->isLeaf() )
            {
                item->setExpandedStatus( expanded, true );
            }
        }
    }
}

void
TreeItem::setSelectionStatus( bool selected )
{
    this->selected = selected;
}


// sets the depth of this items and all its children recursively
void
TreeItem::setDepth( int value )
{
    this->depth = value;

    if ( children.size() > 0 )
    {
        QList<TreeItem*> list;
        list.append( this );
        while ( !list.isEmpty() )
        {
            TreeItem* item = list.takeLast();
            foreach( TreeItem * child, item->getChildren() )
            {
                child->depth = item->depth + 1;
                list.append( child );
            }
        }
    }
}

DisplayType
TreeItem::getDisplayType() const
{
    return tree->getType();
}

TreeType
TreeItem::getTreeType() const
{
    return tree->getTreeType();
}

QVariant
TreeItem::convertToQVariant()
{
    QList<QVariant> list;

    TreeItem* p = this;
    while ( p->parentItem )
    {
        list.insert( 0, p->parentItem->children.indexOf( const_cast<TreeItem*>( p ) ) );
        p = p->parentItem;
    }

    list.insert( 0, tree->getTreeType() );
    return QVariant( list );
}

void
TreeItem::setCalculationEnabled( bool enabled )
{
    toCalculate = enabled;
}

TreeItem*
TreeItem::convertQVariantToTreeItem( QVariant entry )
{
    QList<QVariant> list = entry.toList();

    TreeType type = ( TreeType )list.takeFirst().toInt();
    Tree*    tree = Globals::getTabManager()->getTree( type );

    TreeItem* item = tree->getRootItem();
    foreach( QVariant pos, list )
    {
        if ( pos.toInt() < item->children.size() )
        {
            item = item->children.at( pos.toInt() );
        }
        else
        {
            return NULL; // position not valid
        }
    }
    return item;
}

QString
TreeItem::getUrl() const
{
    if ( getCubeObject() == NULL )
    {
        return QString( "" );
    }

    // we must distinguish on the item type to call the right get method
    if ( getType() == METRICITEM )
    {
        return QString::fromStdString( ( ( cube::Metric* )( getCubeObject() ) )->get_url() );
    }
    else if ( getType() == CALLITEM )
    {
        return QString::fromStdString( ( ( cube::Cnode* )( getCubeObject() ) )->get_callee()->get_url() );
    }
    else if ( getType() == REGIONITEM && getCubeObject() != NULL )
    {
        return QString::fromStdString( ( ( cube::Region* )( getCubeObject() ) )->get_url() );
    }

    return QString();
}
