/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"

#include "CubeIdIndexMap.h"
#include "CubeLocation.h"
#include "CubeMetric.h"
#include "CubeProxy.h"
#include "CallTree.h"
#include "FlatTree.h"
#include "MetricTree.h"
#include "SystemTree.h"
#include "TabManager.h"
#include "TreeConfig.h"
#include "TreeView.h"
#include <QDebug>

using namespace cube;
using namespace std;
using namespace cubegui;

SystemTree::SystemTree( CubeProxy* cubeProxy ) : Tree( cubeProxy ), future( nullptr )
{
    treeType    = SYSTEMTREE;
    displayType = SYSTEM;
    label       = tr( "System tree" );
}

list_of_sysresources
SystemTree::getSelectedNodes()
{
    QList< TreeItem* >   selected_systemnodes = selectionList;
    list_of_sysresources sysres_selection;
    foreach( TreeItem * item, selected_systemnodes )
    {
        sysres_pair sp;
        sp.first  = static_cast<cube::Sysres*> ( item->getCubeObject() );
        sp.second = ( item->isExpanded() ) ? cube::CUBE_CALCULATE_EXCLUSIVE : cube::CUBE_CALCULATE_INCLUSIVE;
        sysres_selection.push_back( sp );
    }
    return sysres_selection;
}

/**
 * creates a system tree
 * The first level of the system tree consists of items of type SYSTEMTREENODEITEM. The children
 * of this level can be also SYSTEMTREENODEITEM.
 * The last two levels of the tree are items of type LOCATIONGROUPITEM which have children of
 * type LOCATION.
 */
TreeItem*
SystemTree::createTree()
{
    TreeItem* top = new TreeItem( this, QString(), SYSTEMTREENODEITEM, 0 );

    // insert root systen tree nodes and its children recursively
    createItems( top, cube->getRootSystemTreeNodes(), SYSTEMTREENODEITEM );

    // insert LOCATIONGROUP items but not its children of type LOCATION
    QList<TreeItem*> groups;
    createItems( top, cube->getLocationGroups(), LOCATIONGROUPITEM, &groups, false );

    // insert LOCATION items
    createItems( top, cube->getLocations(), LOCATIONITEM );

    // for single-threaded system trees : hide the threads
    bool singleThreaded = true;
    foreach( TreeItem * group, groups ) // for all items of type LOCATIONGROUPITEM
    {
        if ( group->getChildren().size() > 1 )
        {
            singleThreaded = false;
            break;
        }
    }
    if ( singleThreaded )
    {
        foreach( TreeItem * group, groups )
        {
            if ( group->getChildren().size() == 1 )
            {
                TreeItem*       child    = group->child( 0 );
                cube::Location* location = dynamic_cast<cube::Location*>( child->getCubeObject() );
                if ( location != nullptr && location->get_type() == cube::CUBE_LOCATION_TYPE_CPU_THREAD )
                {
                    group->removeChild( child );
                    treeItemHash.erase( child->getCubeObject() );
                    delete child;
                }
            }
        }
    }
    // end handling of single-threaded trees

    return top;
}

QString
SystemTree::getItemName( cube::Vertex* vertex ) const
{
    return QString( ( ( cube::Sysres* )vertex )->get_name().c_str() );
}

void
SystemTree::computeMinMax()
{
    double         tmp = 0.0;
    vector<double> min_vector;
    vector<double> max_vector;
    vector<bool>   depth_seen;
    foreach( TreeItem * item, getItems() )
    {
        size_t depth = item->getDepth();

        if ( min_vector.size() <= depth )
        {
            min_vector.resize( depth + 1 );
        }
        if ( max_vector.size() <= depth )
        {
            max_vector.resize( depth + 1 );
        }
        while ( depth_seen.size() <= depth )
        {
            depth_seen.push_back( false );
        }

        tmp = item->totalValue;

        if ( !depth_seen[ depth ] )
        {
            depth_seen[ depth ] = true;
            min_vector[ depth ] = tmp;
            max_vector[ depth ] = tmp;
        }
        else
        {
            min_vector[ depth ] = ( min_vector[ depth ] > tmp ) ? tmp : min_vector[ depth ];
            max_vector[ depth ] = ( max_vector[ depth ] < tmp ) ? tmp : max_vector[ depth ];
        }
    }

    foreach( TreeItem * item, getItems() )
    {
        int depth = item->getDepth();
        item->minValue = min_vector[ depth ];
        item->maxValue = max_vector[ depth ];
    }
}


void
SystemTree::assignValues( TreeItem*               item,
                          const vector< Value* >& inclusive_values,
                          const vector< Value* >& exclusive_values )
{
    assert( item );
    assert( item->getCubeObject() );

    SystemTreeNode* stn = static_cast<SystemTreeNode*>( item->getCubeObject() );
    uint32_t        id  = stn->get_sys_id();
    // value vector is empty, if selected left tree item is zero -> set to NULL
    item->setValues( ( ( id >= inclusive_values.size() ) ? NULL : inclusive_values[ id ] ),
                     ( ( id >= exclusive_values.size() ) ? NULL : exclusive_values[ id ] ) );

    foreach( TreeItem * child, item->getChildren() )
    {
        assignValues( child, inclusive_values, exclusive_values );
    }
}

/**
   returns true, if all items have to be recalculated and calculation isn't disabled
 */
bool
SystemTree::calculationRequired( const QList<TreeItem*>& itemsToCalculateNow )
{
    if ( !top || top->getChildren().size() == 0 )
    {
        return false;
    }
    if ( itemsToCalculateNow.size() > 0 )
    {
        return true;
    }
    TreeItem* visibleTop = top->getChildren().first();
    if ( visibleTop->isCalculationEnabled() && !visibleTop->isCalculated() )
    {
        return true;
    }
    return false;
}

bool
SystemTree::isValidSelection() const
{
    if ( !isValid )
    {
        return false;
    }
    for ( TreeItem* item : selectionList )
    {
        if ( item->isExpanded() && item->children.size() > 0 ) // only leafs and collapsed items are valid
        {
            return false;
        }
    }
    return true;
}

QList<Task*>
SystemTree::setBasicValues( const QList<Tree*>&     leftTrees,
                            const QList<Tree*>&     rightTrees,
                            const QList<TreeItem*>& itemsToCalculate )
{
    QList<Task*> workerData;
    /* No update of tree is necessary if the system tree is to the left of the metric tree */
    if ( ( leftTrees.size() == 0 ) || ( ( leftTrees.size() == 1 ) && ( leftTrees[ 0 ]->getType() != METRIC ) ) )
    {
        return workerData;                   // the system tree is left of the metric tree.
    }

    // all visible items are already up to date and selectionSummary is a single item
    if ( !calculationRequired( itemsToCalculate ) && ( selectionList.size() == 1 ) )
    {
        return workerData;
    }

    QSet<DisplayType>         isLeft;
    QHash<DisplayType, Tree*> trees;
    for ( Tree* tree : leftTrees )
    {
        isLeft.insert( tree->getType() );
        trees.insert( tree->getType(), tree );
    }
    for ( Tree* tree : rightTrees )
    {
        trees.insert( tree->getType(), tree );
    }

    // Set up selection for metric tree
    list_of_metrics metric_selection;
    MetricTree*     metricTree = static_cast<MetricTree*>( trees[ METRIC ] );
    metric_selection = metricTree->getSelectedMetrics();

    // Set up selection for the active call tree
    CallTree*             callTree        = static_cast<CallTree*> ( trees[ CALL ] );
    const list_of_cnodes& cnode_selection = isLeft.contains( CALL ) ? callTree->getSelectedNodes() : callTree->getNodes();

    // temporary hack to support artificial kernel nodes, which only exist in GUI -> cubeProxy change is planned
    {
        if ( isLeft.contains( CALL ) && cnode_selection.empty() )
        {
            TreeItem* callItem = Globals::getTabManager()->getActiveTree( CALL )->getSelectionList().first();
            if ( callItem->getCallType() == CallItemType::ArtificialKernelRoot )
            {
                this->invalidateItems();
                return workerData; // no valid call tree selection (expanded artificial kernel root)
            }
        }
    }

    SystemTreeData* data = new SystemTreeData( this,
                                               metric_selection,
                                               cnode_selection );
    workerData.append( data );


    // calculate sum of the selected items for the statics in the value view
    if ( selectionList.size() > 1 )
    {
        TreeTask* sum = new TreeTask( selectionSummary,
                                      metric_selection,
                                      cnode_selection,
                                      getSelectedNodes() );
        workerData.append( sum );
    }

    computeMinMax();
    return workerData;
}
// end of computeBasicValues()

/**
 * @brief calculateVisitedItems
 * Calculates all threads with visited > 0
 * The signal visitedCalculated( QList<TreeItem*> ) is emitted when done.
 */
void
SystemTree::calculateVisitedItems( bool useCallTreeSelections )
{
    QList<TreeItem*> visited;

    Tree* metricTree = getActiveTree( METRIC );
    Tree* callTree   = getActiveTree( CALL );

    TreeItem* visitedMetric = 0;
    foreach( TreeItem * item, metricTree->getItems() )
    {
        if ( item->getName() == "Visits" )
        {
            visitedMetric = item;
            break;
        }
    }
    if ( !visitedMetric )
    {
        return;
    }

    // build up metric selection
    list_of_metrics metric_selection;
    metric_selection.push_back( make_pair( static_cast<Metric*>( visitedMetric->getCubeObject() ), CUBE_CALCULATE_INCLUSIVE ) );

    // build up cnode selection
    CallTree*      call            = static_cast<CallTree*> ( callTree );
    list_of_cnodes cnode_selection = useCallTreeSelections ? call->getNodes( call->getSelectionList() ) : CubeProxy::ALL_CNODES;

    vector<Value*> inclusive_values;
    vector<Value*> exclusive_values;

    // calculated data for metric "visited"
    QList<Task*> workerData;
    VisitedData* sum = new VisitedData( this,
                                        metric_selection,
                                        cnode_selection );
    workerData.append( sum );
    future.addCalculations( workerData );
    future.startCalculation( true );
}

void
VisitedData::calculate()
{
    // request bulk data from CubeProxy
    vector< Value* > inclusive_values;
    vector< Value* > exclusive_values;

    // get values for full tree
    cube::CubeProxy* cube = tree->getCube();
    cube->getSystemTreeValues( metric_selection,
                               cnode_selection,
                               inclusive_values,
                               exclusive_values );

    QList<TreeItem*> visited;
    for ( TreeItem* item: tree->getItems() )
    {
        if ( ( ( item->getType() == LOCATIONGROUPITEM && item->getChildren().size() == 0 )
               || ( item->getType() == LOCATIONITEM ) )
             && ( inclusive_values[ static_cast<Sysres*>( item->getCubeObject() )->get_sys_id() ]->getSignedLong() > 0 ) )
        {
            visited.append( item );
        }
    }
    emit tree->visitedCalculated( visited );
}

TreeItem*
SystemTree::getTreeItem( uint32_t systemId ) const
{
    foreach( TreeItem * item, treeItems )
    {
        cube::Sysres* sysres = static_cast<cube::Sysres*> ( item->getCubeObject() );
        if ( sysres->get_sys_id() == systemId )
        {
            return item;
        }
    }
    return 0;
}

void
SystemTree::assignValues( TreeItem*               item,
                          const vector< Value* >& inclusive_values,
                          const vector< Value* >& exclusive_values,
                          int                     depth,
                          cube::IdIndexMap*       indexMap
                          )
{
    assert( item );

    if ( item->getDepth() > depth )
    {
        return;
    }

    if ( item->getCubeObject() )
    {
        size_t id = item->getCubeObject()->get_id();
        if ( indexMap )
        {
            id = indexMap->getIndex( id );
        }

        if ( inclusive_values.size() > 0 )
        {
            item->setInclusiveValue( ( id >= inclusive_values.size() ) ? NULL : inclusive_values[ id ] );
        }
        if ( exclusive_values.size() > 0 )
        {
            item->setExclusiveValue( ( id >= exclusive_values.size() ) ? NULL : exclusive_values[ id ] );
        }

        foreach( TreeItem * child, item->getChildren() )
        {
            assignValues( child, inclusive_values, exclusive_values, depth, indexMap );
        }
    }
}

/** Sets items that should remain hidden independant of other hiding mechanisms.
   This refers to unvisited locations, if this feature is enabled in Display->Trees.
 */
void
SystemTree::setPredefinedHiddenItems()
{
    if ( TreeConfig::getInstance()->isHidingOfUnvisitedEnabled() )
    {
        connect( this, &SystemTree::visitedCalculated,
                 this, &SystemTree::setPredefinedHiddenItemDone );
        calculateVisitedItems( false ); // async calculation
    }
}
void
SystemTree::setPredefinedHiddenItemDone( QList<TreeItem*> visitedSubset )
{
    disconnect( this, &SystemTree::visitedCalculated,
                this, &SystemTree::setPredefinedHiddenItemDone );
    if ( TreeConfig::getInstance()->isHidingOfUnvisitedEnabled() )
    {
        predefinedHidden.clear();
        for ( TreeItem* item : qAsConst( treeItems ) )
        {
            if ( item->isLeaf() && !visitedSubset.contains( item ) )
            {
                cube::Sysres* sys  = static_cast<cube::Sysres* >( item->getCubeObject() );
                LocationType  type = ( static_cast<cube::Location*>( sys ) )->get_type();

                // exception metric locations: not related to the visit metric
                if ( type != cube::CUBE_LOCATION_TYPE_METRIC )
                {
                    predefinedHidden.append( item );
                    setHidden( item, true );
                }
            }
        }
    }
    else // unhide tree items
    {
        for ( TreeItem* item : qAsConst( predefinedHidden ) )
        {
            setHidden( item, false );
        }
        predefinedHidden.clear();
    }
}

// ---------------------------------------------------------------------------------

void
SystemTreeData::calculate()
{
    // request bulk data from CubeProxy
    vector< Value* > inclusive_values;
    vector< Value* > exclusive_values;

    // get values for full tree
    cube::CubeProxy* cube = tree->getCube();
    cube->getSystemTreeValues( metric_selection,
                               cnode_selection,
                               inclusive_values,
                               exclusive_values );

    // Assign values to each toplevel tree
    foreach( TreeItem * child, tree->getRootItem()->getChildren() )
    {
        tree->assignValues( child, inclusive_values, exclusive_values );
    }
    tree->computeMinMax();
}

void
SystemTreeFlatData::calculate()
{
    // request bulk data from CubeProxy
    vector< Value* > inclusive_values;
    vector< Value* > exclusive_values;

    // get values for full tree
    cube::CubeProxy* cube = tree->getCube();
    cube->getSystemTreeValues( metric_selection,
                               region_selection,
                               inclusive_values,
                               exclusive_values );

    // Assign values to each toplevel tree
    foreach( TreeItem * child, tree->getRootItem()->getChildren() )
    {
        tree->assignValues( child, inclusive_values, exclusive_values );
    }
    tree->computeMinMax();
}
