/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#ifndef CUBEGUI_MAINWIDGET_H
#define CUBEGUI_MAINWIDGET_H

#include "cubegui-concurrent.h"
#include "CubeTypes.h"
#include "Constants.h"
#include "Future.h"
#include "Globals.h"
#include "Settings.h"
#include "SettingsHandler.h"
#include "StatusBar.h"
#include <QComboBox>
#include <QLineEdit>
#include <QMainWindow>
#include <QMenuBar>
#include <QProgressDialog>
#include <QPushButton>
#include <QScrollArea>
#include <QSettings>
#include <QSplitter>
#include <QStackedWidget>
#include <QStatusBar>
#include <QTextBrowser>
namespace cube
{
class CubeProxy;
}

namespace cubepluginapi
{
class PluginServices;
}

namespace cubegui
{
class TabWidget;
class ColorWidget;
class ValueWidget;
class PrecisionWidget;
class ColorScale;
class Settings;
class TabManager;
class ColorMap;
class SynchronizationToolBar;
class Synchronization;
class WidgetWithBackground;
class ReadCubeThread;
class CubeApplication;
class ToolTip;

// this class defines the main window of the GUI
class MainWidget : public QMainWindow, public InternalSettingsHandler
{
    Q_OBJECT

public:
    enum CubeContext { CONTEXT_CUBE, CONTEXT_INIT, CONTEXT_FREE, CONTEXT_EMPTY };

    /********************public methods *****************/

    MainWidget( CubeApplication&   app,
                cube::CubeStrategy strategy = cube::CUBE_ALL_IN_MEMORY_STRATEGY );
    ~MainWidget();

    // load the contents of a cube fie into a cube object
    // and display the contents
    void
    loadFile( const QString fileName );

    // called from Future to load the contents of a cube file in a thread
    void
    readCubeTask( const QString& fileName );

    // sets the status bar text
    void
    setMessage( const QString& message = "",
                MessageType    type = Information,
                bool           isLogged = true );

    // InternalSettingsHandler interface
    void
    loadExperimentSettings( QSettings& ) override;
    void
    saveExperimentSettings( QSettings& ) override;
    void
    saveGlobalStartupSettings( QSettings& settings ) override;
    void
    loadGlobalStartupSettings( QSettings& settings ) override;
    void
    saveGlobalSettings( QSettings& settings ) override;
    void
    loadGlobalSettings( QSettings& settings ) override;
    void
    loadStatus( QSettings& settings ) override;
    void
    saveStatus( QSettings& settings ) override;
    QString
    settingName() override;

    void
    setContext( const CubeContext& context );

    void
    initGeometry( bool init )
    {
        _initGeometry = init;
    }

    void
    wheelEvent( QWheelEvent* event ) override;

    QString initialSystemTab; // for internal use

    void
    setPresentationMode( bool enable );

#ifdef __EMSCRIPTEN__
    void
    changeEvent( QEvent* event ) override;

#endif

private:

    void
    cubeReportLoaded();

    void
    openCube( cube::CubeProxy* cube,
              const QString&   filename = "" );

#ifndef ANDROID
    // input a file name to open and open it via loadFile(...)
    void
    openFile();

#endif

    //input a URL to open a remote file and... //TODO
    void
    openRemote();

    // re-open the last file, the second last file, etc.
    void
    openLastFiles();

    // remove loaded data
    void
    closeFile();


    // save a copy of cube
    void
    saveAs();

    // input a file name to open as external file for external percentage
    // and open it via readExternalFile(...)
    bool
    openExternalFile();

    // remove loaded external data for external percentage
    void
    closeExternalFile();

    // closes the application after eventually saving settings
    void
    closeApplication();

    // set the order of metric/call/system splitter elements
    void
    setDimensionOrder();

    // displays a short intro to cube
    void
    introduction();

    // shows the about message
    void
    about();

    // list all mouse and keyboard control
    void
    keyHelp();

    // saves a screenshot in a png file
    // (it is without the shell frame, I could not find any possibilities to include that frame within qt)
    void
    screenshot();

    void
    updateColormapMenu();

    // causes to open a color dialog to allow the user to change color-specific settings
    void
    editColorMap();

    // allow one to select one of the registered colormaps
    void
    selectColorMap();

    // react on signal ColorMap::colorMapChanged()
    void
    updateColorMap();

    // causes to open a precision dialog to allow the user to change precision-specific settings
    void
    setPrecision();

    // distribute the available width between the 3 tabs such that
    // from each scroll area the same percentual amount is visible
    void
    distributeWidth();

    // adapt the size of the main window such that
    // all information are visible
    // and resize the splitter widget accordingly
    void
    adaptWidth();

    void
    selectExternalValueMode( TabWidget* widget );

    void
    updateProgress();

    void
    recentFileSelected( const QString& link );

    void
    recentFileHovered( const QString& url );

    void
    configureValueView();

    void
    showGuide();

    void
    showGuideExternal();

    void
    enableWebEngine( bool enable );

    void
    zoom( bool showTooltip = false );

signals:

    /******************** signals *****************/

    void
    cubeIsClosed();
    void
    enableExtClose( bool );

protected:
    virtual void
    closeEvent( QCloseEvent* event ) override;
    virtual void
    dragEnterEvent( QDragEnterEvent* event ) override;
    virtual void
    dropEvent( QDropEvent* event ) override;

    /** handle status tip event, because ToolBar class is used instead of QToolBar */
    bool
    event( QEvent* e ) override
    {
        if ( e->type() == QEvent::StatusTip )
        {
            QStatusTipEvent* ev = ( QStatusTipEvent* )e;
            statusBar->addLine( ev->tip(), Information, false );
            ev->accept();
            return true;
        }
        return QMainWindow::event( e );
    }

private:
    /******************** GUI elements **********************/
    QStackedWidget* stackedWidget;
    StatusBar*      statusBar;
    TabManager*     tabManager;
    ColorScale*     colorScale;
    QTimer*         timer;
    QMenu*          fileMenu;           // file menu containing u.o. the last file actions
    QMenu*          treeMenu;
    QMenu*          mapMenu;            // color map selection menu
    QMenu*          contextFreeMenu;
    QAction*        colorsAct;          // color map edit action
    QAction*        valueViewAct;       // ValueView configuration
    QAction*        performanceInfoAct; // help menu item
    QAction*        regionInfoAct;      // help menu item
    QAction*        presentationAction; // presentation cursor
    QAction*        emulateMouseAction; // emulate right mouse click
    QAction*        syncAction;
    QAction*        openExtAct;
    QAction*        saveAsAct;
    QAction*        selection1Act, * selection2Act;
    QAction*        splitterOrderAct;
    QAction*        closeAct;
    ToolTip*        zoomTip;
    QLabel*         recentFileWidget;
    QWidget*        contextFreeWidget;
#ifdef WITH_WEB_ENGINE
    QAction* webengine;
#endif
    WidgetWithBackground*   initialScreen;
    SynchronizationToolBar* syncToolBar;

    /******************** data **********************/
    Settings*        settings;
    bool             _initGeometry;
    QString          lastColorMapName;
    bool             fileLoaded;          // true, if cube data is sucessfully loaded
    int              zoomSteps;
    CubeApplication& app;

    Future  future;
    QString futureError;
    QTimer  connectionMonitorTimer;

    // flag to select load strategy...
    cube::CubeStrategy strategy;
    // the cube database for loaded data
    cube::CubeProxy* cube;
    // the cube database for loaded external data for external percentage
    cube::CubeProxy* cubeExternal;

    // stores at most 5 last opened files
    QStringList openedFiles;

    /// Stores the last successfully opened URLs
    QStringList openedUrls;

    // stores the name of the current external file
    QString lastExternalFileName;
    // stores the actions for opening the last at most 5 files
    std::vector<QAction*> lastFileAct;

    QHash<QObject*, int> initialFontSizesMain; // for zooming of static widgets
    QHash<QObject*, int> initialFontSizes;     // for zooming of new widgets
    // threshold for file size above which dynamic metric loading is enabled
    int dynloadThreshold;

    /******************** private methods **********************/
    // same as setMessage but can only be called from GUI thread

    void
    setMessage_( const QString& message,
                 MessageType    type,
                 bool           isLogged );

    // writes the current tree status to file
    void
    writeCube( const QString& filename ) const;

    // read the contents of a cube file into a temporary cube object for
    // the external percentage modus and compute the relevant values for the display
    bool
    readExternalFile( const QString fileName );

    // remember the names of the last 5 files that were opened;
    // they can be re-opened via the display menu
    void
    rememberFileName( QString fileName );

    // creates the pull-down menu
    void
    createMenu();

    // update the title of the application to show infos to the loaded file and external file
    void
    updateWidgetTitle();
    void
    setColorMap( ColorMap* map );

    /**
     * Go through color maps loaded in plugins in check whether one of them fits to name remembered in variable lastColorMapName
     * If yes, then set this map as active for Cube.
     */
    void
    loadColorMap();
    void
    createInitialScreen( QWidget* mainWidget );
    void
    fontDialog();
    void
    updateContextFreePluginWidget();
    void
    contextFreePluginStarted( bool started );
    void
    emulateRightMouse( bool enable );
    void
    deleteCube();

    // if CubeProxy uses a network connection this function is used to monitor it
    void
    monitorConnection();
};  // class MainWidget


/** tool tip widget, which can be altered */
class ToolTip : public QWidget
{
public:
    ToolTip( QWidget* parent );
    void
    setText( const QString& txt );

private:
    QLabel* label      = nullptr;
    int     fontsize   = 16;
    int     hideInMSec = 1000;
};

class WidgetWithBackground : public QWidget
{
public:
    void
    setImage( const QPixmap& pixmap )
    {
        if ( !pixmap.isNull() )
        {
            QImage image = pixmap.toImage();
            for ( int i = 0; i < image.height(); i++ )
            {
                for ( int j = 0; j < image.width(); j++ )
                {
                    //int gray = qGray(image.pixel(j, i));
                    QColor color = QColor::fromRgb( image.pixel( j, i ) );
                    color.setHsv( color.hue(), color.saturation() * .5, color.value() * .7 );
                    image.setPixel( j, i, color.rgba() );
                }
            }
            pix = QPixmap::fromImage( image );
        }
        else
        {
            pix = QPixmap();
        }
        update();
    }

protected:
    void
    paintEvent( QPaintEvent* )
    {
        if ( !pix.isNull() )
        {
            QPainter painter( this );
            painter.setBackgroundMode( Qt::TransparentMode );

            if ( pix.size() != this->size() )
            {
                painter.drawPixmap( QPointF( 0, 0 ), pix.scaled( this->size(), Qt::KeepAspectRatio, Qt::SmoothTransformation ) );
            }
            else
            {
                painter.drawPixmap( QPointF( 0, 0 ), pix );
            }
        }
    }

private:
    QPixmap pix;
};
}
#endif
