/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#define HAVE_CUBE_NETWORKING

#include "config.h"

#include <cmath>
#include <QAction>
#include <QApplication>
#include <QDateTime>
#include <QDialogButtonBox>
#include <QFileDialog>
#include <QHBoxLayout>
#include <QInputDialog>
#include <QMessageBox>
#include <QPixmap>
#include <QProgressDialog>
#include <QPushButton>
#include <QDialogButtonBox>
#include <QApplication>
#include <QShortcut>
#ifdef ANDROID
#include <QStandardPaths>
#if QT_VERSION < QT_VERSION_CHECK( 6, 0, 0 )
#include <QtAndroid>
#endif
#endif
#include <QVariant>
#include <QWhatsThis>
#include <QScrollArea>
#include <QScrollBar>

#include "ColorMap.h"
#include "ColorScale.h"
#include "Constants.h"
#include "ContextFreeServices.h"
#include "CubeApplication.h"
#include "CubeError.h"
#include "CubeIoProxy.h"
#include "CubeNetworkProxy.h"
#include "CubeOperationProgress.h"
#include "CubePlatformsCompat.h"
#include "CubeProxy.h"
#include "CubeServices.h"
#include "CubeTypes.h"
#include "CubeUrl.h"
#include "CubeWriter.h"
#include "DefaultColorMap.h"
#include "DefaultValueView.h"
#include "DimensionOrderDialog.h"
#include "Future.h"
#include "Globals.h"
#include "HelpBrowser.h"
#include "MainWidget.h"
#include "MetricTree.h"
#include "PluginManager.h"
#include "PluginServices.h"
#include "PluginServices.h"
#include "PrecisionWidget.h"
#include "RemoteFileDialog.h"
#include "Settings.h"
#include "StyleSheetEditor.h"
#include "SynchronizationToolBar.h"
#include "TabManager.h"
#include "TreeConfig.h"
#include "TreeView.h"
#include "ValueViewConfig.h"
#include "VersionCheck.h"

#include <cassert>
#include <iostream>
// #ifndef CUBE_COMPRESSED
#include <fstream>
// #else
#include "CubeZfstream.h"
// #endif
#include "HmiInstrumentation.h"
#ifdef WEB_SOCKETS
#include "CubeQtWebSocket.h"
#else
#include "CubeQtStreamSocket.h"
#endif


const static int FILE_HISTORY_COUNT = 20;
const static int minFontSize        = 6;
const static int maxFontSize        = 36;

using namespace std;
using namespace cubegui;
using namespace cubepluginapi;
using namespace cube;

MainWidget::MainWidget( CubeApplication& _app, cube::CubeStrategy _strategy ) : app( _app ), strategy( _strategy )
{
    cube                 = nullptr;
    cubeExternal         = nullptr;
    _initGeometry        = false;
    fileLoaded           = false;
    zoomSteps            = 0;
    lastExternalFileName = "";
    lastColorMapName     = "";
    recentFileWidget     = new QLabel();
    settings             = new Settings();
    tabManager           = new TabManager();
    zoomTip              = new ToolTip( this );

    Globals::getInstance()->setMainWidget( this );
    Globals::getInstance()->setTabManager( tabManager );
    Globals::getInstance()->setSettings( settings );
    resize( 800, 600 ); // fixed size ?
    setUpdatesEnabled( false );
    setAcceptDrops( true );
    setWindowIcon( QIcon( ":images/CubeIcon.xpm" ) );

#ifndef WEB_SOCKETS
    cube::Socket::setSocketFactory( cube::QtStreamSocket::create );
#else
    cube::Socket::setSocketFactory( cube::CubeQtWebSocket::create );
#endif

    stackedWidget = new QStackedWidget();
    stackedWidget->setWindowIcon( QIcon( ":images/CubeIcon.xpm" ) );

    StatusBarWidget* central = new StatusBarWidget();
    statusBar = central->getStatusBar();
    central->setWidget( stackedWidget );
    setCentralWidget( central );

    tabManager->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );

    colorScale = new ColorScale();
    colorScale->setVisible( false );

    QWidget* mainWidget = new QWidget();
    mainWidget->setObjectName( "main" );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    // LAYOUT_MARGIN and LAYOUT_SPACING are defined in constants.h
    mainLayout->setContentsMargins( LAYOUT_MARGIN, LAYOUT_MARGIN, LAYOUT_MARGIN, LAYOUT_MARGIN );
    mainLayout->setSpacing( LAYOUT_SPACING );
    mainWidget->setLayout( mainLayout );
    mainLayout->addWidget( tabManager );
    mainLayout->addWidget( colorScale );

    PrecisionWidget* precisionWidget = Globals::getPrecisionWidget();
    connect( precisionWidget, &PrecisionWidget::apply, tabManager, &TabManager::updateTreeItemProperties );

    /* initialize cube dependent plugins */
    PluginManager::getInstance()->setMainWindow( this );

    /* initialize context free plugins */
    ContextFreeServices* cfs = ContextFreeServices::getInstance();

    connect( PluginManager::getInstance(), &PluginManager::pluginPathChanged, this, &MainWidget::updateContextFreePluginWidget );

    initialScreen = new WidgetWithBackground();
    QWidget* contextFree = new QWidget( this );
    cfs->setWidget( contextFree );
    connect( cfs, &ContextFreeServices::openCubeRequest, this, [ this ]( cube::CubeProxy* cube ) {
        this->openCube( cube, "" );
    } );

    stackedWidget->addWidget( mainWidget );    // loaded cube (CONTEXT_CUBE)
    stackedWidget->addWidget( initialScreen ); // start screen (CONTEXT_INIT)
    stackedWidget->addWidget( contextFree );   // context free plugin (CONTEXT_FREE)
    stackedWidget->addWidget( new QWidget() ); // CONTEXT_EMPTY
    setContext( CONTEXT_INIT );

    syncToolBar = new SynchronizationToolBar();
    syncToolBar->setVisible( false );
    addToolBar( syncToolBar );

    // menu creation
    // note: system tab widget must be created before createMenu() gets called
    createMenu();

    setWhatsThis( CUBEGUI_FULL_NAME + tr( " is a presentation component suitable for displaying performance data for parallel programs including MPI and OpenMP applications. Program performance is represented in a multi-dimensional space including various program and system resources. The tool allows the interactive exploration of this space in a scalable fashion and browsing the different kinds of performance behavior with ease. Cube also includes a library to read and write performance data as well as operators to compare, integrate, and summarize data from different experiments." ) );

    menuBar()->setWhatsThis( tr( "The menu bar consists of three menus, a file menu, a display menu, and a help menu. Some menu functions have also a keyboard shortcut, which is written beside the menu item's name in the menu. E.g., you can open a file with Ctrl+O without going into the menu.  A short description of the menu items is visible in the status bar if you stay for a short while with the mouse above a menu item." ) );

    addToolBar( settings->getToolBar() );

    settings->registerSettingsHandler( this );
    settings->registerSettingsHandler( dynamic_cast<InternalSettingsHandler*> ( Globals::defaultColorMap ) );
    settings->registerSettingsHandler( dynamic_cast<InternalSettingsHandler*> ( Globals::defaultValueView ) );
    settings->registerSettingsHandler( PluginManager::getInstance() );

#ifdef WITH_WEB_ENGINE
    webengine->setChecked( Globals::optionIsSet( WebEngine ) );
#endif

    PluginManager::getInstance()->initializePlugins(); // requires settings to store plugin path
    createInitialScreen( initialScreen );
    setColorMap( Globals::getColorMap() );

    tabManager->setFocusPolicy( Qt::StrongFocus );
    tabManager->setFocus();

    // set all splitter childs to same size
    int        width = this->width();
    QList<int> sizes;
    sizes << width / 3 << width / 3 << width / 3;
    tabManager->setSizes( sizes );
    settings->loadGlobalSettings();
    setUpdatesEnabled( true );

    const int margin = 10;
    recentFileWidget->setContentsMargins( margin, margin, margin, margin );
    recentFileWidget->connect( recentFileWidget, &QLabel::linkActivated, this, &MainWidget::recentFileSelected );
    recentFileWidget->connect( recentFileWidget, &QLabel::linkHovered, this, &MainWidget::recentFileHovered );

    QShortcut* zoomPlus  = new QShortcut( QKeySequence( Qt::CTRL | Qt::Key_Plus ), this );
    QShortcut* zoomMinus = new QShortcut( QKeySequence( Qt::CTRL | Qt::Key_Minus ), this );
    zoomPlus->setContext( Qt::ApplicationShortcut );
    zoomMinus->setContext( Qt::ApplicationShortcut );

    connect( zoomPlus, &QShortcut::activated, this, [ this ](){
        zoomSteps++;
        zoom( true );
    } );
    connect( zoomMinus, &QShortcut::activated, this, [ this ](){
        zoomSteps--;
        zoom( true );
    } );

    if ( zoomSteps != 0 )
    {
        zoom(); // apply initial zoom from settings
    }
}
// end of constructor

#ifdef __EMSCRIPTEN__
// Workaround for wasm to prevent that a dialog gets unreachable when the main window gets on
// top of the dialog. Qt::WindowStaysOnTopHint for new dialogs doesn't work well with combo boxes
void
MainWidget::changeEvent( QEvent* event )
{
    QMainWindow::changeEvent( event );
    if ( event->type() == QEvent::ActivationChange )
    {
        // Raise all visible QDialogs
        for ( QWidget* widget : QApplication::allWidgets() )
        {
            QDialog* dialog = qobject_cast<QDialog*>( widget );
            if ( dialog && dialog->isVisible() )
            {
                dialog->raise();
                dialog->activateWindow();
            }
        }
    }
}
#endif

void
MainWidget::setContext( const CubeContext& context )
{
    stackedWidget->setCurrentIndex( context );
}

/** A scroll area which adapts the size to its contents.
 *  The size hint of QScrollArea doesn't consider the scollbar size, if one scrollbar is required */
class ScrollArea : public QScrollArea
{
public:
    ScrollArea( QWidget* parent = nullptr ) : QScrollArea( parent )
    {
        setWidgetResizable( true );
    }
    QSize
    sizeHint() const override
    {
        QSize size = QScrollArea::sizeHint();
        size.setHeight( size.height() + horizontalScrollBar()->sizeHint().height() );
        size.setWidth( size.width() + verticalScrollBar()->sizeHint().width() );
        return size;
    }
};

void
MainWidget::createInitialScreen( QWidget* mainWidget )
{
    const int padding = 40;

    //======== right part: choose cube input file
#if defined( ANDROID )
    QPushButton* open = new QPushButton( QApplication::style()->standardIcon( QStyle::SP_DirOpenIcon ), tr( " Open URL " ) );
    connect( open, &QPushButton::clicked, this, &MainWidget::openRemote );
#elif defined( __EMSCRIPTEN__ )  // implicitly open remote
    QPushButton* open = new QPushButton( QApplication::style()->standardIcon( QStyle::SP_DirOpenIcon ), tr( " Open Cube File " ) );
    connect( open, &QPushButton::clicked, this, &MainWidget::openRemote );
#else
    QPushButton* open = new QPushButton( QApplication::style()->standardIcon( QStyle::SP_DirOpenIcon ), tr( " Open Cube File " ) );
    connect( open, &QPushButton::clicked, this, &MainWidget::openFile );
#endif

    QFont font = open->font();
    font.setPointSize( font.pointSize() + 2 );
    open->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
    open->setFont( font );
    //======== left part: context free plugins
    contextFreeWidget = new QWidget();
    QVBoxLayout* contextL = new QVBoxLayout();
    contextL->setContentsMargins( 10, 10, 10, 10 );
    contextFreeWidget->setLayout( contextL );
    contextFreeWidget->setAutoFillBackground( true ); // don't fill with parents background
    ScrollArea* contextScroll = new ScrollArea();
    contextScroll->setWidget( contextFreeWidget );
    contextScroll->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Fixed );


    updateContextFreePluginWidget();

    // recent cube files ---------------
    QLabel* label = new QLabel( tr( "Recent Cube Files" ) );
    label->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
    ScrollArea* fileArea = new ScrollArea();
    fileArea->setWidget( recentFileWidget );
    fileArea->setAutoFillBackground( true );

    //======== right part: file browser and recent files
    QWidget*     fileWidget = new QWidget();
    QVBoxLayout* vbox       = new QVBoxLayout();
    fileWidget->setLayout( vbox );
    vbox->addSpacerItem( new QSpacerItem( 0, padding, QSizePolicy::Minimum, QSizePolicy::Expanding ) );
    vbox->addWidget( open );
    vbox->addSpacing( 10 );
    vbox->addWidget( label );
    vbox->addWidget( fileArea );
    vbox->addSpacerItem( new QSpacerItem( 0, padding, QSizePolicy::Minimum, QSizePolicy::Expanding ) );

    fileWidget->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Expanding );
    // spacing/layout
    QVBoxLayout* vboxOut   = new QVBoxLayout();
    QWidget*     widgetOut = new QWidget();
    widgetOut->setLayout( vboxOut );
    vboxOut->addSpacerItem( new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum ) );
    vboxOut->addWidget( fileWidget );
    vboxOut->addSpacerItem( new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum ) );

    // horizontal layout: use as much space for the file area as required, but not more
    QHBoxLayout* horizontal = new QHBoxLayout();
    horizontal->addSpacerItem( new QSpacerItem( padding, padding, QSizePolicy::Expanding, QSizePolicy::Preferred ) );
    horizontal->addWidget( contextScroll, 0 );
    horizontal->addSpacerItem( new QSpacerItem( padding, padding, QSizePolicy::Expanding, QSizePolicy::Preferred ) );
    horizontal->addWidget( fileWidget, 0 );
    horizontal->addSpacerItem( new QSpacerItem( padding, padding, QSizePolicy::Expanding, QSizePolicy::Preferred ) );

    // main layout
    QVBoxLayout* main = new QVBoxLayout();
    mainWidget->setLayout( main );
    main->addSpacerItem( new QSpacerItem( 0, padding, QSizePolicy::Preferred, QSizePolicy::Preferred ) );
    main->addLayout( horizontal );
    main->addSpacerItem( new QSpacerItem( 0, padding, QSizePolicy::Preferred, QSizePolicy::Preferred ) );

    // zooming: save font size for the app and for widgets that aren't deleted after cube is closed
    for ( QWidget* widget : QApplication::allWidgets() )
    {
        QFont font = widget->font();
        int   size = font.pixelSize() > 0 ? font.pixelSize() : font.pointSize();
        initialFontSizesMain[ widget ] = size;
    }
    font                         = qApp->font();
    initialFontSizesMain[ qApp ] = font.pixelSize() > 0 ? font.pixelSize() : font.pointSize();

    connect( StyleSheetEditor::getInstance(), &StyleSheetEditor::styleSheetChanged, this, [ this ](){
        // initialFontSizes.clear();
        zoom();
    } );

    connect( PluginManager::getInstance(), &PluginManager::contextFreePluginStarted,
             this, &MainWidget::contextFreePluginStarted );
}

#ifdef ENSCRIPTEN
// workaround for emscripten to prevent that a dialog disappears when the main window is on front
// Qt::WindowStaysOnTopHint for new dialogs doesn't work well with combo boxes
void
MainWidget::activateWindow()
{
    QMainWindow::activateWindow();
    this->lower();
}
#endif

void
MainWidget::contextFreePluginStarted( bool started )
{
    if ( started )
    {
        setContext( MainWidget::CONTEXT_FREE );
        closeAct->setEnabled( true );
    }
    else
    {
        setContext( MainWidget::CONTEXT_INIT );
        closeAct->setEnabled( false );
    }
}

void
MainWidget::updateContextFreePluginWidget()
{
    qDeleteAll( contextFreeWidget->findChildren<QWidget*>( QString(), Qt::FindDirectChildrenOnly ) );
    QList<QAction*> actions  = PluginManager::getInstance()->getContextFreeActions();
    QLayout*        contextL = contextFreeWidget->layout();
    if ( actions.size() > 0 )
    {
        contextL->addWidget( new QLabel( tr( "Open context free plugin:" ) ) );

        foreach( QAction * action, actions )
        {
            QToolButton* button = new QToolButton();
            button->setDefaultAction( action );
            contextL->addWidget( button );
        }
    }
    contextFreeWidget->updateGeometry();
}

void
MainWidget::recentFileSelected( const QString& link )
{
    int idx = openedFiles.indexOf( link );
    if ( idx >= 0 )
    {
        QAction* action = lastFileAct.at( idx );
        action->trigger();
    }
}

void
MainWidget::recentFileHovered( const QString& url )
{
    if ( url.length() == 0 )
    {
        return;
    }
    setMessage( "", Information, false );
    initialScreen->setImage( QPixmap() );

    QString message = url;
    if ( !message.startsWith( "cube://" ) ) // local file -> check for ini-file
    {
        // read preview image from settings
        QFileInfo file( url );
        QString   iniFile = QString( cube::services::get_cube_name( file.absoluteFilePath().toStdString() ).c_str() ).append( ".ini" );
        if ( QFile( iniFile ).exists() )
        {
            QSettings experimentSettings( iniFile, QSettings::IniFormat );
            experimentSettings.beginGroup( "lastState" );
            QPixmap pixmap    = experimentSettings.value( "image" ).value<QPixmap>();
            int     locations = experimentSettings.value( "locations" ).toInt();
            experimentSettings.endGroup();

            if ( locations > 0 )
            {
                message += ",  #locations: " + QString::number( locations );
            }
            initialScreen->setImage( pixmap );
        }
    }
    setMessage( message, Information, false );
}

MainWidget::~MainWidget()
{
    delete settings; // saves global settings on deletation
    delete cube;
}

/**********************************************************/
/****************** menu **********************************/
/**********************************************************/

// creates the pull-down menu
//
void
MainWidget::createMenu()
{
    // file menu

#ifdef ANDROID
    menuBar()->setNativeMenuBar( false );
#endif
    fileMenu = menuBar()->addMenu( tr( "&File" ) );
    fileMenu->setStatusTip( tr( "Ready" ) );

#if !defined( ANDROID )
    QAction* openAct = new QAction( tr( "&Open..." ), this );
    openAct->setShortcut( tr( "Ctrl+O" ) );
    openAct->setStatusTip( tr( "Opens a Cube file" ) );
#ifdef __EMSCRIPTEN__
    connect( openAct, &QAction::triggered, this, &MainWidget::openRemote );
#else
    connect( openAct, &QAction::triggered, this, &MainWidget::openFile );
#endif
    fileMenu->addAction( openAct );
    openAct->setWhatsThis( tr( "Offers a selection dialog to open a Cube file. In case of an already opened file, it will be closed before a new file gets opened. If a file got opened successfully, it gets added to the top of the recent files list (see below). If it was already in the list, it is moved to the top." ) );
#endif

#if defined( HAVE_CUBE_NETWORKING ) && !defined( __EMSCRIPTEN__ )
    QAction* openremoteAct = new QAction( tr( "&Open URL..." ), this );
    openremoteAct->setShortcut( tr( "Shift+Ctrl+O" ) );
    openremoteAct->setStatusTip( tr( "Opens a file referenced as URL." ) );
    connect( openremoteAct, &QAction::triggered, this, &MainWidget::openRemote );
    fileMenu->addAction( openremoteAct );
    openremoteAct->setWhatsThis( tr( "Offers an input field for a URL to open a remote file." ) );
#endif

    contextFreeMenu = PluginManager::getInstance()->getContextFreePluginMenu();
    fileMenu->addMenu( contextFreeMenu );

    saveAsAct = new QAction( tr( "&Save as..." ), this );
    saveAsAct->setShortcut( tr( "Ctrl+S" ) );
    saveAsAct->setStatusTip( tr( "Save the current cube under another name..." ) );
    saveAsAct->setEnabled( false );
    connect( saveAsAct, &QAction::triggered, this, &MainWidget::saveAs );
    fileMenu->addAction( saveAsAct );
    saveAsAct->setWhatsThis( tr( "Offers a selection dialog to select a new name for a Cube file." ) );


    closeAct = new QAction( tr( "&Close" ), this );
    closeAct->setShortcut( tr( "Ctrl+W" ) );
    closeAct->setStatusTip( tr( "Closes the file" ) );
    closeAct->setEnabled( false );
    connect( closeAct, &QAction::triggered, this, &MainWidget::closeFile );
    fileMenu->addAction( closeAct );
    closeAct->setWhatsThis( tr( "Closes the currently opened Cube file. Disabled if no file is opened." ) );

    openExtAct = new QAction( tr( "Open &external..." ), this );
    openExtAct->setStatusTip( tr( "Opens a second cube file for external percentage" ) );
    connect( openExtAct, &QAction::triggered, this, &MainWidget::openExternalFile );
    fileMenu->addAction( openExtAct );
    openExtAct->setWhatsThis( tr( "Opens a file for the external percentage value mode (see further help on the value modes)." ) );
    openExtAct->setEnabled( false );

    QAction* closeExtAct = new QAction( tr( "Close e&xternal" ), this );
    closeExtAct->setStatusTip( tr( "Closes the second cube file for external percentage" ) );
    closeExtAct->setEnabled( false );
    connect( closeExtAct, &QAction::triggered, this, &MainWidget::closeExternalFile );
    connect( this, &MainWidget::enableExtClose, closeExtAct, &QAction::setEnabled );
    connect( this, &MainWidget::enableExtClose, closeExtAct, &QAction::setEnabled );
    fileMenu->addAction( closeExtAct );
    closeExtAct->setWhatsThis( tr( "Closes the current external file and removes all corresponding data. Disabled if no external file is opened." ) );

    fileMenu->addMenu( settings->getMenu() );
    fileMenu->addSeparator();

    QAction* screenshotAct = new QAction( tr( "Sc&reenshot..." ), this );
    screenshotAct->setStatusTip( tr( "Saves a screenshot into the file cube.png" ) );
    connect( screenshotAct, &QAction::triggered, this, &MainWidget::screenshot );
    fileMenu->addAction( screenshotAct );
    screenshotAct->setWhatsThis( tr( "The function offers you to save a screenshot in a png file. Unfortunately the outer frame of the main window is not saved, only the application itself." ) );

    fileMenu->addSeparator();

    QAction* quitAct = new QAction( tr( "&Quit" ), this );
    quitAct->setShortcut( tr( "Ctrl+Q" ) );
    quitAct->setStatusTip( tr( "Exits the application" ) );
    connect( quitAct, &QAction::triggered, this, &MainWidget::closeApplication );
    fileMenu->addAction( quitAct );
    quitAct->setWhatsThis( tr( "Closes the application." ) );

    fileMenu->addSeparator();

    // display menu

    QMenu* displayMenu = menuBar()->addMenu( tr( "&Display" ) );
    displayMenu->setStatusTip( tr( "Ready" ) );

    splitterOrderAct = new QAction( tr( "D&imension order..." ), this );
    splitterOrderAct->setStatusTip( tr( "Sets the order of the dimensions metric, call chain, and system." ) );
    connect( splitterOrderAct, &QAction::triggered, this, &MainWidget::setDimensionOrder );
    displayMenu->addAction( splitterOrderAct );
    splitterOrderAct->setWhatsThis( tr( "As explained above, Cube has three resizable panes. Initially the metric pane is on the left, the call pane is in the middle, and the system pane is on the right-hand-side. However, sometimes you may be interested in other orders, and that is what this menu item is about. It offers all possible pane orderings.  For example, assume you would like to see the metric and call values for a certain thread.  In this case, you should place the system pane on the left, the metric pane in the middle, and the call pane on the right. Note that in panes left-hand-side of the metric pane we have no meaningful values, since they miss a reference metric; in this case we specify the values to be undefined, denoted by a \"-\" sign." ) );

    displayMenu->addSeparator();
    mapMenu = displayMenu->addMenu( tr( "Choose colormap" ) );
    mapMenu->setStatusTip( tr( "Choose a colormap" ) );
    connect( mapMenu, &QMenu::aboutToShow, this, &MainWidget::updateColormapMenu );
    mapMenu->setEnabled( false );

    colorsAct = new QAction( tr( "Edit colormap..." ), this );
    colorsAct->setStatusTip( tr( "Edit the selected colormap" ) );
    connect( colorsAct, &QAction::triggered, this, &MainWidget::editColorMap );
    displayMenu->addAction( colorsAct );
    colorsAct->setWhatsThis( tr( "Opens a dialog that allows the user to edit the selected colormap" ) );
    colorsAct->setEnabled( false );
    displayMenu->addSeparator();

    QAction* fontAct = new QAction( tr( "Set font size..." ), this );
    connect( fontAct, &QAction::triggered, this, &MainWidget::fontDialog );
    displayMenu->addAction( fontAct );
    fontAct->setWhatsThis( tr( "" ) );

    QAction* styleSheetAct = new QAction( tr( "Customize style sheets..." ), this );
    connect( styleSheetAct, &QAction::triggered, StyleSheetEditor::getInstance(), &StyleSheetEditor::configureStyleSheet );
    displayMenu->addAction( styleSheetAct );
    styleSheetAct->setWhatsThis( tr( "Opens a dialog that allows the user to configure the appearance of the controls" ) );

    valueViewAct = new QAction( tr( "Configure value view..." ), this );
    connect( valueViewAct, &QAction::triggered, this, &MainWidget::configureValueView );
    displayMenu->addAction( valueViewAct );
    valueViewAct->setWhatsThis( tr( "Opens a dialog that allows the user to configure the value view of the tree items" ) );
    valueViewAct->setEnabled( false );

    QAction* precisionAct = new QAction( tr( "&Precision..." ), this );
    precisionAct->setStatusTip( tr( "Defines the precision of the display for numbers." ) );
    connect( precisionAct, &QAction::triggered, this, &MainWidget::setPrecision );
    precisionAct->setWhatsThis( tr( "Activating this menu item opens a dialog for precision settings. See also help on the dialog itself." ) );
    displayMenu->addAction( precisionAct );

    treeMenu = TreeConfig::getInstance()->getMenu();
    displayMenu->addMenu( treeMenu );

    QMenu* widthMenu = displayMenu->addMenu( tr( "Optimize width" ) );
    widthMenu->setStatusTip( tr( "Ready" ) );
    widthMenu->setWhatsThis( tr( "Under this menu item Cube offers widget rescaling such that the amount of information shown is maximized, i.e., Cube optimally distributes the available space between its components. You can chose if you would like to stick to the current main window size, or if you allow to resize it." ) );

    QAction* width1Act = new QAction( tr( "&Keep main window size" ), this );
    width1Act->setStatusTip( tr( "Optimize widths of tabs in order to show equally large percentages." ) );
    connect( width1Act, &QAction::triggered, this, &MainWidget::distributeWidth );
    width1Act->setWhatsThis( tr( "Under this menu item Cube offers widget rescaling with keeping the main window size, such that the amount of information shown is maximized, i.e., Cube optimally distributes the available space between its components." ) );
    widthMenu->addAction( width1Act );
    displayMenu->addSeparator();

    QAction* width2Act = new QAction( tr( "&Adapt main window size" ), this );
    width2Act->setStatusTip( tr( "Resizes the main window width and adapt tab widths in order to show all information" ) );
    connect( width2Act, &QAction::triggered, this, &MainWidget::adaptWidth );
    width2Act->setWhatsThis( tr( "Under this menu item Cube offers widget rescaling possibly changing the main window size, such that the amount of information shown is maximized, i.e., Cube optimally distributes the available space between its components." ) );
    widthMenu->addAction( width2Act );

    displayMenu->addSeparator();
    presentationAction = new QAction( tr( "Enable presentation mode" ), this );
    presentationAction->setCheckable( true );
    connect( presentationAction, &QAction::toggled, this, &MainWidget::setPresentationMode );
    displayMenu->addAction( presentationAction );

    emulateMouseAction = new QAction( tr( "Emulate right mouse button" ), this );
    width2Act->setStatusTip( tr( "A long press with the left mouse button or a long touch is interpreted as right mouse button click." ) );
    emulateMouseAction->setCheckable( true );
    connect( emulateMouseAction, &QAction::toggled, this, &MainWidget::emulateRightMouse );
    displayMenu->addAction( emulateMouseAction );
    displayMenu->addSeparator();

    syncAction = new QAction( tr( "Show synchronization toolbar" ), this );
    syncAction->setCheckable( true );
    connect( syncAction, &QAction::toggled, syncToolBar, &SynchronizationToolBar::setVisible );
    displayMenu->addAction( syncAction );

    displayMenu->addAction( settings->getBookmarkToolbarAction() );

#ifdef WITH_WEB_ENGINE
    webengine = new QAction( tr( "Enable QWebEngine" ), this );
    webengine->setStatusTip( tr( "QWebEngine is used to show the documation, but it requires OpenGL which may cause problems on some systems. Use this option to disable QWebEngineWidgets and show the documentation in a basic layout." ) );
    webengine->setCheckable( true );
    connect( webengine, &QAction::toggled, this, &MainWidget::enableWebEngine );
    displayMenu->addAction( webengine );
#endif

    QMenu* pluginMenu = PluginManager::getInstance()->getPluginMenu();
    menuBar()->addMenu( pluginMenu );

    // help menu

    QMenu* helpMenu = menuBar()->addMenu( tr( "&Help" ) );
    helpMenu->setStatusTip( tr( "Ready" ) );

    QAction* introAct = new QAction( tr( "Getting started" ), this );
    introAct->setStatusTip( tr( "Displayes a short introduction to Cube." ) );
    connect( introAct, &QAction::triggered, this, &MainWidget::introduction );
    helpMenu->setWhatsThis( tr( "Opens a dialog with some basic information on the usage of Cube." ) );
    helpMenu->addAction( introAct );

    QAction* manual = new QAction( tr( "User guide" ), this );
    manual->setStatusTip( tr( "Shows Cube user guide" ) );
    connect( manual, &QAction::triggered, this, &MainWidget::showGuide );
    manual->setWhatsThis( tr( "Shows Cube GUI user guide in the help browser." ) );
    helpMenu->addAction( manual );

#ifndef __EMSCRIPTEN__
    manual = new QAction( tr( "Show User guide in external browser" ), this );
    connect( manual, &QAction::triggered, this, &MainWidget::showGuideExternal );
    manual->setWhatsThis( tr( "Shows Cube GUI user guide in the default web browser." ) );
    helpMenu->addAction( manual );
#endif

    QAction* keysAct = new QAction( tr( "Mouse and keyboard control" ), this );
    keysAct->setStatusTip( tr( "Shows all supported mouse and keyboard controls" ) );
    connect( keysAct, &QAction::triggered, this, &MainWidget::keyHelp );
    keysAct->setWhatsThis( tr( "List all control possibilities for keyboard and mouse." ) );
    helpMenu->addAction( keysAct );

    QAction* whatsThisAct = QWhatsThis::createAction();
    whatsThisAct->setStatusTip( tr( "Change into help mode for display components" ) );
    whatsThisAct->setWhatsThis( tr( "Here you can get more specific information on parts of the Cube GUI. If you activate this menu item, you switch to the \"What's this?\" mode. If you now click on a widget an appropriate help text is shown. The mode is left when help is given or when you press Esc.\n\nAnother way to ask the question is to move the focus to the relevant widget and press Shift+F1." ) );
    helpMenu->addAction( whatsThisAct );

    QAction* aboutAct = new QAction( tr( "About" ), this );
    aboutAct->setStatusTip( tr( "Shows Cube's about box" ) );
    connect( aboutAct, &QAction::triggered, this, &MainWidget::about );
    aboutAct->setWhatsThis( tr( "Opens a dialog with some release information." ) );
    helpMenu->addAction( aboutAct );

    helpMenu->addSeparator();

    helpMenu->addMenu( PluginManager::getInstance()->getPluginInfoMenu() );
    helpMenu->addMenu( PluginManager::getInstance()->getPluginHelpMenu() );
    helpMenu->addSeparator();

    performanceInfoAct = new QAction( tr( "Selected metrics description" ), this );
    performanceInfoAct->setStatusTip( tr( "Shows the online description of the selected metrics." ) );
    performanceInfoAct->setWhatsThis( tr( "Shows some (usually more extensive) online description for the selected metrics. For metrics it might point to an online documentation explaining their semantics." ) );
    helpMenu->addAction( performanceInfoAct );
    performanceInfoAct->setEnabled( false );

    regionInfoAct = new QAction( tr( "Selected regions description" ), this );
    regionInfoAct->setStatusTip( tr( "Shows the online description of the selected regions in a  program." ) );
    regionInfoAct->setWhatsThis( tr( "Shows some (usually more extensive) online description for the clicked region in program. For regions representing library functions it might point to the corresponding library documentation." ) );
    helpMenu->addAction( regionInfoAct );
    regionInfoAct->setEnabled( false );

    // will be started for network connections to automatically reconnect
    connectionMonitorTimer.setInterval( 5000 ); // interval in ms
    connect( &connectionMonitorTimer, &QTimer::timeout,
             this, &MainWidget::monitorConnection );
}

void
MainWidget::fontDialog()
{
    bool    ok;
    QString text = tr( "You may also use Ctrl and Mouse wheel or Ctrl and +/- <br>to increase/decrease font size.<br><br>Font size:" );

    int initialFontSize = initialFontSizesMain[ qApp ] + zoomSteps;
    int size            = QInputDialog::getInt( this, tr( "Set font size" ), text,
                                                initialFontSize, minFontSize, maxFontSize, 1, &ok );
    if ( ok )
    {
        zoomSteps += size - initialFontSize;
        zoom();
    }
}

void
MainWidget::introduction()
{
    QString url = Globals::getOption( DocPath ) + "userguide.html#intro";
    HelpBrowser::getInstance()->showUrl( url, tr( "Cube User Guide is not installed" ) );
}

void
MainWidget::keyHelp()
{
    QString url = Globals::getOption( DocPath ) + "keyboardcontrol.html";
    HelpBrowser::getInstance()->showUrl( url, tr( "Cube User Guide is not installed" ) );
}

void
MainWidget::showGuide()
{
    QString url = Globals::getOption( DocPath ) + "index.html";
    HelpBrowser::getInstance()->showUrl( url, tr( "Cube User Guide is not installed" ) );
}

void
MainWidget::showGuideExternal()
{
    QString url = Globals::getOption( DocPath ) + "index.html";
    QDesktopServices::openUrl( QUrl( url ) );
}

/*************************************************************/
/************ widget title, status bar ***********************/
/*************************************************************/

// update the title of the application to show infos to the loaded file and external file
//
void
MainWidget::updateWidgetTitle()
{
    // CUBE_NAME is defined in constants.h
    QString widgetTitle( CUBEGUI_FULL_NAME );

    // append the name of the currently opened file
    if ( cube != NULL && fileLoaded )
    {
        QUrl      fileUrl       = QUrl( cube->getAttribute( "cubename" ).c_str() );
        QFileInfo openFileName  = QFileInfo( fileUrl.path() );
        QString   fileName      = openFileName.fileName();
        QString   lastDirectory = openFileName.dir().dirName();

        widgetTitle.append( ": " );
        widgetTitle.append( lastDirectory );
        widgetTitle.append( QDir::separator() );
        widgetTitle.append( fileName );
    }

    // append the name of the current external file
    QString externalName = lastExternalFileName;
    if ( externalName != "" )
    {
        QFileInfo externalFile  =  QFileInfo( externalName );
        QString   fileName      = externalFile.fileName();
        QString   lastDirectory = externalFile.dir().dirName();

        widgetTitle.append( tr( " external: " ) );
        widgetTitle.append( lastDirectory );
        widgetTitle.append( QDir::separator() );
        widgetTitle.append( fileName );
    }

    setWindowTitle( widgetTitle );
}

// sets the status bar (can be called from any thread)
//
void
MainWidget::setMessage( const QString& message, MessageType type, bool isLogged )
{
    if ( QThread::currentThread() == this->thread() )
    {
        setMessage_( message, type, isLogged );
    }
    else
    {
        QTimer::singleShot( 0, this, [ = ]() {
            setMessage_( message, type, isLogged );
        } );
    }
}

void
MainWidget::setMessage_( const QString& message, MessageType type, bool isLogged )
{
    QString msg( message );
    if ( message.isEmpty() && cube )
    {
        msg = QUrl( cube->get_cubename().c_str() ).path();
        msg = msg.startsWith( "//" ) ? msg.mid( 1 ) : msg;
    }
    statusBar->addLine( msg, type, isLogged );
    statusBar->repaint(); // use repaint to show current status immediately
}


/*************************************************************/
/******************** file handling **************************/
/*************************************************************/


// remember the names of the last FILE_HISTORY_COUNT files that were opened;
// they can be re-opened via the display menu
//
void
MainWidget::rememberFileName( QString fileName )
{
    cube::Url fileUrl( fileName.toStdString() );
    QString   savedName = fileName;

    if ( fileUrl.getProtocol() == "file" )
    {
        savedName = QString( fileUrl.getPath().c_str() );
    }

#ifdef obsolete
    // Everything but 'file' is handled as generic URL.
    if ( fileUrl.getProtocol() != "file" )
    {
        openedUrls.prepend( fileName );
        openedUrls.removeDuplicates();

        return;
    }
#endif

    // clear the QActions from the display menu to re-open files
    for ( int i = 0; i < ( int )lastFileAct.size(); i++ )
    {
        fileMenu->removeAction( lastFileAct.at( i ) );
    }
    lastFileAct.clear();

    // insert the new fileName at the beginning of the openedFiles vector
    // and remove eventual duplicates
    openedFiles.push_back( QString( "" ) );
    for ( int i = openedFiles.size() - 1; i > 0; i-- )
    {
        openedFiles[ i ] = openedFiles[ i - 1 ];
    }
    openedFiles[ 0 ] = savedName;
    for ( int i = 1; i < ( int )openedFiles.size(); i++ )
    {
        if ( openedFiles[ i ].compare( openedFiles[ 0 ] ) == 0 )
        {
            for ( int j = i; j < ( int )openedFiles.size() - 1; j++ )
            {
                openedFiles[ j ] = openedFiles[ j + 1 ];
            }
            openedFiles.pop_back();
            break;
        }
    }

    // we remember at most FILE_HISTORY_COUNT file names
    if ( openedFiles.size() > FILE_HISTORY_COUNT )
    {
        openedFiles.pop_back();
    }

    for ( int i = 0; i < openedFiles.size(); i++ )
    {
        QString   filename = openedFiles.at( i );
        QFileInfo file( filename );
        QString   lastDir     = file.absolutePath().split( '/' ).last();
        QString   displayName = lastDir + QDir::separator() + file.fileName();

        cube::Url fileUrl( filename.toStdString() );
        if ( fileUrl.getProtocol() == "cube" )
        {
            displayName = QString( fileUrl.getProtocol().c_str() ) + "://.../" + displayName;
        }

        QAction* act = new QAction( displayName, this );
        if ( i < 5 )
        {
            fileMenu->addAction( act );
        }
        act->setData( filename );
        connect( act, &QAction::triggered, this, &MainWidget::openLastFiles );

        act->setStatusTip( filename );
        act->setWhatsThis( tr( "At the bottom of the File menu the last opened files are offered for re-opening, the top-most being the most recently opened one. A full path to the file is visible in the status bar if you move the mouse above one of the recent file items in the menu." ) );
        lastFileAct.push_back( act );
    }

    if ( recentFileWidget ) // update initial screen contents and size
    {
        QString text;
        text += "<p style=\"line-height:1.1\">";
        for ( unsigned i = 0; i < lastFileAct.size(); i++ )
        {
            QFileInfo file( openedFiles[ i ] );
            QString   lastDir = file.absolutePath().split( '/' ).last();
            QString   name    = lastDir + QDir::separator() + file.fileName();

            cube::Url fileUrl( openedFiles[ i ].toStdString() );
            if ( fileUrl.getProtocol() == "cube" )
            {
                name = QString( fileUrl.getProtocol().c_str() ) + "://.../" + name;
            }
            text += "<a href=\"" + openedFiles[ i ] + "\">" + name + "</a>";
            text += ( i < lastFileAct.size() - 1 ) ? "<br>" : "";
        }
        recentFileWidget->setText( text );

        QPalette palette = recentFileWidget->palette();
        palette.setColor( QPalette::Base, palette.window().color() );
        recentFileWidget->setPalette( palette );
    }
}

// end of rememberFileName(...)

// private slot connected to a file menu item which shows one of the recently opened files
void
MainWidget::openLastFiles()
{
    QAction* action = qobject_cast<QAction*>( sender() );
    if ( action != 0 )
    {
        QString filename = action->data().toString();
        loadFile( filename );
    }
}

// input a file name to open and open it via loadFile(...)
//
void
MainWidget::saveAs()
{
    QString saveAsFileName = QFileDialog::getSaveFileName( this,
                                                           tr( "Choose a file to open" ),
                                                           "",
                                                           tr( "Cube4 files (*.cubex);;All files (*.*);;All files (*)" ) );

    if ( saveAsFileName.length() == 0 )
    {
        setMessage();
        return;
    }
    // here is the saving name selecetd

    setMessage( tr( "Saving " ).append( saveAsFileName ).append( "..." ) );
    setCursor( Qt::WaitCursor );
    writeCube( saveAsFileName );

    unsetCursor();
    setMessage();
}


#ifndef ANDROID
// input a file name to open and open it via loadFile(...)
//
void
MainWidget::openFile()
{
    QString lastName     = ( openedFiles.size() == 0 ? "." : openedFiles[ 0 ] );
    QString openFileName = QFileDialog::getOpenFileName( this,
                                                         tr( "Choose a file to open" ),
                                                         lastName,
                                                         "Cube3/4 files (*cube *cube.gz *.cubex);;Cube4 files (*.cubex);;Cube3 files (*.cube.gz *.cube);;All files (*.*);;All files (*)" );

    if ( openFileName.length() == 0 )
    {
        setMessage();
        return;
    }
    setMessage( tr( "Open " ).append( openFileName ).append( "..." ) );
    loadFile( openFileName );
}


#endif

//input a URL to open a remote file and... //TODO
//
void
MainWidget::openRemote()
{
    closeFile(); // close connection before opening remote file dialog
    QString openRemoteName = RemoteFileDialog::getFileUrl( "" );

    //QThread::msleep( 2000 );
    if ( !openRemoteName.isEmpty() )
    {
        setMessage( tr( "Open " ).append( openRemoteName ).append( "..." ) );

        loadFile( openRemoteName );
    }
}

#ifdef ANDROID
#if QT_VERSION >= QT_VERSION_CHECK( 6, 2, 0 )

#include <QtCore/private/qandroidextras_p.h>

bool
checkPermission()
{
    const QVector<QString> permissions( {
        "android.permission.INTERNET"
        "android.permission.WRITE_EXTERNAL_STORAGE",
        "android.permission.READ_EXTERNAL_STORAGE",
        "android.permission.ACCESS_NETWORK_STATE",
        "android.permission.CHANGE_NETWORK_STATE"
    } );

    for ( const QString& permission : permissions )
    {
        auto r = QtAndroidPrivate::checkPermission( QtAndroidPrivate::Storage ).result();
        if ( r == QtAndroidPrivate::Denied )
        {
            r = QtAndroidPrivate::requestPermission( QtAndroidPrivate::Storage ).result();
            if ( r == QtAndroidPrivate::Denied )
            {
                return false;
            }
        }
    }
    return true;
}



#else // old qt >6.2
QPixmap pixmap = style()->standardPixmap( QStyle::SP_TrashIcon );
QIcon   icon( pixmap );
static bool
__requestAndroidPermissions()
{
//Request requiered permissions at runtime

    const QVector<QString> permissions( {
        "android.permission.INTERNET"
        "android.permission.WRITE_EXTERNAL_STORAGE",
        "android.permission.READ_EXTERNAL_STORAGE",
        "android.permission.ACCESS_NETWORK_STATE",
        "android.permission.CHANGE_NETWORK_STATE"
    } );

    for ( const QString& permission : permissions )
    {
        auto result = QtAndroid::checkPermission( permission );
        if ( result == QtAndroid::PermissionResult::Denied )
        {
            auto resultHash = QtAndroid::requestPermissionsSync( QStringList( { permission } ) );
            if ( resultHash[ permission ] == QtAndroid::PermissionResult::Denied )
            {
                return false;
            }
        }
    }
    return true;
}

#endif
#endif

// load the contents of a cube file into a cube object
// and display the contents
//
void
MainWidget::loadFile( const QString fileName )
{
    qDebug() << "+++++++++++++ loadFile " << fileName << " +++++++++++++";
    closeFile();

    // ensure that context free connection is closed before a new connection is created
    ContextFreeServices::getInstance()->disconnect();

    fileLoaded = false;
    stackedWidget->setCurrentIndex( CONTEXT_EMPTY );
    assert( cube == NULL );

    timer = new QTimer();
    timer->start( 300 );
    connect( timer, &QTimer::timeout, this, &MainWidget::updateProgress );

    // create proxy
    Task* task = new Task([ this, fileName ] { readCubeTask( fileName );
                          } );
    connect( &future, &Future::calculationFinished, this, &MainWidget::cubeReportLoaded );
    future.addCalculation( task );
    future.startCalculation();
}
void
MainWidget::readCubeTask( const QString& filename )
{
    futureError.clear();
    CubeProxy* cube = nullptr;
    try
    {
        cube = cube::CubeProxy::create( filename.toStdString() );
        if ( !cube )
        {
            futureError = "Cube file couldn't be opened";
            return;
        }
        cube->openReport();
    }
    catch ( const std::exception& e  )
    {
        futureError = QString( tr( "Error. \n" ) ).append( QString( e.what() ) );
    }
    this->cube = cube;
}

void
MainWidget::updateProgress()
{
    if ( fileLoaded || cube == NULL )
    {
        return;
    }
    const cube::ProgressStatus& status  = cube->getOperationProgressStatus();
    int                         percent = ( int )lround( status.status * 100 );
    QString                     message = QString( status.message.c_str() );

    message += " (" + QString::number( percent ) + "%)";
    //qDebug() << "current progress " << percent << message;
    Globals::setStatusMessage( message, Information, false );
}

void
MainWidget::writeCube( const QString& filename ) const
{
    CubeWriter( cube,
                tabManager->getTree( METRICTREE ),
                tabManager->getTree( DEFAULTCALLTREE ),
                tabManager->getTree( TASKTREE ),
                tabManager->getTree( SYSTEMTREE ) ).writeCube( filename );
}

void
MainWidget::cubeReportLoaded()
{
    disconnect( &future, &Future::calculationFinished, this, &MainWidget::cubeReportLoaded );

    timer->stop();
    timer->deleteLater();
    unsetCursor();
    if ( !futureError.isEmpty() )
    {
        fileLoaded = false;
        setMessage( tr( "Error during opening cube file: " ) + QString( futureError ), Error );
        deleteCube();
        stackedWidget->setCurrentIndex( CONTEXT_INIT );
    }
    else
    {
        fileLoaded = true;
        setMessage( tr( "Finished parsing..." ) );
        setMessage( tr( "Creating GUI..." ) );

        try
        {
            openCube( cube, cube->getUrl().toString().c_str() );
            // remember the file name for opening recent files
            rememberFileName( cube->getUrl().toString().c_str() );
            settings->saveGlobalStartupSettings(); // save filename

            updateWidgetTitle();
            stackedWidget->setCurrentIndex( CONTEXT_CUBE );
            if ( !initialSystemTab.isNull() )
            {
                Globals::getTabManager()->getTab( SYSTEM )->toFront( initialSystemTab );
            }

            // monitor the etablished connection
            cube::CubeNetworkProxy* cubeNet = dynamic_cast< cube::CubeNetworkProxy* >( cube );
            if ( cubeNet )
            {
                connectionMonitorTimer.start();
            }
        }
        catch ( const exception& e )
        {
            setMessage( tr( "Error opening cube file: " ) + QString( e.what() ), Error );
            deleteCube();
            setContext( CONTEXT_INIT );
        }
    }
}

void
MainWidget::openCube( cube::CubeProxy* cube, const QString& fileName )
{
    // reset default font to create new widgets without zoom to collect valid initial font sizes
    QFont font = qApp->font();
    font.setPointSize( initialFontSizesMain[ qApp ] );
    qApp->setFont( font );

    setMessage( tr( "Creating display..." ) );

    stackedWidget->setVisible( false );

    PluginManager::getInstance()->closeContextFreePlugin();
    setContext( CONTEXT_CUBE ); // switch to cube context after context free plugins have been closed

    this->cube = cube;

    // initialize the tree views in their display order
    setMessage( tr( "Creating trees..." ) );

    tabManager->cubeOpened( cube );
    tabManager->show();
    foreach( TabWidget * widget, tabManager->getTabWidgets() )
    {
        connect( widget, &TabWidget::externalValueModusSelected,
                 this,   &MainWidget::selectExternalValueMode );
    }

    setMessage( tr( "Computing values..." ) );

    closeAct->setEnabled( true );
    contextFreeMenu->setEnabled( false );

    setMessage( tr( "Opening plugins..." ) );
    // open all plugins, in a final step all plugin tabs will be added
    QApplication::setOverrideCursor( Qt::WaitCursor );
    PluginManager::getInstance()->opened( cube, fileName, tabManager );
    QApplication::restoreOverrideCursor();
    setMessage( tr( "Opening plugins done" ), Information, false );

    setMessage( tr( "Loading Settings..." ) );
    // load experiment specific settings
    setUpdatesEnabled( false );
    settings->cubeLoaded( fileName );
    setUpdatesEnabled( true );

    setMessage( tr( "Calculating tree values..." ) );

    // activate loaded experiment specific settings or set initial status
    setMessage( tr( "Initializing menu..." ) );
    tabManager->initialize();
    // enable menu items
    colorsAct->setEnabled( true );
    treeMenu->setEnabled( true );
    mapMenu->setEnabled( true );
    colorScale->setVisible( true );
    saveAsAct->setEnabled( true );
    performanceInfoAct->setEnabled( true );
    regionInfoAct->setEnabled( true );
    valueViewAct->setEnabled( true );
    openExtAct->setEnabled( true );

    // connect help menu items to slots of the current views
    connect( performanceInfoAct, &QAction::triggered,
             tabManager->getActiveView( METRIC ), &TreeView::onInfo );
    connect( regionInfoAct, &QAction::triggered,
             tabManager->getActiveView( CALL ), &TreeView::onInfo );

    // register for synchronization
    QList<SettingsHandler*> handlerList;

    // notify the synchronization toolbar about all changes in the tree view
    for ( TreeView* view : Globals::getTabManager()->getViews() )
    {
        connect( view, &TreeView::selectedItemsChanged, syncToolBar, &SynchronizationToolBar::send );
        connect( view, &TreeView::itemExpanded,     syncToolBar, &SynchronizationToolBar::send );
        handlerList.append( view );
    }
    syncToolBar->setSettingsHandlerList( handlerList );

    zoom();

    stackedWidget->setVisible( true );
}

void
MainWidget::monitorConnection()
{
#ifdef __EMSCRIPTEN__
    cube::CubeNetworkProxy* cubeNet = dynamic_cast< cube::CubeNetworkProxy* >( cube );
    bool                    stopped = cubeNet && cubeNet->getConnection()->isStopped();
    if ( stopped )   // reconnect if the connection is broken
    {
        QDateTime current = QDateTime::currentDateTime();
        QString   time    = current.toString( "dd.MM hh:mm:ss" );
        qInfo() << "connection stopped at " << time;

        Task* task = new Task([ this ] {
            cube::CubeNetworkProxy* cubeNet = dynamic_cast< cube::CubeNetworkProxy* >( cube );
            assert( cubeNet );
            cubeNet->reconnect();
            setMessage( "Reconnected" );
        } );
        future.addCalculation( task );
        future.startCalculation();
        setMessage( tr( "Connection has been closed. Trying to reconnect..." ), Warning );
    }
#endif
}

void
MainWidget::deleteCube()
{
    if ( cube != nullptr )
    {
        // deleting network based cube proxy should be done in a separate thread and is required for WASM
        CubeProxy*  toDelete = cube;
        std::thread thr([ toDelete, this ]() {
            try
            {
                toDelete->closeReport();
            }
            catch ( const exception& e )
            {
                cerr << "error closing report: " << e.what() << endl;
            }
            delete toDelete;
        } );
        thr.detach();
    }
    cube = nullptr;
}

// remove loaded data
//
void
MainWidget::closeFile()
{
    // stop monitoring connection
    connectionMonitorTimer.stop();

    if ( cube == 0 )
    {
        PluginManager::getInstance()->closeContextFreePlugin();
        return;              // already closed
    }
    setMessage( tr( "Closing previous Cube report..." ) );

    fileLoaded = false;
    syncToolBar->clearSettingsHandler();

    settings->cubeClosed();
    lastColorMapName = Globals::getColorMap()->getMapName();

    PluginManager::getInstance()->closed();

    tabManager->cubeClosed();

    deleteCube();

    updateWidgetTitle();

    closeAct->setEnabled( false );
    contextFreeMenu->setEnabled( true );

    emit cubeIsClosed();
    setMessage();

    initialFontSizes.clear();

    // disable menu items
    saveAsAct->setEnabled( false );
    colorsAct->setEnabled( false ); // disable colormap selection
    colorScale->setVisible( false );
    treeMenu->setEnabled( false );
    mapMenu->setEnabled( false );
    performanceInfoAct->setEnabled( false );
    regionInfoAct->setEnabled( false );
    valueViewAct->setEnabled( false );
    openExtAct->setEnabled( false );

    performanceInfoAct->disconnect(); // disconnect all slots
    regionInfoAct->disconnect();

    setContext( CONTEXT_INIT );
}

/**
 * This method is called, if the user selects EXTERNAL value modus. If no external cube file is loaded,
 * a file dialog is opened.
 */
void
MainWidget::selectExternalValueMode( TabWidget* widget )
{
    bool ok = false;
    if ( lastExternalFileName.size() > 0 )
    {
        ok = true;
    }
    else
    {
        ok = openExternalFile();
    }
    if ( ok )
    {
        widget->setValueModus( EXTERNAL_VALUES );
    }
    else
    {
        widget->setValueModus( ABSOLUTE_VALUES );
    }
}

// input a file name to open as external file for external percentage
// and open it via readExternalFile(...)
//
bool
MainWidget::openExternalFile()
{
    QString fileName = QFileDialog::getOpenFileName( this,
                                                     tr( "Choose a file for external percentage" ),
                                                     lastExternalFileName,
                                                     "Cube3/4 files (*cube *cube.gz *.cubex);;Cube4 files (*.cubex);;Cube3 files (*.cube.gz *.cube);;All files (*.*);;All files (*)" );

    if ( fileName.length() == 0 )
    {
        setMessage();
        return false;
    }
    bool result = readExternalFile( fileName );
    if ( result )
    {
        updateWidgetTitle();
        emit enableExtClose( true );
    }

    return result;
}


// read the contents of a cube file into a temporary cube object for
// the external percentage modus and compute the relevant values for the display
//
bool
MainWidget::readExternalFile( const QString fileName )
{
    setMessage( tr( "Parsing..." ) );
    setCursor( Qt::WaitCursor );

    // #ifndef CUBE_COMPRESSED
    //     std::ifstream in( fileName.toStdString().c_str() );
    // #else
    //     gzifstream    in( fileName.toStdString().c_str(), ios_base::in | ios_base::binary );
    // #endif

    lastExternalFileName = fileName;

    // read in external data into the external cube object
    assert( cubeExternal == NULL );
    cubeExternal = new cube::CubeIoProxy();
    try
    {
        cubeExternal->openReport( fileName.toStdString() );
        cubeExternal->setMemoryStrategy( strategy );
        // compute external reference values for metric tree
        MetricTree* metricTree = static_cast<MetricTree*> ( tabManager->getActiveTree( METRIC ) );
        metricTree->computeExternalReferenceValues( cubeExternal );
    }
    catch ( const std::exception& e )
    {
        setMessage( e.what(), Error );
    }

    // the relevant values got stores in the metric tab widget
    // and we can delete the external cube object
    delete cubeExternal;
    cubeExternal = NULL;

    unsetCursor();
    setMessage();

    return true;
}

// remove loaded external data for external percentage
//
void
MainWidget::closeExternalFile()
{
    // check if external modus is set in one of the tabs;

    foreach( TabWidget * widget, tabManager->getTabWidgets() )
    {
        if ( widget->getValueModus() == EXTERNAL_VALUES )
        {
            widget->setValueModus( ABSOLUTE_VALUES );
        }
    }

    lastExternalFileName = "";
    updateWidgetTitle();
    emit enableExtClose( false );
}


/*********************************************************/
/****************** miscellaneous ************************/
/*********************************************************/

void
MainWidget::closeApplication()
{
    close(); // closeEvent is called
}

// clean-up before close
// closes the application after eventually saving settings


// for calling scorep_finalize(), should be function-name in C++ have 'C' linkage : extern "C" void scorep_finalize(void);
DEFINE_SCOREP_FINALIZE();

void
MainWidget::closeEvent( QCloseEvent* event )
{
    // Before closing the window by choosing "Close" from the menu, or clicking the X button, scorep_finalize() should be called.
    //CALL_SCOREP_FINALIZE();

    Q_UNUSED( event );

    closeFile();
    HelpBrowser::deleteInstance();

#ifdef WEB_SOCKETS
    // to give the server some time to close the connection
    QTimer::singleShot( 500, qApp, SLOT( quit() ) );
#else
    qApp->quit();
#endif
}

// set the order of metric/call/system splitter elements
//
void
MainWidget::setDimensionOrder()
{
    DimensionOrderDialog dialog( this, tabManager->getOrder() );
    if ( dialog.exec() )
    {
        QList<DisplayType> order = dialog.getOrder();

        tabManager->setOrder( order );
    }
}

void
MainWidget::updateColormapMenu()
{
    mapMenu->clear();

    QList<ColorMap*> maps = PluginManager::getInstance()->getColorMaps();
    maps.prepend( Globals::defaultColorMap );

    QActionGroup* selectionActionGroup = new QActionGroup( this );
    selectionActionGroup->setExclusive( true );
    QString currentMapName = Globals::getColorMap()->getMapName();
    foreach( ColorMap * map, maps )
    {
        QAction* action = mapMenu->addAction( map->getMapName() );
        action->setCheckable( true );
        selectionActionGroup->addAction( action );
        action->setData( QVariant::fromValue( ( void* )map ) );
        connect( action, &QAction::triggered, this, &MainWidget::selectColorMap );
        if ( map->getMapName() == currentMapName )
        {
            action->setChecked( true );
        }
    }
}

// causes to open a color dialog to allow the user to change the colormap
void
MainWidget::editColorMap()
{
    ColorMap* map = Globals::getColorMap();
    map->showDialog();
}

// slot for color map menu item
void
MainWidget::selectColorMap()
{
    QAction*  action = qobject_cast<QAction*>( sender() );
    ColorMap* map    = static_cast<ColorMap*> ( action->data().value<void*>() );
    setColorMap( map );
}

void
MainWidget::setColorMap( ColorMap* map )
{
    ColorMap* old = Globals::getColorMap();
    disconnect( old, &ColorMap::colorMapChanged, this, &MainWidget::updateColorMap );
    connect(    map, &ColorMap::colorMapChanged, this, &MainWidget::updateColorMap );

    // remember current map name to store in settings
    lastColorMapName = map->getMapName();

    if ( old != map )
    {
        Globals::setColorMap( map );
        updateColorMap();
    }
}

// slot, called if apply is pressed in edit dialog
void
MainWidget::updateColorMap()
{
    Globals::setColorMap( Globals::getColorMap() ); // update tree colors
    if ( cube )                                     // notify plugin tabs
    {
        tabManager->updateColors();
    }
}

void
MainWidget::configureValueView()
{
    QSet<cube::DataType> usedTypes;

    Tree*                  tree     = this->tabManager->getActiveTree( METRIC );
    const QList<TreeItem*> toplevel = tree->getRootItem()->getChildren();
    foreach( TreeItem * topItem, tree->getRootItem()->getChildren() )
    {
        usedTypes.insert( topItem->getValueObject()->myDataType() );
    }

    cube::DataType   type   = tree->getLastSelection()->getValueObject()->myDataType();
    ValueViewConfig* config = new ValueViewConfig( usedTypes.values(), type, this );
    config->setVisible( true );
    connect( this, &MainWidget::cubeIsClosed, config, &ValueViewConfig::close );
}



// causes to open a precision dialog to allow the user to change precision-specific settings
//
void
MainWidget::setPrecision()
{
    PrecisionWidget* precisionWidget = Globals::getPrecisionWidget();
    precisionWidget->exec();
    setMessage();
}

// shows the about message
//
void
MainWidget::about()
{
    QDialog dialog( this );
    dialog.setWindowIcon( QIcon( ":images/CubeIcon.xpm" ) );
    dialog.setWindowTitle( QObject::tr( "About " ).append( CUBEGUI_FULL_NAME ) );

    /* add "-dev" suffix to show that the current version is not tagged. This is useful for tarball builds */
    QString dev       = QString( LIBRARY_INTERFACE_VERSION ) == "0:0:0" ? "-dev" : "";
    QString title_str = QString( "<b>This is %1</b>%2 (rev. <b>%3</b> ) (c) 2009-2024" ).arg( CUBEGUI_FULL_NAME, dev, CUBEGUI_REVISION );

    QLabel* cube_title = new QLabel( title_str );
    QLabel* qt_version = new QLabel( tr( "Built with <b>Qt v" ).append( QT_VERSION_CUBE " </b>" ) );
#ifdef CONFIG_COMMON_H // autoconf with qmake
    QLabel* qmake = new QLabel( tr( "Built with qmake: <b>" ).append(
                                    QT_PATH "</b>" ) );
#endif
    QLabel* cube_homepage = new QLabel( tr( "Home page" ).append( "        : <a href=\"http://www.scalasca.org\"> " PACKAGE_URL " </a>" ) );
    cube_homepage->setOpenExternalLinks( true );
    QLabel* cube_support = new QLabel( tr( "Technical support: " ).append( " <a href=\"mailto:scalasca@fz-juelich.de?subject=" CUBEGUI_FULL_NAME ).append( tr( " Feedback.&body=Dear Scalasca Team, \n\nwe would like to inform you, that " )  ).append( CUBEGUI_FULL_NAME " (rev. " CUBEGUI_REVISION ")" ).append( tr( "is awesome and we would like to have additional feature: a Teleport and an X-Wing fighter support. \n\n\n\n\n Sincerely,\n User of " ) ).append( CUBEGUI_FULL_NAME "\">" PACKAGE_BUGREPORT "</a>" ) );
    cube_support->setOpenExternalLinks( true );

    QLabel* versionLabel = new QLabel();
    versionLabel->setOpenExternalLinks( true );

    QLabel* fz_copyright = new QLabel( tr( "Juelich Supercomputing Centre,\nForschungszentrum Juelich GmbH" ) );
    QLabel* fz_logo      = new QLabel( "" );
    fz_logo->setPixmap( QPixmap( ":/images/fzjlogo.png" ).scaledToWidth( 200 / 1.3,  Qt::SmoothTransformation ) );
    QLabel* cube_logo = new QLabel();
    cube_logo->setPixmap( QPixmap( ":/images/CubeLogo.xpm" ).scaledToWidth( 200 / 1.3,  Qt::SmoothTransformation ) );

    QVBoxLayout* cube_labels_layout = new QVBoxLayout();
    cube_labels_layout->setContentsMargins( 0, 0, 0, 0 );
    cube_labels_layout->setSpacing( 0 );
    QGridLayout* grid = new QGridLayout();
    grid->setHorizontalSpacing( 20 );
    grid->addWidget( cube_logo, 0, 0 );
    grid->addWidget( fz_logo, 1, 0 );
    grid->addLayout( cube_labels_layout, 0, 1 );
    grid->addWidget( fz_copyright, 1, 1 );

    cube_labels_layout->addWidget( cube_title );
    cube_labels_layout->addWidget( cube_homepage );
    cube_labels_layout->addWidget( cube_support );
    cube_labels_layout->addWidget( qt_version );
#ifdef CONFIG_COMMON_H // autoconf with qmake
    cube_labels_layout->addWidget( qmake );
#endif

    QDialogButtonBox* buttonBox  = new QDialogButtonBox( &dialog );
    QPushButton*      versionAct = buttonBox->addButton( tr( "Check for updates" ), QDialogButtonBox::ApplyRole );
    QPushButton*      close      = buttonBox->addButton( QDialogButtonBox::Close );
    close->setDefault( true );

    connect( buttonBox, &QDialogButtonBox::rejected, &dialog, &QDialog::accept );
    VersionCheck* versionCheck = new VersionCheck();
    connect( versionAct, &QPushButton::clicked, versionCheck, &VersionCheck::checkForUpdates );
    connect( versionAct, &QPushButton::clicked, versionCheck, [ close ](){
        close->setFocus( Qt::OtherFocusReason );
    } );
    connect( versionCheck, &VersionCheck::updateCheckFinished, versionLabel, [ versionCheck, versionLabel ]( const QString& msg ) {
        versionLabel->setText( msg );
        versionLabel->setToolTip( versionCheck->getLatestRelease() );
    } );

    QVBoxLayout* main_layout = new QVBoxLayout();
    dialog.setLayout( main_layout );
    main_layout->addSpacing( 10 );
    main_layout->addLayout( grid );
    QHBoxLayout* line = new QHBoxLayout();
    main_layout->addLayout( line );
    line->addWidget( versionLabel, 0, Qt::AlignLeft );
    line->addWidget( buttonBox );

    dialog.exec();

    setMessage();
}

void
MainWidget::setPresentationMode( bool enable )
{
    if ( presentationAction->isChecked() != enable ) // slot has not been activated by the action -> check action to update menu item
    {
        presentationAction->setChecked( enable );
    }
    else
    {
        app.setPresentationMode( enable );
    }
}

void
MainWidget::emulateRightMouse( bool enable )
{
    if ( emulateMouseAction->isChecked() != enable ) // slot has not been activated by the action -> check action to update menu item
    {
        emulateMouseAction->setChecked( enable );
    }
    else
    {
        app.emulateRightMouseMode( enable );
    }
}

void
MainWidget::enableWebEngine( bool enable )
{
    Globals::setOption( WebEngine,  enable ? "enabled" : "" );
    HelpBrowser::deleteInstance();
    InfoWidget::deleteInstance();
}

// saves a screenshot in a png file
// (it is without the shell frame, I could not find any possibilities to include that frame within qt)
//
void
MainWidget::screenshot()
{
    QPixmap pixmap;
    pixmap = this->grab();

    QImage image = pixmap.toImage();

#ifdef __EMSCRIPTEN__
    QByteArray ba;
    QBuffer    buffer( &ba );
    buffer.open( QIODevice::WriteOnly );
    image.save( &buffer, "png" );
    buffer.close();

    QFileDialog::saveFileContent( ba, "cube.png" ); // since Qt-6.2
#else
    const char* format      = "png";
    QString     initialPath = QDir::currentPath() + tr( "/cube." ) + format;
    QString     fileName    = QFileDialog::getSaveFileName( this, tr( "Save As" ),
                                                            initialPath,
                                                            tr( "%1 Files (*.%2);;All Files (*)" )
                                                            .arg( QString( format ).toUpper() )
                                                            .arg( format ) );
    if ( !fileName.isEmpty() )
    {
        image.save( fileName, format );
    }
#endif

    setMessage();
}

/** distribute the available width between the 3 tabs such that from each scroll area the same percentual
 * amount is visible
 */
void
MainWidget::distributeWidth()
{
    // get the current sizes of the splitter's widgets
    QList<int> sizes   = tabManager->sizes();
    int        current = 0;
    foreach( int val, sizes )
    {
        current += val;
    }

    // get optimal widths for the splitter's widgets
    int preferred = 0;
    int idx       = 0;
    foreach( TabWidget * widget, tabManager->getTabWidgets() )
    {
        sizes[ idx ] = widget->sizeHint().width();
        preferred   += sizes[ idx++ ];
    }
    // distribute the available width
    for ( int i = 0; i < 3; i++ )
    {
        sizes[ i ] = ( int )( ( double )current * ( double )sizes[ i ] / ( double )preferred );
    }
    // resize
    tabManager->setSizes( sizes );
}

/** adapt the size of the main window such that all information are visible and resize the splitter widget
 *  accordingly */
void
MainWidget::adaptWidth()
{
    // get optimal widths for the splitter's widgets
    QList<int> sizes     = tabManager->sizes();
    int        current   = 0; // sum of current splitter sizes
    int        preferred = 0; // sum of preferred content size
    int        idx       = 0;
    foreach( TabWidget * widget, tabManager->getTabWidgets() )
    {
        current += sizes.at( idx );
        int preferredWidth = widget->sizeHint().width();
        sizes[ idx ] = preferredWidth;
        preferred   += preferredWidth;
        idx++;
    }

    int pad = this->width() - current; // get space without tab widget contents, e.g. splitter control, paddings
    pad += 10;
    resize( preferred + pad, height() );
    tabManager->setSizes( sizes );
}

// ------------ setting handler implementation ----------------------------------

/** saves settings that do not depend on an experiment */
void
MainWidget::saveGlobalSettings( QSettings& settings )
{
    // save ValueView settings
    for ( auto& entry : Globals::getInstance()->valueViews )
    {
        const cube::DataType&     type  = entry.first;
        cubepluginapi::ValueView* value = entry.second;
        settings.setValue( "DataType" + QString::number( type ), value->getName() );
    }

    settings.setValue( "showSyncToolBar", syncAction->isChecked() );

    TreeConfig::getInstance()->saveGlobalSettings( settings );
    tabManager->saveGlobalSettings( settings );

    // save precision values
    PrecisionWidget* precisionWidget = Globals::getPrecisionWidget();
    settings.setValue( "precision/prec0", precisionWidget->getPrecision( FORMAT_TREES ) );
    settings.setValue( "precision/round0", precisionWidget->getRoundNr( FORMAT_TREES ) );
    settings.setValue( "precision/exp0", precisionWidget->getUpperExpNr( FORMAT_TREES ) );
    settings.setValue( "precision/prec1", precisionWidget->getPrecision( FORMAT_DEFAULT ) );
    settings.setValue( "precision/round1", precisionWidget->getRoundNr( FORMAT_DEFAULT ) );
    settings.setValue( "precision/exp1", precisionWidget->getUpperExpNr( FORMAT_DEFAULT ) );

    // save external file name
    settings.setValue( "lastExternalFileName", lastExternalFileName );
    settings.setValue( "lastColorMapName", lastColorMapName );
}

void
MainWidget::loadGlobalStartupSettings( QSettings& settings )
{
    // restore window geometry only on startup or if settings "restoreGeometry" is set
    if ( !_initGeometry )
    {
        _initGeometry = true;
        // restore position and size of the main window
        resize( settings.value( "coords/size", QSize( 1200, 800 ) ).toSize() );
        move( settings.value( "coords/pos", QPoint( 50, 50 ) ).toPoint() );
    }

    // restore names of recent files
    QStringList files = settings.value( "openedFiles" ).toStringList();
    for ( int i = files.size() - 1; i >= 0; --i )
    {
        if ( files[ i ].startsWith( "cube://" ) || QFile( files[ i ] ).exists() )
        {
            rememberFileName( files[ i ] );
        }
    }
    openedUrls = settings.value( "openedURLs" ).toStringList();

    StyleSheetEditor::getInstance()->loadGlobalStartupSettings( settings );
    bool enabled = settings.value( "enableWebEngine", "true" ).toBool();
    Globals::setOption( WebEngine, enabled ? "enabled" : "" );
    zoomSteps = settings.value( "zoom", 0 ).toInt();
}

void
MainWidget::saveGlobalStartupSettings( QSettings& settings )
{
    // save position and size of the main window
    settings.setValue( "coords/size", size() );
    settings.setValue( "coords/pos", pos() );

    settings.setValue( "openedFiles", openedFiles );
    settings.setValue( "openedURLs",  openedUrls );
    StyleSheetEditor::getInstance()->saveGlobalStartupSettings( settings );
    settings.setValue( "enableWebEngine", Globals::optionIsSet( WebEngine ) );
    settings.setValue( "zoom", zoomSteps );
}

void
MainWidget::loadGlobalSettings( QSettings& settings )
{
    if ( cube )
    {
        for ( int type = 1; type < 18; type++ ) // values for enum cube::DataType
        {
            QString name = settings.value( tr( "DataType" ) + QString::number( type ), "" ).toString();
            if ( !name.isEmpty() )
            {
                ValueView* view = PluginManager::getInstance()->getValueView( name );
                if ( view )
                {
                    Globals::setValueView( ( cube::DataType )type, view );
                }
            }
        }
    }

    bool sync = settings.value( "showSyncToolBar", false ).toBool();
    syncAction->setChecked( sync );

    TreeConfig::getInstance()->loadGlobalSettings( settings );
    tabManager->loadGlobalSettings( settings );

    // restore precision values
    PrecisionWidget* precisionWidget = Globals::getPrecisionWidget();
    precisionWidget->setPrecision( settings.value( "precision/prec0", 2 ).toInt(), FORMAT_TREES );
    precisionWidget->setUpperExpNr( settings.value( "precision/exp0", 4 ).toInt(), FORMAT_TREES );
    precisionWidget->setRoundNr( settings.value( "precision/round0", 7 ).toInt(), FORMAT_TREES );
    precisionWidget->setPrecision( settings.value( "precision/prec1", 2 ).toInt(), FORMAT_DEFAULT );
    precisionWidget->setUpperExpNr( settings.value( "precision/exp1", 4 ).toInt(), FORMAT_DEFAULT );
    precisionWidget->setRoundNr( settings.value( "precision/round1", 7 ).toInt(), FORMAT_DEFAULT );

    lastColorMapName = settings.value( "lastColorMapName", "" ).toString();
    // should be safe - the call to settings is AFTER initializing tabs
    loadColorMap();
}

void
MainWidget::loadStatus( QSettings& settings )
{
    loadGlobalSettings( settings );
}

void
MainWidget::saveStatus( QSettings& settings )
{
    saveGlobalSettings( settings );
}

QString
MainWidget::settingName()
{
    return "Settings";
}

/** resizes font if mouse wheel + control button are used */
void
MainWidget::wheelEvent( QWheelEvent* event )
{
    /** QTextBrowser already implements zoom -> zoom only browser contents */
    QWidget* sender    = this->childAt( mapFromGlobal( QCursor::pos() ) );
    bool     isBrowser = ( sender->parentWidget() &&
                           !strcmp( sender->parentWidget()->metaObject()->className(), "cubegui::HtmlTextBrowser" ) );
    bool enableZoom = ( event->modifiers() & Qt::ControlModifier ); // wheel + control required to zoom
    if ( isBrowser || !enableZoom )
    {
        return;
    }

    int step = event->angleDelta().y() > 0 ? 1 : -1;
    zoomSteps += step;
    zoom( true );
}

/** zooms the fonts of all widgets by zoomSteps and repaints them */
void
MainWidget::zoom( bool showTooltip )
{
    /* setting of style sheet is slow and causes segfaults, if it is often called
        QFont font = qApp->font();
        QString userStyle = StyleSheetEditor::getInstance()->styleSheet();
        qApp->setStyleSheet( userStyle+style );
       } */

    /** save initial font sizes before any change has been made (font changes can be propagated to children) */
    QFont font = qApp->font();
    if ( initialFontSizesMain[ qApp ] <= 0 )
    {
        initialFontSizesMain[ qApp ] = font.pixelSize() > 0 ? font.pixelSize() : font.pointSize();
    }

    zoomSteps = std::max( zoomSteps, minFontSize - initialFontSizesMain[ qApp ] );
    zoomSteps = std::min( zoomSteps, maxFontSize - initialFontSizesMain[ qApp ] );

    for ( QWidget* widget : QApplication::allWidgets() )
    {
        font = widget->font();
        int fontsize = initialFontSizes[ widget ] > 0 ? initialFontSizes[ widget ] : initialFontSizesMain[ widget ];
        if ( fontsize <= 0 )
        {
            fontsize                   = font.pixelSize() > 0 ? font.pixelSize() : font.pointSize();
            initialFontSizes[ widget ] = fontsize;
        }
    }
    // known issue: if a new window is created and zoom() is called while the window is open, the font size
    // will be wrong because the initial font size is invalid. The initial font has already been changed
    // with qapp->setFont() and would have to be resetted before the new window is opened.

    // set new zoomed font sizes for existing windows
    for ( QWidget* widget : QApplication::allWidgets() )
    {
        font = widget->font();
        int fontsize = initialFontSizes[ widget ] > 0 ? initialFontSizes[ widget ] : initialFontSizesMain[ widget ];
        font.setPointSize( fontsize + zoomSteps );
        widget->setFont( font );
        widget->update();
    }

    // set global qApp font, which will be used for new windows
    font = qApp->font();
    font.setPointSize( initialFontSizesMain[ qApp ] + zoomSteps );
    qApp->setFont( font );

    if ( showTooltip )                          // show percentage with tooltip in zoomed size
    {
        int     percent = 100 + zoomSteps * 10; // rough estimate to show nice value (default font is 10pt)
        QString txt     = "Zoom: " + QString::number( percent ) + "%";
        zoomTip->setText( txt );
    }
}

void
MainWidget::saveExperimentSettings( QSettings& experimentSettings )
{
    tabManager->saveExperimentSettings( experimentSettings );
}

void
MainWidget::loadExperimentSettings( QSettings& experimentSettings )
{
    tabManager->loadExperimentSettings( experimentSettings );
}


void
MainWidget::loadColorMap()
{
    QList<ColorMap*> maps = PluginManager::getInstance()->getColorMaps();
    foreach( ColorMap * map, maps )
    {
        if ( lastColorMapName == map->getMapName() )
        {
            setColorMap( map );
            break;
        }
    }
}

void
MainWidget::dragEnterEvent( QDragEnterEvent* event )
{
    const QMimeData* mime = event->mimeData();
    if ( mime->hasUrls() &&  mime->text().startsWith( "file://" ) && ( mime->text().endsWith( ".cube" ) || mime->text().endsWith( ".cubex" ) || mime->text().endsWith( ".cube.gz" ) ) )
    {
        event->acceptProposedAction();
    }
}

void
MainWidget::dropEvent( QDropEvent* event )
{
    QString file = event->mimeData()->text();
    file.remove( "file://" );
    loadFile( file );
    event->acceptProposedAction();
}


//=======================================================================================
ToolTip::ToolTip( QWidget* parent ) : QWidget( parent )
{
    setWindowFlags( Qt::ToolTip );
    setLayout( new QHBoxLayout() );
    //layout()->setContentsMargins( 0, 0, 0, 0);
    label = new QLabel();
    layout()->addWidget( label );

    // swap foreground and background color
    label->setAutoFillBackground( true );
    QPalette      pal;
    const QColor& text = pal.color( QPalette::Window );
    pal.setColor( QPalette::Window, pal.color( QPalette::WindowText ) );
    pal.setColor( QPalette::WindowText, text );
    label->setPalette( pal );
    this->setPalette( pal );
}

void
ToolTip::setText( const QString& txt )
{
    QPoint pos = QCursor::pos();
    label->setText( txt );
    QRect     rect = this->rect();
    const int pad  = 20;
    pos -= QPoint( rect.width() / 2, rect.height() + pad );
    rect.moveTo( pos );
    QFont font = label->font();
    font.setPointSize( fontsize );
    label->setFont( font );
    static int counter = 0;
    counter++;
    QTimer::singleShot( hideInMSec, this, [ this ](){
        counter--;
        if ( counter == 0 )
        {
            this->hide();
        }
    } );

    this->setGeometry( rect );
    this->show();
}
