/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include <cubelib-version.h>
#include <iostream>
#include <QRegularExpression>
#include "Globals.h"
#include "DefaultColorMap.h"
#include "ColorMap.h"
#include "PrecisionWidget.h"
#include "MainWidget.h"
#include "ColorMap.h"
#include "TabManager.h"
#include "DefaultValueView.h"
#include "TreeItem.h"
#include "Settings.h"
#include "CubeProxy.h"

using namespace cubegui;
using namespace cubepluginapi;
using namespace std;

Globals*   Globals::globals          = nullptr;
ValueView* Globals::defaultValueView = nullptr;
ColorMap*  Globals::defaultColorMap  = nullptr;

class DebugIODevice : public QIODevice
{
public:
    DebugIODevice()
    {
        open( QIODevice::WriteOnly | QIODevice::Text );
        enabled = true;
    }

    void
    setEnabled( bool enabled )
    {
        this->enabled = enabled;
    }
protected:
    qint64
    readData( char*, qint64 )
    {
        return 0;
    }
    qint64
    writeData( const char* data, qint64 maxSize )
    {
        if ( enabled )
        {
            QString msg = QString( data );
            msg = msg.remove( QRegularExpression( "\\s+$" ) );
            Globals::setStatusMessage( msg, Verbose );
        }
        return maxSize;
    }
private:
    bool enabled;
};

QTextStream&
Globals::debug( const QString& sender )
{
    getInstance();
    if ( globals->outStream == nullptr )
    {
        DebugIODevice* device = new DebugIODevice();
        device->setEnabled( optionIsSet( VerboseMode ) );
        globals->outStream = new QTextStream( device );
    }
    *( globals->outStream ) << sender << ": ";
    return *( globals->outStream );
}

Globals*
Globals::getInstance()
{
    if ( !globals )
    {
        globals = new Globals();
    }
    return globals;
}

Globals::Globals()
{
    mainWidget       = 0;
    tabManager       = 0;
    colorMap         = 0;
    precisionWidget  = 0;
    outStream        = 0;
    settings         = 0;
    defaultValueView = new DefaultValueView();
    defaultColorMap  = new DefaultColorMap();
    precisionWidget  = new PrecisionWidget();
    colorMap         = defaultColorMap;
}

void
Globals::setTabManager( TabManager* t )
{
    tabManager = t;
}
void
Globals::setMainWidget( MainWidget* m )
{
    mainWidget = m;
}

int
Globals::cubelibVersion()
{
    getInstance();
    int version = 0;
#if ( CUBELIB_VERSION_NUMBER > CUBELIB_VERSION_CHECK( 4, 6, 0 ) )
    try
    {
        cube::CubeProxy* cube = globals->tabManager->getCube();
        version = cube->getCubelibVersionNumber();
    }
    catch ( cube::FatalError& e )
    {
        globals->setStatusMessage( QString( e.what() ), Critical );
    }
#endif

    return version;
}

PrecisionWidget*
Globals::getPrecisionWidget()
{
    return getInstance()->precisionWidget;
}

QColor
Globals::getColor( double value, double minValue, double maxValue, bool whiteForZero )
{
    return getInstance()->colorMap->getColor( value, minValue, maxValue, whiteForZero );
}

void
Globals::setStatusMessage( const QString& msg, MessageType type, bool isLogged )
{
    getInstance();
    if ( globals->mainWidget )
    {
        globals->mainWidget->setMessage( msg, type, isLogged );
    }
    else
    {
        std::cerr << msg.toStdString() << std::endl;
    }
}

/** converts the number using the given precision format; in scientific format, the same exponent as
 * the reference value is used
 * @param value the number to convert
 * @param integerType the type of the number is integer if "integerType" is true, otherwise the type is double
 * @param format FORMAT_TREES or FORMAT_DEFAULT
 */
QPair<QString, QString>
Globals::formatNumberAndUnit( double value, const QString& unit, bool integerType, PrecisionFormat format, const cubegui::TreeItem* item )
{
    return getInstance()->precisionWidget->numberToQStringAndUnit( value, unit, integerType, format, item );
}


/** converts the number using the given precision format; in scientific format, the same exponent as
 * the reference value is used
 * @param value the number to convert
 * @param integerType the type of the number is integer if "integerType" is true, otherwise the type is double
 * @param format FORMAT_TREES or FORMAT_DEFAULT
 */
QString
Globals::formatNumber( double value, bool integerType, PrecisionFormat format, const cubegui::TreeItem* item )
{
    return getInstance()->precisionWidget->numberToQString( value, integerType, format, item );
}

/** converts the number using the given precision format; in scientific format, the same exponent as
 * the reference value is used
 * @param value the number to convert
 * @param referenceValue the number that is used to determine the format
 * @param integerType the type of the number is integer if "integerType" is true, otherwise the type is double
 * @param format FORMAT_TREES or FORMAT_DEFAULT
 * @param scientificSmallValue use scientificSmallValue use scientific notations for values below 0.1
 */
QString
Globals::formatNumber( double value, double referenceValue, bool integerType, bool scientificSmall, PrecisionFormat format )
{
    return getInstance()->precisionWidget->numberToQString( value, referenceValue, integerType, scientificSmall, format );
}

double
Globals::getRoundThreshold( PrecisionFormat format )
{
    return getInstance()->precisionWidget->getRoundThreshold( format );
}

double
Globals::getRoundNumber( PrecisionFormat format )
{
    return getInstance()->precisionWidget->getRoundNr( format );
}

double
Globals::getPrecision( PrecisionFormat format )
{
    return getInstance()->precisionWidget->getPrecision( format );
}

QMainWindow*
Globals::getMainWindow()
{
    return getInstance()->mainWidget;
}

TabManager*
Globals::getTabManager()
{
    return getInstance()->tabManager;
}

// -------------- private ---------------------

ColorMap*
Globals::getColorMap()
{
    return getInstance()->colorMap;
}

void
Globals::setColorMap( ColorMap* map )
{
    getInstance();
    if ( map )
    {
        globals->colorMap = map;
    }
    else
    {
        globals->colorMap = globals->defaultColorMap;
    }
    globals->tabManager->updateTreeItemProperties(); // update trees
    globals->mainWidget->update();                   // update colormap widget
}

void
Globals::setValueView( cube::DataType type, ValueView* view )
{
    getInstance();
    if ( !view )
    {
        view = globals->defaultValueView;
    }
    globals->valueViews[ type ] = view;

    if ( globals->tabManager->isVisible() )
    {
        globals->tabManager->reinit(); // can be optimized, complete recalculation isn't required but min/maximum value may have changed
        globals->tabManager->updateValueViews();
    }
}

// removes all map entries with view as value
void
Globals::removeValueView( ValueView* view )
{
    getInstance();
    for ( auto it = globals->valueViews.begin(); it != globals->valueViews.end(); )
    {
        cubepluginapi::ValueView* value = it->second;
        if ( view == value )
        {
            it = globals->valueViews.erase( it );
        }
        else
        {
            it++;
        }
    }
}

ValueView*
Globals::getValueView( cube::DataType type )
{
    getInstance();
    const auto& it = globals->valueViews.find( type );
    return ( it == globals->valueViews.end() ) ? globals->defaultValueView : it->second;
}

QString
Globals::getOption( UserOption option )
{
    getInstance();
    QString     value;
    const auto& it = globals->userOptions.find( option );
    if ( it != globals->userOptions.end() )
    {
        value = it->second;
    }
    return value;
}

Settings*
Globals::getSettings()
{
    return globals ? globals->settings : nullptr;
}

void
Globals::setSettings( Settings* newSettings )
{
    settings = newSettings;
}

bool
Globals::optionIsSet( UserOption option )
{
    getInstance();
    return !globals->getOption( option ).isEmpty();
}

void
Globals::setOption( UserOption option, QString value )
{
    getInstance();
    globals->userOptions[ option ] = value;
}
