/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2023                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/



#include "config.h"


#include "CubeRegion.h"


#include "FlatTreeView.h"
#include "FlatTree.h"
#include "Globals.h"
#include "TabManager.h"

using namespace cubegui;
using namespace std;

FlatTreeView::FlatTreeView( TreeModelInterface* modelInterface,
                            const QString&      tabLabel ) : TreeView( modelInterface, tabLabel )
{
    isActive               = false;
    selectionIsInvalidated = false;
    selectedTree           = nullptr;
    setIcon( QIcon( ":images/list-icon.png" ) );
}


/**
 * called, if user selects an item in the call tree view -> select corresponding items, if flat tree becomes active
 */
void
FlatTreeView::invalidateSelection( Tree* tree )
{
    selectionIsInvalidated = true;
    selectedTree           = tree;
}

/* select all entries in the flat tree, that have the same label and same region as the selected entries
   in the call tree
 */
void
FlatTreeView::copySelection()
{
    QStringList selectedItems;
    foreach( TreeItem * item, selectedTree->getSelectionList() )
    {
        if ( !selectedItems.contains( item->getName() ) )
        {
            selectedItems.append( item->getName() );
        }
    }
    QModelIndexList found = searchItems( selectedItems );
    QModelIndexList foundRegions;

    // only select items from the same region
    for ( QModelIndex idx : found )
    {
        TreeItem* item      = modelInterface->getTreeItem( idx );
        uint32_t  regionId1 = ( static_cast<cube::Region* > ( item->getCubeObject() ) )->get_id();
        for ( TreeItem* sItem : selectedTree->getSelectionList() ) // previous tree ( call tree )
        {
            if ( !sItem->getCubeObject() )
            {
                continue; // ignore artificial nodes
            }
            uint32_t regionId2 = sItem->getCnode()->get_callee()->get_id();
            if ( regionId1 == regionId2 )
            {
                foundRegions.append( idx );
                break;
            }
        }
    }

    this->blockSignals( true );  // selected items will be handled in TreeView::setActive
    selectItems( foundRegions );
    this->blockSignals( false );
}

void
FlatTreeView::setActive( bool active )
{
    isActive = active;

    if ( isActive && selectionIsInvalidated )
    {
        copySelection();
        selectionIsInvalidated = false;
    }

    TreeView::setActive( active );
}

/**
 * creates context menu items for call trees
 */
void
FlatTreeView::fillContextMenu()
{
    if ( getTree()->isManualCalculation() )
    {
        QAction* calculateAct = new QAction( tr( "Calculate flat tree" ), this );
        connect( calculateAct, SIGNAL( triggered() ), this, SLOT( calculateTree() ) );
        contextMenu->addAction( calculateAct );
        contextMenu->addSeparator();
    }

    contextMenu->addAction( contextMenuHash.at( TreeItemInfo ) );
    contextMenu->addSeparator();
    contextMenu->addAction( contextMenuHash.at( ExpandMenu ) );
    contextMenu->addSeparator();
    contextMenu->addAction( contextMenuHash.at( ExpandMenu ) );

    QMenu* hidingMenu = contextMenu->addMenu( tr( "Hiding" ) );
    hidingMenu->setWhatsThis( tr( "Hide subtrees." ) );
    hidingMenu->addAction( contextMenuHash.at( HidingThreshold ) );
    hidingMenu->addSeparator();
    hidingMenu->addAction( contextMenuHash.at( DynamicHiding ) );
    hidingMenu->addAction( contextMenuHash.at( StaticHiding ) );
    hidingMenu->addAction( contextMenuHash.at( HideItem ) );
    hidingMenu->addAction( contextMenuHash.at( UnhideItem ) );
    hidingMenu->addAction( contextMenuHash.at( NoHiding ) );

    contextMenu->addAction( contextMenuHash.at( FindItems ) );
    contextMenu->addAction( contextMenuHash.at( ClearFound ) );
    contextMenu->addAction( contextMenuHash.at( SortingMenu ) );
    contextMenu->addSeparator();
    contextMenu->addAction( contextMenuHash.at( UserDefinedMinMax ) );
    contextMenu->addSeparator();
    contextMenu->addAction( contextMenuHash.at( CopyClipboard ) );
    contextMenu->addSeparator();

    // HidingThreshold can only be set if hiding is enabled
    contextMenuHash.at( HidingThreshold )->setEnabled( !contextMenuHash.at( NoHiding )->isChecked() );
    if ( contextMenuItem && contextMenuItem->isTopLevelItem() )
    {
        contextMenuHash.at( HideItem )->setEnabled( false ); // disabled for root item
    }
}

QString
FlatTreeView::getContextDescription( TreeItem* item ) const
{
    QList<QPair<QString, QString> > list; // QPair<description, value>
    if ( item->getCubeObject() && item->getType() == REGIONITEM )
    {
        cube::Region* region = static_cast<cube::Region* >( item->getCubeObject() );

        list.append( QPair<QString, QString>( tr( "Region name" ), region->get_name().c_str() ) );
        list.append( QPair<QString, QString>( tr( "Region description" ), region->get_descr().c_str() ) );

        FlatTree*       flat         = static_cast<FlatTree*> ( getTree() );
        TreeView*       callTreeView = Globals::getTabManager()->getView( flat->callTree );
        QModelIndexList found        = callTreeView->searchItems( QStringList() << item->getName() );
        list.append( QPair<QString, QString>( tr( "References in call tree" ), QString::number( found.size() ) ) );

        QString beginLine = ( region->get_begn_ln() != -1 ) ? QString::number( region->get_begn_ln() ) : "undefined";
        QString endLine   = ( region->get_end_ln() != -1 ) ? QString::number( region->get_end_ln() ) : "undefined";

        list.append( QPair<QString, QString>( tr( "Beginning line" ), beginLine ) );
        list.append( QPair<QString, QString>( tr( "Ending line" ), endLine ) );
        list.append( QPair<QString, QString>( tr( "Paradigm" ), region->get_paradigm().c_str() ) );
        list.append( QPair<QString, QString>( tr( "Role" ), region->get_role().c_str() ) );
        list.append( QPair<QString, QString>( tr( "Source file" ), region->get_mod().c_str() ) );
        list.append( QPair<QString, QString>( tr( "Url" ), region->get_url().c_str() ) );

        QString info = InfoWidget::formatAsTable( list );
        info += formatAttributes( region->get_attrs() );

        return info;
    }
    return "";
}

// ManualCalculation mode
void
FlatTreeView::calculateTree()
{
    // only mark first visible item because flat tree is calculated in one chunk
    QList<TreeItem*> items;
    items.append( getTree()->getRootItem()->getChildren().first() );
    emit calculateRequest( items );
}
