/****************************************************************************
**  CUBE        http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2025                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include "config.h"
#include "DefaultCallTree.h"
#include "CubeRegion.h"
#include "MetricTree.h"
#include "SystemTree.h"
#include "TreeItem.h"
#include "AggregatedTreeItem.h"
#include "TreeModel.h"
#include "CubeProxy.h"
#include "TabManager.h"

#include <QLayout>
#include <QPropertyAnimation>

using namespace std;
using namespace cube;
using namespace cubegui;

DefaultCallTree::DefaultCallTree( CubeProxy* cubeProxy, const std::vector< cube::Cnode* >& cnodes ) : CallTree( cubeProxy ), topCnodes( cnodes )
{
    treeType             = DEFAULTCALLTREE;
    iterationsAreHidden_ = false;
    loopRoot             = 0;
    aggregatedLoopRoot   = 0;
    initialNodes         = CubeProxy::ALL_CNODES;
    label                = ( cube->getAttribute( "calltree title" ) != "" ) ? cube->getAttribute( "calltree title" ).c_str() : tr( "Call tree" );
}

/**
 * @brief DefaultCallTree::initializeCudaKernels
 * - adds relationship cuda kernel -> callsite
 * - reorders kernel items
 * - make the label more human readable
 */
void
DefaultCallTree::initializeCudaKernels()
{
    QHash<double, TreeItem*> callHashMap;  // callside hash id -> item (temporary)
    QList<TreeItem*>         kernelItems;  // list of kernel items
    QList<TreeItem*>         callers;      // list of callers of the kernel items

    // fills map callSiteItems which assign kernel to its call site and vice versa. These items are identified by
    // the additional parameter "callsite id=..." in their parameter
    QList<TreeItem*> allItems = treeItems;
    Tree*            taskTree = Globals::getTabManager()->getTree( TASKTREE );
    if ( taskTree )
    {
        allItems.append( taskTree->getItems() );
    }

    /* Find artificial kernel root item(s) ( should contain KERNELS in its name ) */
    QSet<TreeItem*> kernelRoots;
    for ( TreeItem* item : allItems )
    {
        cube::Cnode* cnode = item->getCnode();
        if ( cnode && item->getName().contains( "KERNELS" ) )
        {
            cube::Region* region = cnode->get_callee();
            if ( region->get_role() == "artificial" )
            {
                kernelRoots.insert( item );
            }
        }
    }

    for ( TreeItem* item : allItems )
    {
        cube::Cnode* cnode = item->getCnode();
        if ( !cnode )
        {
            continue;
        }
        vector<pair<string, double> > str_params = cnode->get_num_parameters();
        for ( auto iter = str_params.begin(); iter != str_params.end(); ++iter )
        {
            if ( iter->first == "callsite id" )                  // parameter found -> link to caller
            {
                if ( kernelRoots.contains( item->getParent() ) ) // todo: better check not yet available
                {
                    item->setCallType( CallItemType::KernelItem );
                    kernelItems.append( item );
                }
                else
                {
                    callers.append( item );
                }
                double id = iter->second;

                TreeItem* relatedItem = callHashMap.value( id );
                if ( relatedItem )
                {
                    callSiteItems.insert( item, relatedItem );
                    callSiteItems.insert( relatedItem, item );
                    item->displayName = item->name;
                }
                else
                {
                    callHashMap[ id ] = item;
                }
            }
        }
    }

    // beautify label: replace call site=xxx with call path id
    static QRegularExpression re( "callsite id=[\\d\\.e+]+" );
    for ( TreeItem* child : kernelItems )
    {
        if ( callSiteItems.value( child ) )
        {
            QString replacement = "caller id=" + QString::number( callSiteItems.value( child )->getCubeObject()->get_id() );
            child->name.replace( re, replacement );
            child->displayName = child->name; // todo
        }
    }
    for ( TreeItem* child : callers )
    {
        if ( callSiteItems.value( child ) )
        {
            QString replacement = "callee id=" + QString::number( callSiteItems.value( child )->getCubeObject()->get_id() );
            child->name.replace( re, replacement );
            child->displayName = child->name; // todo
        }
    }

    reorderCudaKernels( kernelItems );
}

/**
 * @brief DefaultCallTree::reorderSubTree reorders the tree by grouping items with the same value of orderCriteria
 * @param root item, whose children will be reordered
 * @param orderCriteria function that returns the value to be used for ordering
 * @param groupName function that returns the name of the group
 */
void
DefaultCallTree::reorderSubTree( TreeItem* root, std::function<QString( TreeItem* )> orderCriteria, std::function<QString( TreeItem* )> groupName )
{
    QList<TreeItem*> flat = root->getChildren();
    root->children.clear();

    // all items with the same value of level1Name are grouped together using the hash groupedItems
    QHash<QString, QList< TreeItem* > > groupedItems;
    for ( TreeItem* item : flat )
    {
        QString name = orderCriteria( item );
        if ( groupedItems.contains( name ) )
        {
            groupedItems[ name ].append( item );
        }
        else
        {
            groupedItems.insert( name, QList< TreeItem* >() << item );
        }
    }

    // function to get cnode id from tree item
    auto getCnodeId = [ ]( const TreeItem* item ) {
                          cube::Cnode* cnode = item->getCnode();
                          return cnode ? cnode->get_id() : 0;
                      };
    // Convert QHash to a vector of pairs
    QVector<QPair<QString, QList<TreeItem*> > > pairVector;
    for ( auto it = groupedItems.begin(); it != groupedItems.end(); ++it )
    {
        pairVector.append( qMakePair( it.key(), it.value() ) );
    }
    // Sort the vector of pairs based on the cnode id
    std::sort( pairVector.begin(), pairVector.end(), [ getCnodeId ]( const QPair<QString, QList<TreeItem*> >& a, const QPair<QString, QList<TreeItem*> >& b ) {
        return getCnodeId( a.second.first() ) < getCnodeId( b.second.first() );
    } );

    // add the restructured items to the root item
    for ( auto it = pairVector.begin(); it != pairVector.end(); ++it )
    {
        QString          group = it->first;
        QList<TreeItem*> items = it->second;

        if ( items.size() > 1 ) // generate sub tree for items that are called from multiple callsites
        {
            TreeItem* newItem = new TreeItem( this, groupName( items.first() ), items.first()->getType(), nullptr );
            newItem->rootItem = items.first()->rootItem;
            newItem->setCallType( CallItemType::ArtificialKernelRoot );
            root->addChild( newItem );
            for ( TreeItem* item : items )
            {
                newItem->addChild( item );
                callSiteItems.insert( newItem, callSiteItems.value( item ) ); // set the callsite information from all children to the new artificial root
            }
        }
        else
        {
            root->addChild( items.first() );
        }
    }
}

void
DefaultCallTree::reorderCudaKernels( QList<TreeItem*> flatList )
{
    if ( flatList.size() == 0 )
    {
        return;
    }
    // create a lambda for level1 of the tree, that returns the function name without the parameter list
    auto functionName = [ ]( TreeItem* item )
                        {
                            QString name = item->displayName;
                            int     idx  = name.lastIndexOf( "(" );
                            if ( idx >= 0 )
                            {
                                return name.left( idx - 1 );
                            }
                            return name;
                        };

    // create a lambda for level2 of the tree, that returns the id of the call site cube object
    auto callSiteId = [ this ]( TreeItem* item )
                      {
                          TreeItem* caller = callSiteItems.value( item );
                          return caller ? QString::number( callSiteItems.value( item )->getCubeObject()->get_id() ) : "";
                      };

    auto functionNameWithCallsite = [ &functionName, &callSiteId ]( TreeItem* item )
                                    {
                                        return functionName( item ) + " (caller id=" + callSiteId( item ) + ")";
                                    };

    // remove all kernel items from the original tree and add them to a temporary tree (kernelParent)
    TreeItem* kernelParent = flatList.first()->getParent();
    foreach( TreeItem * child, flatList )
    {
        this->removeItem( child );
    }
    TreeItem* root = new TreeItem( this, "", CALLITEM, nullptr ); // use temporary item for kernel subtree for reordering
    root->children = flatList;

    // reorder temporary tree
    reorderSubTree( root, functionName, functionName ); // 1. level: order by function name
    for ( TreeItem* subroot : root->getChildren() )
    {
        if ( subroot->getChildren().size() > 1 )
        {
            reorderSubTree( subroot, callSiteId, functionNameWithCallsite ); // 2. level: order by call site id
        }
    }

    // add the reordered kernel items back to the original tree
    for ( TreeItem* item : root->getChildren() )
    {
        this->addItem( item, kernelParent ); // add to original tree
    }
}



DefaultCallTree::~DefaultCallTree()
{
    unsetLoop(); // delete merged iterations
}

list_of_cnodes
DefaultCallTree::getNodes() const
{
    // empty list => CubeProxy::ALL_CNODES
    return this->activeNodes;
}

list_of_cnodes
DefaultCallTree::getSelectedNodes() const
{
    return getNodes( selectionList );
}

/**
 * @brief DefaultCallTree::getNodes
 * @return the corresponding cnodes to the given list of tree items.
 */
list_of_cnodes
DefaultCallTree::getNodes( const QList<TreeItem*> items ) const
{
    list_of_cnodes list;

    for ( TreeItem* item : items )
    {
        cube::Cnode* cnode = 0;
        if ( item->isAggregatedRootItem() && item->isExpanded() ) // special case aggregated loops: root item of the loop
        {
            cnode = item->getCnode();

            // If iterations are merged, the tree level with the iteration items is removed. The iterations items
            // may have exclusive values, which have to be added to the exclusive value of its parent.
            for ( TreeItem* child : loopRoot->getChildren() )
            {
                cnode_pair pair;
                pair.first  = child->getCnode();
                pair.second = cube::CUBE_CALCULATE_EXCLUSIVE;
                list.push_back( pair );
            }
        }
        else if ( item->isAggregatedLoopItem() ) // add all loop entries
        {
            const QList<cube::Cnode*>& iterations = static_cast<AggregatedTreeItem*>( item )->getIterations();
            for ( cube::Cnode* cnode : iterations )
            {
                if ( cnode )
                {
                    cnode_pair pair;
                    pair.first  = cnode;
                    pair.second =  ( item->isExpanded() ) ? cube::CUBE_CALCULATE_EXCLUSIVE : cube::CUBE_CALCULATE_INCLUSIVE;
                    list.push_back( pair );
                }
            }
        }
        else if ( item->getCallType() == CallItemType::ArtificialKernelRoot )
        {
            if ( !item->isExpanded() )
            {
                QList<TreeItem*> subitems; // all subitems of this artificial item and the children of their artificial subitems
                subitems = item->getChildren();
                while ( !subitems.isEmpty() )
                {
                    TreeItem* child = subitems.takeFirst();
                    if ( child->getCallType() != CallItemType::ArtificialKernelRoot )
                    {
                        cnode_pair pair;
                        pair.first  = child->getCnode();
                        pair.second = cube::CUBE_CALCULATE_EXCLUSIVE;
                        list.push_back( pair );
                    }
                    else
                    {
                        subitems.append( child->getChildren() );
                    }
                }
            }
            else
            {
                item->setExclusiveValue( nullptr ); // artificial items don't have an exclusive value
            }
        }
        else // normal item (which may have pruned children)
        {
            cnode = item->getCnode();

            // add inclusive value of all pruned children to exclusive value of parent
            const QList<TreeItem*>& pruned = prunedChildren.value( item );
            if ( item->isExpanded() && pruned.size() > 0 )
            {
                // add inclusive values of all pruned children to parent item
                for ( TreeItem* item : pruned )
                {
                    cnode_pair pair;
                    pair.first  = item->getCnode();
                    pair.second = CUBE_CALCULATE_INCLUSIVE;
                    list.push_back( pair );
                }
            }
        }

        // if expanded item has hidden children: add inclusive value of all hidden children to exclusive value of the parent
        if ( item->isExpanded() )
        {
            auto it = hiddenItemMap.find( item );
            if ( it != hiddenItemMap.end() ) // item has hidden children
            {
                const QList<TreeItem*>& hidden = it->second;

                // add inclusive values of all hidden children to parent item
                for ( TreeItem* child : hidden )
                {
                    Cnode* cnode = child->getCnode();
                    if ( cnode )
                    {
                        cnode_pair pair;
                        pair.first  = child->getCnode();
                        pair.second = CUBE_CALCULATE_INCLUSIVE;
                        list.push_back( pair );
                    }
                    else // special case: hidden children are artificial kernel nodes
                    {
                        for ( cnode_pair pair : getNodes( QList<TreeItem*>() << child ) )
                        {
                            list.push_back( pair );
                        }
                    }
                }
            }
        }

        if ( cnode )
        {
            cnode_pair pair;
            pair.first  = cnode;
            pair.second = item->isExpanded() ? CUBE_CALCULATE_EXCLUSIVE :  CUBE_CALCULATE_INCLUSIVE;
            list.push_back( pair );
        }
    }
    return list;
}

QString
DefaultCallTree::getLabel() const
{
    return label;
}

void
DefaultCallTree::initTree()
{
    userRoots.clear();
    activeNodes = initialNodes;
    metricValueHash.clear();
    prunedChildren.clear();
    TreeItem* loop = getDefinedLoopRoot();
    if ( loop )
    {
        setAsLoop( loop, false );
    }
}

TreeItem*
DefaultCallTree::createTree()
{
    top = new TreeItem( this, QString( "Invisible call tree root" ), CALLITEM, 0 );
    createItems<cube::Cnode>( top, topCnodes, CALLITEM );
    initTree();

    return top;
}

/** checks if an item is defined as a loop ("CLUSTER ROOT CNODE ID" is set) */
TreeItem*
DefaultCallTree::getDefinedLoopRoot()
{
    TreeItem* loop = nullptr;
    QString   nrs  = QString( cube->getAttribute( "CLUSTER ROOT CNODE ID" ).c_str() );
    if ( nrs.length() > 0 )
    {
        uint             cnodeID = nrs.toUInt();
        QList<TreeItem*> list;
        list.append( top->getChildren() );
        while ( !list.isEmpty() )
        {
            TreeItem* it = list.takeFirst();
            if ( it->getCubeObject() )
            {
                uint itemCnode = it->getCubeObject()->get_id();
                if ( itemCnode == cnodeID )
                {
                    loop = it;
                    break;
                }
            }
            list.append( it->getChildren() );
        }
    }
    return loop;
}

QString
DefaultCallTree::getItemName( cube::Vertex* vertex ) const
{
    std::string name = ( static_cast<cube::Cnode* > ( vertex ) )->get_callee()->get_name();

    std::vector<std::pair<std::string, double> > num_params = ( static_cast<cube::Cnode* > ( vertex ) )->get_num_parameters();
    std::string                                  sep        = " (";
    std::string                                  end        = "";
    for ( unsigned i = 0; i < num_params.size(); i++ )
    {
        char buffer[ 100 ];
        sprintf( buffer, "%g", num_params[ i ].second );
        name += sep + num_params[ i ].first + "=" + std::string( buffer );
        sep   = ", ";
        end   = ")";
    }
    std::vector<std::pair<std::string, std::string> > str_params = ( static_cast<cube::Cnode* > ( vertex ) )->get_str_parameters();
    for ( unsigned i = 0; i < str_params.size(); i++ )
    {
        name += sep + str_params[ i ].first + "=" + str_params[ i ].second;
        sep   = ", ";
        end   = ")";
    }
    name += end;

    return QString( name.c_str() );
}


// -----------------------------------------------------------------
// ---------------- iteration handling --------------------------
// -----------------------------------------------------------------



/** The given item is set as a loop. This allows to hide its iterations.
 * @param item the top loop item
 * @param reinit it true, the values of the loop root and visible children and all tabs right to this tree are recalculated
 */
void
DefaultCallTree::setAsLoop( TreeItem* item, bool reinit )
{
    assert( item );
    if ( item == aggregatedLoopRoot || item == loopRoot || item->children.size() == 0 )
    {
        return;
    }

    unsetLoop(); // unset previously set loop
    loopRoot           = item;
    aggregatedLoopRoot = mergeIterations( loopRoot );

    int iterations = aggregatedLoopRoot->iterations.size();
    aggregatedLoopRoot->name.append( " [" + QString::number( iterations ) + tr( " iterations" ) + "]" );

    loopRoot->displayName.clear();
    loopRoot->invalidateLabel(); // update tree item label of new loop root
    if ( reinit )
    {
        Globals::getTabManager()->reinit( loopRoot );
    }
}

void
DefaultCallTree::unsetLoop()
{
    if ( loopRoot != 0 )
    {
        if ( iterationsAreHidden_ )
        {
            showIterations();
        }

        deleteMergedIterations( aggregatedLoopRoot );           // recursively delete all merged items
        treeItemHash[ loopRoot->getCubeObject() ] = loopRoot;   // same key as aggregatedLoopRoot

        TreeItem* oldLoop = loopRoot;
        aggregatedLoopRoot = 0;
        loopRoot           = 0;

        oldLoop->displayName.clear();
        oldLoop->invalidateLabel(); // update tree item label after loopRoot has been changed
    }
}


/**
   Hides the iteration of the currenly defined loop.
 */
void
DefaultCallTree::hideIterations()
{
    if ( !loopRoot  || iterationsAreHidden_ )
    {
        return;
    }
    iterationsAreHidden_ = true;

    // replace loop structure with aggregated structure
    treeModel->replaceSubtree( loopRoot, aggregatedLoopRoot );

    aggregatedLoopRoot->select();
    aggregatedLoopRoot->setExpanded( true );

    aggregatedLoopRoot->displayName.clear();
    aggregatedLoopRoot->invalidateLabel();
}

void
DefaultCallTree::showIterations()
{
    if ( !loopRoot || !iterationsAreHidden_ )
    {
        return;
    }
    iterationsAreHidden_ = false;

    // replace aggregated loop structure with iterations
    treeModel->replaceSubtree( aggregatedLoopRoot, loopRoot );

    loopRoot->select();
    loopRoot->setExpanded( true );

    loopRoot->displayName.clear();
    loopRoot->invalidateLabel();
}

bool
DefaultCallTree::iterationsAreHidden() const
{
    return iterationsAreHidden_;
}

TreeItem*
DefaultCallTree::getLoopRootItem() const
{
    return loopRoot;
}

AggregatedTreeItem*
DefaultCallTree::getAggregatedRootItem() const
{
    return aggregatedLoopRoot;
}

/**
   Treats the current item as root of a loop and merges its children. References from the merged iterations
   to the single iterations and vice versa are inserted.
 */
AggregatedTreeItem*
DefaultCallTree::mergeIterations( TreeItem* loopRoot )
{
    AggregatedTreeItem* aggregatedRoot = new AggregatedTreeItem( this, loopRoot->getDepth(), loopRoot );
    aggregatedRoot->cubeObject = loopRoot->cubeObject;
    aggregatedRoot->parentItem = loopRoot->parentItem;
    aggregatedRoot->setCallType( CallItemType::AggregatedLoopRoot );

    foreach( TreeItem * iteration, loopRoot->getChildren() )
    {
        aggregatedRoot->iterations.append( iteration->getCnode() );
    }

    mergeIterations( aggregatedRoot, loopRoot->getChildren() );

    return aggregatedRoot;
}


/**
 * @brief mergeIterations recursively merges iteration items with the same name into
 * new items and appends them to newParent.
 * Only items of the same tree depth which have the same name are merged.
 */
void
DefaultCallTree::mergeIterations( TreeItem* newParent, const QList<TreeItem*>& iterations )
{
    QSet<QString>                       allChildren; // all distinct children names of the current tree level
    QHash<QString, AggregatedTreeItem*> hash;        // child name -> aggregated tree item

    // find all children of iterations, eleminate those with same name and append a copy to newparent
    foreach( TreeItem * iteration, iterations )
    {
        if ( iteration )
        {
            foreach( TreeItem * child, iteration->getChildren() )
            {
                allChildren.insert( child->name );
                if ( !hash.contains( child->name ) ) // create pseudo-item which contains iterations
                {
                    AggregatedTreeItem* newChild = new AggregatedTreeItem( this, newParent->getDepth() + 1, child );
                    newChild->setCallType( CallItemType::LoopItem );
                    newParent->addChild( newChild );
                    hash.insert( child->name, newChild );
                }
                //AggregatedTreeItem* aggregated = hash.value( child->name );
                //aggregated->iterations.append( static_cast<cube::Cnode*> ( child->cubeObject ) );
            }
        }
    }

    // required if functions are not called in all iterations -> append null cnode for non existing iterations
    // (optimize: inner loop not necessary if function is called in all iterations)
    foreach( TreeItem * iteration, iterations )
    {
        foreach( QString childName, allChildren.values() )
        {
            AggregatedTreeItem* aggregated = hash.value( childName );
            cube::Cnode*        cnode      = 0;
            if ( iteration )
            {
                foreach( TreeItem * currentIterChild, iteration->getChildren() ) // seach child in current iteration
                {
                    if ( childName == currentIterChild->name )
                    {
                        cnode = currentIterChild->getCnode();
                        break;
                    }
                }
                if ( cnode )
                {
                    aggregated->iterations.append( cnode );
                }
            }
        }
    }

    // if newly added children of newParent have children, call this method recursively
    foreach( TreeItem * item, newParent->getChildren() )
    {
        QString childName = item->name;

        int              childCount = 0;
        QList<TreeItem*> listOfChildren;
        foreach( TreeItem * iteration, iterations )
        {
            TreeItem* nextChild = 0;
            if ( iteration )
            {
                foreach( TreeItem * child, iteration->getChildren() )
                {
                    if ( child->name == childName )
                    {
                        nextChild = child;
                        childCount++;
                        break;
                    }
                }
            }
            listOfChildren.push_back( nextChild );
        }

        if ( childCount > 0 )
        {
            mergeIterations( item, listOfChildren );
        }
    }
}

/**
   delete the loop root of merged iterations and all its children recursively
 */
void
DefaultCallTree::deleteMergedIterations( TreeItem* aggregated )
{
    foreach( TreeItem * item, aggregated->getChildren() )              // for all aggregated items
    {
        deleteMergedIterations( item );
    }
    treeItemHash.erase( aggregated->getCubeObject() );
    delete aggregated;
}

// -----------------------------------------------------------------
// ---------------- end iterations ------ --------------------------
// -----------------------------------------------------------------



// -----------------------------------------------------------------
// ---------------- calculation functions --------------------------
// -----------------------------------------------------------------

/**
 * @brief getSelectedFromHiddenIterations returns a list of items of the hidden iterations which correspond
 * to the selected merged item
 * @param iterations a list of hidden iterations, starting with tree level after root
 * @param selected path to the selected item, contains the names of each ancestor of the selected item,
 * starting with the level after root
 * @return list with the selected item for each iteration where an item with the same name at the same tree
 * level is found
 */
QList<TreeItem*>
getSelectedFromHiddenIterations( const QList<TreeItem*>& iterations, QStringList& selected )
{
    QList<TreeItem*> nextLevel;
    QList<TreeItem*> ret;

    QString selectedName = selected.takeFirst();
    foreach( TreeItem * iter, iterations )
    {
        foreach( TreeItem * child, iter->getChildren() )
        {
            if ( child->getName() == selectedName )             // found selected item in different iteration
            {
                nextLevel.push_back( child );
            }
        }
    }
    if ( !selected.empty() )
    {
        ret = getSelectedFromHiddenIterations( nextLevel, selected );
    }
    else
    {
        ret = nextLevel;
    }
    return ret;
}

QList<Task*>
DefaultCallTree::setBasicValues( const QList<Tree*>& leftTrees,
                                 const QList<Tree*>& rightTrees, const QList<TreeItem*>& itemsToCalculate )
{
    QList<Task*> workerData;
    /*
     * No update of tree is necessary if the call tree is to the left of the metric tree
     */
    if ( ( leftTrees.size() == 0 ) || ( ( leftTrees.size() == 1 ) && ( leftTrees[ 0 ]->getType() != METRIC ) ) )
    {
        return workerData;
    }

    // all visible items are already up to date and selectionSummary is a single item
    if ( ( itemsToCalculate.size() == 0 ) && ( selectionList.size() == 1 ) )
    {
        return workerData;
    }
    if ( ( itemsToCalculate.size() == 0 ) && this->isManualCalculation() )
    {
        return workerData;
    }

    QSet<DisplayType>         isLeft;
    QHash<DisplayType, Tree*> trees;
    for ( Tree* tree : leftTrees )
    {
        isLeft.insert( tree->getType() );
        trees.insert( tree->getType(), tree );
    }
    for ( Tree* tree : rightTrees )
    {
        trees.insert( tree->getType(), tree );
    }

    // Metric tree is definitely in the left trees, as the initial if-statement at the
    // beginning of the call already filtered out the other cases.
    list_of_metrics metric_selection;
    MetricTree*     metricTree = static_cast<MetricTree*>( trees[ METRIC ] );
    metric_selection = metricTree->getSelectedMetrics();

    // Set up selection for system tree, leave empty if system tree is right to call tree
    list_of_sysresources sysres_selection;
    SystemTree*          systemTree = static_cast<SystemTree*>( trees[ SYSTEM ] );
    sysres_selection = isLeft.contains( SYSTEM ) ? systemTree->getSelectedNodes() : CubeProxy::ALL_SYSTEMNODES;

    // calculate values for all invalidated tree items
    foreach( TreeItem * item, itemsToCalculate )
    {
        const list_of_cnodes& cnodes = getNodes( QList<TreeItem*>() << item );

        if ( cnodes.empty() && item->getCallType() == CallItemType::ArtificialKernelRoot )
        {
            continue; // skip artificial kernal items ( empty cnode list == ALL_CNODES == call tree root )
        }
        TreeTask* data = new TreeTask( item,
                                       metric_selection,
                                       cnodes,
                                       sysres_selection );
        workerData.append( data );
    }

    if ( selectionList.size() > 1 && !this->isManualCalculation() )
    {
        // calculate sum of the selected items for the statics in the value view
        TreeTask* sum = new TreeTask( selectionSummary,
                                      metric_selection,
                                      getSelectedNodes(),
                                      sysres_selection );
        workerData.append( sum );

        // calculate sum of the roots of the selected values for the selected-root value modes
        QSet<TreeItem*> roots;
        for ( TreeItem* sel : selectionList ) // collect all root level parents
        {
            roots.insert( sel->rootItem );
        }
        list_of_cnodes selectionRootNodes = getNodes( roots.values() );
        for ( cnode_pair& cn : selectionRootNodes )
        {
            cn.second = CUBE_CALCULATE_INCLUSIVE;
        }
        sum = new TreeTask( selectionRootSummary,
                            metric_selection,
                            selectionRootNodes,
                            sysres_selection );
        workerData.append( sum );
    }

    return workerData;
}

// -----------------------------------------------------------------
// ---------------- end calculation functions ----------------------
// -----------------------------------------------------------------

TreeItem*
DefaultCallTree::getTreeItem( uint32_t cnodeId ) const
{
    for ( TreeItem* item : treeItems )
    {
        cube::Cnode* cnode = item->getCnode();
        if ( cnode && cnode->get_id() == cnodeId )
        {
            return item;
        }
    }
    return nullptr;
}

/**
   updates the list of cnodes, that will be used for calculation of the trees left to the call tree
 */
void
DefaultCallTree::updateActiveNodes()
{
    // reduce active nodes to items that are available in model
    if ( !userRoots.isEmpty() ) // user has defined a new root => use inclusive value of the root
    {
        activeNodes = cube::list_of_cnodes();
        for ( TreeItem* userRoot : userRoots )
        {
            cnode_pair cp;
            cp.first  =  userRoot->getCnode();
            cp.second = CUBE_CALCULATE_INCLUSIVE;
            activeNodes.push_back( cp );
        }
    }
    else
    {
        activeNodes = initialNodes;
    }
}

/** sets current item as call tree root item
 */
void
DefaultCallTree::setAsRoot( QList<TreeItem*> newRoots )
{
    userRoots = newRoots;

    // update tree item list after tree items have been removed
    updateActiveNodes();

    // invalidate all trees left of the call tree
    TabManager*  tab = Globals::getTabManager();
    QList<Tree*> left, right;
    tab->getNeighborTrees( left, right, this );
    foreach( Tree * tree, left )
    {
        tree->invalidateItems();
    }

    // update selection list: search for selected items in the new tree and select them again
    for ( TreeItem* item : treeItems )
    {
        if ( item->isSelected() )
        {
            selectItem( item, true );
        }
    }
    if ( selectionList.isEmpty() ) // at least one item has to be selected
    {
        selectItem( newRoots.first() );
    }

    tab->reinit(); // recalculate trees
}

/** removes the given item from the tree
 */
void
DefaultCallTree::pruneItem( TreeItem* item )
{
    QList<TreeItem*>& prunedItems = prunedChildren[ item->getParent() ];
    prunedItems.append( item );

    TabManager* tab = Globals::getTabManager();
    item->getParent()->invalidate();  // parent's exclusive value has to be recalculated
    tab->reinit( item->getParent() ); // recalculate parent
}
